/**
 * easy BPEL software - Copyright (c) 2009 PetalsLink, 
 * http://www.petalslink.com/ 
 *  
 * This library is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2.1 of the License, or (at your option) 
 * any later version. This library is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
 * General Public License for more details. 
 *  
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 *  
 * ------------------------------------------------------------------------- 
 * $Id$ 
 * ------------------------------------------------------------------------- 
 */
package com.ebmwebsourcing.easybpmn.bpmn2bpel;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.namespace.QName;

import org.apache.commons.lang.NotImplementedException;
import org.ow2.easywsdl.schema.api.abstractElmt.AbstractSchemaElementImpl;
import org.petalslink.abslayer.Factory;
import org.petalslink.abslayer.service.api.Description;
import org.w3c.dom.Document;

import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.with.WithName;
import com.ebmwebsourcing.easybpel.model.bpel.api.BPELElement;
import com.ebmwebsourcing.easybpel.model.bpel.api.BPELElementImpl;
import com.ebmwebsourcing.easybpel.model.bpel.api.BPELException;
import com.ebmwebsourcing.easybpel.model.bpel.api.BPELProcess;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Activity;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Assign;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Empty;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Exit;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Flow;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.If;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Invoke;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Pick;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Receive;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Reply;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Scope;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Sequence;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.Wait;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.While;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.element.elements4assign.Copy;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.element.elements4assign.From;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.element.elements4assign.To;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.element.elements4pick.OnAlarm;
import com.ebmwebsourcing.easybpel.model.bpel.api.activity.element.elements4pick.OnMessage;
import com.ebmwebsourcing.easybpel.model.bpel.api.partnerLink.PartnerLink;
import com.ebmwebsourcing.easybpel.model.bpel.api.variable.BPELVariable;
import com.ebmwebsourcing.easybpel.model.bpel.api.variable.BPELElementVariable.VariableType;
import com.ebmwebsourcing.easybpel.model.bpel.api.wsdlImports.Import;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TAssign;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TBooleanExpr;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TCopy;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TDeadlineExpr;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TDurationExpr;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TEmpty;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TExit;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TFlow;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TFrom;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TIf;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TImport;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TInvoke;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TOnAlarmPick;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TOnMessage;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TPick;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TProcess;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TReceive;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TReply;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TScope;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TSequence;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TTo;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TWait;
import com.ebmwebsourcing.easybpel.model.bpel.executable.TWhile;
import com.ebmwebsourcing.easybpel.model.bpel.impl.BPELFactoryImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.BPELProcessImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.AssignImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.EmptyImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.ExitImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.FlowImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.IfImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.InvokeImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.PickImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.ReceiveImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.ReplyImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.ScopeImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.SequenceImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.WaitImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.WhileImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.element.elements4assign.CopyImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.element.elements4assign.FromImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.element.elements4assign.ToImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.element.elements4pick.OnAlarmImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.activity.element.elements4pick.OnMessageImpl;
import com.ebmwebsourcing.easybpel.model.bpel.impl.wsdlImports.ImportImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.api.BPELBooleanExpression;
import com.ebmwebsourcing.easybpel.xpath.exp.api.BPELDeadLineExpression;
import com.ebmwebsourcing.easybpel.xpath.exp.api.BPELDurationExpression;
import com.ebmwebsourcing.easybpel.xpath.exp.api.XPathExpressionException;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.BPELBooleanExpressionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.BPELDeadLineExpressionImpl;
import com.ebmwebsourcing.easybpel.xpath.exp.impl.BPELDurationExpressionImpl;
import com.ebmwebsourcing.easybpmn.bpmn20.api.BPMNException;
import com.ebmwebsourcing.easybpmn.bpmn20.api.DefinitionsHelper;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Assignment;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.BaseElement;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.CallActivity;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.CatchEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Collaboration;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.DataAssociation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.DataInputAssociation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.DataOutputAssociation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Definitions;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.EndEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Event;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.EventBasedGateway;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.EventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ExclusiveGateway;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.FlowElement;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.FlowNode;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Gateway;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Interface;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.IntermediateCatchEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.IntermediateThrowEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ItemDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Message;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.MessageEventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Operation;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ParallelGateway;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Participant;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.PartnerRole;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Process;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ReceiveTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.SendTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.SequenceFlow;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ServiceTask;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.StartEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.Task;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.TerminateEventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.ThrowEvent;
import com.ebmwebsourcing.easybpmn.bpmn20.api.element.TimerEventDefinition;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.ItemAwareElement;
import com.ebmwebsourcing.easybpmn.bpmn20.api.with.WithFlowElements;
import com.ebmwebsourcing.easybpmn.bpmn2bpel.wsdl.WSDLGenerator;
import com.ebmwebsourcing.easybpmn.bpmn2bpel.wsdl.WSDLGeneratorHelper;
import com.ebmwebsourcing.easycommons.xml.XMLPrettyPrinter;
import com.ebmwebsourcing.easyplnk20.api.element.PartnerLinkType;
import com.ebmwebsourcing.easyschema10.api.SchemaHelper;
import com.ebmwebsourcing.easyschema10.api.element.Element;
import com.ebmwebsourcing.easyschema10.api.element.Schema;
import com.ebmwebsourcing.easywsdl11.api.Constants;



public class BPELGenerator  {

	public List<BPELProcess> generate(Definitions def, String path) throws BPMNException {
        return generate(def, new GenerationProperties(),path);
    }
	
	public List<BPELProcess> generate(Definitions def) throws BPMNException {
		System.out.println("***** definitions are: "+ def.getId());
		return generate(def, new GenerationProperties(),null);
	}

	public List<BPELProcess> generate(Definitions def, GenerationProperties gp, String writingPath) throws BPMNException {
		List<BPELProcess> res = new ArrayList<BPELProcess>();
		gp.setBPMNDefinitions(def);
		
		WSDLGenerator wsdlGen = new WSDLGenerator();
		Set<com.ebmwebsourcing.easywsdl11.api.element.Definitions> descs = wsdlGen.generate(def, gp);
		gp.setDescs(descs);
		Map<QName,Map<String, XmlObject>> schemaImports = new HashMap<QName, Map<String,XmlObject>>();
		
		for(Collaboration c: def.getCollaborations()) {
			for(Participant participant: c.getParticipant()) {
			    if(!participant.hasProcessRef()) {
			        continue;
			    }
				BPELProcess bpel = this.generate(participant, gp);
				res.add(bpel);
				
				for(com.ebmwebsourcing.easywsdl11.api.element.Definitions defs : descs){
				    addDescriptionImport(bpel, defs);
				}
				
				schemaImports.put(bpel.getQName(), new HashMap<String, XmlObject>());
				for(QName elt : gp.getElementsToImport()){
                    List<Element> elts = bpel.getImports().findElementsInAllSchema(elt);
                    if(elts==null || elts.isEmpty()){
                        for(com.ebmwebsourcing.easybpmn.bpmn20.api.element.Import impt : def.getImports()){
                            if(impt.isXSDImport() && impt.getNamespace().equals(elt.getNamespaceURI())){
                                Schema s = impt.getSchema();
                                if(SchemaHelper.findElementByQName(s, elt)!=null) {
                                    schemaImports.get(bpel.getQName()).put(impt.getLocation(), s);
                                    addSchemaImport(bpel, s, impt.getLocation());
                                    break;
                                }
                            }
                            if(impt.isWSDL11Import() && impt.getNamespace().equals(elt.getNamespaceURI())) {
                                com.ebmwebsourcing.easywsdl11.api.element.Definitions wsdlDefs = impt.getWSDL11Definitions();
                                if(SchemaHelper.findElementByQName(wsdlDefs, elt)!=null) {
                                    schemaImports.get(bpel.getQName()).put(impt.getLocation(), wsdlDefs);
                                    addDescriptionImport(bpel, wsdlDefs);
                                    break;
                                }
                            }
                        }
                    }
                }
			}
		}
		
		if(writingPath!=null){
		    try{
    		    for(com.ebmwebsourcing.easywsdl11.api.element.Definitions defs : descs){
    		        BPELProcessWriterHelper.writeFile(writingPath, defs.getName()+".wsdl", defs);
    		    }
    		    
    		    for(BPELProcess bpel : res){
                    Document doc = BPELFactoryImpl.getInstance().newBPELWriter().getDocument(bpel);
                    BPELProcessWriterHelper.writeFile(writingPath, bpel.getName()+".bpel", XMLPrettyPrinter.prettyPrint(doc));
                    
                    for(String location : schemaImports.get(bpel.getQName()).keySet()){
                        BPELProcessWriterHelper.writeFile(writingPath, BPELGeneratorHelper.getFileNameFromImportLocation(location), schemaImports.get(bpel.getQName()).get(location));
                    }
                }
		    }
		    catch(Exception e){
		        throw new BPMNException("A problem occured during the process writing.",e);
		    }
		}
		
		return res;
	}

	private BPELProcess generate(Participant participant, GenerationProperties gp)
			throws BPMNException {
		Process p = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(),
		        participant.getProcessRef(), Process.class);
		
		String processName = (p.hasName() ? p.getName().replace(" ", "") : p.getId());
		BPELProcess res = new BPELProcessImpl(URI.create(processName), new TProcess(), null, null);
		res.setName(processName);
		res.setTargetNamespace(gp.getBPMNDefinitions().getTargetNamespace());
		gp.setBpelProcess(res);
		
		generatePartnerLinks(participant, gp);
		
		// add variables from messages
		this.generateVariablesFromMessages(getMessagesFromTasks(p.getFlowElementsByClass(Task.class),gp), gp);

		// create Process
		this.generateProcess(res, p, gp);

		return res;
	}
	
	private void generateVariablesFromMessages(List<Message> msgs,
			GenerationProperties gp) throws BPMNException {
		for(Message msg: msgs) {
			this.createOrFindVariableFromMessage(msg, gp);
		}
	}
	
	public void generatePartnerLinks(Participant participant, GenerationProperties gp) {
	    String myRole = null;
	    firstLoop:
	    for(PartnerRole partnerRole : gp.getBPMNDefinitions().getPartnerRoles()){
            if(partnerRole.hasParticipantRef()){
                for(QName participantRef : partnerRole.getParticipantRef()){
                    if(participantRef!=null && participantRef.getLocalPart().equals(participant.getId())){
                        myRole = partnerRole.getName();
                        break firstLoop;
                    }
                }
            }
        }
	    for(PartnerLinkType plt : gp.getArtefactDesc().getAnyXmlObjects(PartnerLinkType.class)) {
	        if(plt.getRoles()!=null && plt.getRoles().length>0) {
	            PartnerLink pl = gp.getBpelProcess().createPartnerLink();
	            String role = plt.getRoles()[0].getName();
	            pl.setName(plt.getName().replace(WSDLGeneratorHelper.PLTSuffix, ""));
	            if(role.equals(myRole)) {
	                pl.setMyRole(role);
	            }
	            else {
	                pl.setPartnerRole(role);
	            }
	            pl.setPartnerLinkType(new QName(gp.getArtefactDesc().getTargetNamespace(), plt.getName()));
	        }
	    }
	}

	private List<Message> getMessagesFromTasks(List<Task> tasks, GenerationProperties gp) throws BPMNException {
		List<Message> msgs = new ArrayList<Message>();
		for(Task t: tasks) {
			if(t instanceof SendTask) {
			    msgs.add(DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), ((SendTask)t).getMessageRef(), Message.class));
			} else if(t instanceof ReceiveTask) {
				msgs.add(DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), ((ReceiveTask)t).getMessageRef(), Message.class));
			} else if(t instanceof ServiceTask) {
			    Operation op = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), ((ServiceTask)t).getOperationRef(), Operation.class);
			    msgs.add(DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), op.getInMessageRef(), Message.class));
			    if(op.hasOutMessageRef()){
			        msgs.add(DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), op.getOutMessageRef(), Message.class));
			    }
			} 
		}
		return msgs;
	}
	
	private void generateProcess(BPELProcess bpel, Process bpmn,
			GenerationProperties gp) throws BPMNException {
		try {
			// create main sequence
			Sequence seqMain = new SequenceImpl(new TSequence(), bpel);
			seqMain.setName("main");
			bpel.setActivity(seqMain);

			// create pick
			List<FlowNode> starts = this.detectAllStartingNodes(bpmn); 
			//Set<FlowNode> starts = BPELGeneratorHelper.getAllStartingNodes(bpmn);
			
			gp.setLoopers(BPELGeneratorHelper.findLoopDivergers(bpmn, starts));
			
			if(starts.size()==1) {
			    generateActivity(starts.iterator().next(), seqMain, gp);
			}
			else if(starts.size()>1){
			    generateStartingPick(starts, seqMain, gp);
			}

		} catch (BPELException e) {
			throw new BPMNException(e);
		}

	}
	
	public List<FlowNode> detectAllStartingNodes(WithFlowElements bpmn) {
	    List<FlowNode> l = new ArrayList<FlowNode>();
	    for(FlowNode fn : bpmn.getFlowElementsByClass(FlowNode.class)) {
	        if(fn instanceof StartEvent) {
	            if(!((StartEvent) fn).hasEventDefinition() || ((StartEvent) fn).getEventDefinition().length==0) {
	                l.addAll(fn.getOutgoingFlowNodes());
	            }
	            else{
	                l.add(fn);
	            }
	        }
	        else if(fn.getIncomingFlowNodes().isEmpty()) { //TODO this condition is not good : loops might loop on a start receive task 
	            l.add(fn);
	        }
	    }
	    return l;
	}

	private Activity generateNextActivity(FlowElement a, BPELElement parent,
			GenerationProperties gp) throws BPELException, BPMNException {
		System.out.println("************* FlowElement: " + a.getName()
				+ " --- instance of class: " + a.getClass());
		Activity ua = null;
		if(a instanceof com.ebmwebsourcing.easybpmn.bpmn20.api.element.Activity) {
		    com.ebmwebsourcing.easybpmn.bpmn20.api.element.Activity t = (com.ebmwebsourcing.easybpmn.bpmn20.api.element.Activity)a;
			System.out.println("************* Taskes: "+ t.getName());
			if(t.getOutgoingFlowNodes().size() > 1) {
				Flow f = generateFlow(t, parent, gp);
				ua = f;
			} else if(t.getOutgoingFlowNodes().size() == 1) {
				Sequence seq = generateSequence(t, parent, gp);
				ua = seq;
			} 
		} else if(a instanceof Gateway) {
			Gateway g = (Gateway)a;
			System.out.println("************* gateways: "+ g.getName());
			if(g.getOutgoingFlowNodes().size() > 1) {
				Activity f = generateActivityFromGateway(g, (Activity) parent, gp);
				ua = f;
			} else if(g.getOutgoingFlowNodes().size() == 1) {
				// end of gateway
				Sequence seq = new SequenceImpl(new TSequence(), parent);
				this.addActivityInParent(seq, parent);
				ua = generateActivity(g.getOutgoingFlowNodes().get(0), seq, gp);
			} 
		}
		else if (a instanceof Event){
		    Event e = (Event)a;
		    System.out.println("************* events: "+ e.getName());
		    if(e.getOutgoingFlowNodes().size() !=0 ){
		        if(e.getOutgoingFlowNodes().size() > 1) {
		            Flow f = generateFlow(
		                    (com.ebmwebsourcing.easybpmn.bpmn20.api.element.Activity) a,
		                    parent, gp);
		            ua = f;
		        } else if(e.getOutgoingFlowNodes().size() == 1) {
		            // end of gateway
		            ua = generateActivity(e.getOutgoingFlowNodes().get(0), (Activity)parent, gp);
		        }
		    }
		}
		return ua;
	}

	private Sequence generateSequence(FlowNode a, BPELElement parent,
			GenerationProperties gp) throws BPELException, BPMNException {
		Sequence seq = null;
		System.out.println("******* Flow node is: "+ a.getName());
		if(parent instanceof Sequence) {
			seq = (Sequence) parent;
		} else {
			seq = new SequenceImpl(new TSequence(), parent);
			this.addActivityInParent(seq, parent);
		}
		
		generateActivity((FlowElement)a.getOutgoingFlowNodes().get(0), seq, gp);

		return seq;
	}

	private Flow generateFlow(
			com.ebmwebsourcing.easybpmn.bpmn20.api.element.Activity a,
			BPELElement parent, GenerationProperties gp) throws BPELException, BPMNException {
		Flow flow = new FlowImpl(new TFlow(), parent);
		this.addActivityInParent(flow, (Activity) parent);

		for(FlowElement elmt: a.getOutgoingFlowNodes()) {
			generateActivity(elmt, (Activity)flow, gp);
		}

		return flow;
	}

	private Activity generateActivity(FlowElement flowElement,
			Activity seqOrFlowOrScope, GenerationProperties gp)
			throws BPELException, BPMNException {
		if(flowElement instanceof Task) {
			return generateActivityFromTask((Task)flowElement, seqOrFlowOrScope, gp);
		} else if(flowElement instanceof CallActivity) {
		    return generateScope((CallActivity)flowElement, seqOrFlowOrScope, gp);
		} else if(flowElement instanceof Gateway) {
			return generateActivityFromGateway((Gateway) flowElement, seqOrFlowOrScope, gp);
		} else if(flowElement instanceof EndEvent) {
			// do nothing
		    return null;
		} else if(flowElement instanceof Event) {
			return generateActivityFromEvent((Event) flowElement, seqOrFlowOrScope, gp);
		} else {
			throw new NotImplementedException("Sorry but this element: " + flowElement.getClass() + " cannot be generated into a bpel activity");
		}
	}
	
	
	private Activity generateActivityFromEvent(Event a,
	        Activity seqOrFlowOrScope, GenerationProperties gp) throws BPELException, BPMNException {
	    Activity res = null;
	    if(a instanceof CatchEvent) {
	        EventDefinition[] eds = ((CatchEvent) a).getEventDefinition();
	        for (int i=0;i<eds.length;i++){
	            if (eds[i] instanceof MessageEventDefinition){
	                // generate receive
	                List<FlowNode> outgoing = a.getOutgoingFlowNodes();
	                if(outgoing.size()==1 && (outgoing.get(0) instanceof ReceiveTask)) {
	                    res = null;
	                }
	                else{
	                    res = this.generateReceiveIntEvent((CatchEvent)a, (MessageEventDefinition)eds[i], seqOrFlowOrScope, gp);
	                }
	            }else if (eds[i] instanceof TimerEventDefinition){
	                //generate wait
	                res = this.generateWait(a, (TimerEventDefinition)eds[i], seqOrFlowOrScope, gp);
	            }
	            //TODO other events
	        }
	        
	        generateAssign(a.getName(), ((CatchEvent)a).getDataOutputAssociation(), seqOrFlowOrScope, gp);
	    }
	    else if(a instanceof ThrowEvent) {
	        generateAssign(a.getName(), ((ThrowEvent)a).getDataInputAssociation(), seqOrFlowOrScope, gp);
	        
	        EventDefinition[] eds = ((ThrowEvent) a).getEventDefinition();
	        for (int i=0;i<eds.length;i++){
	            if (eds[i] instanceof MessageEventDefinition){
	                // generate reply
	                List<FlowNode> incoming = a.getIncomingFlowNodes();
	                if(incoming.size()==1 && (incoming.get(0) instanceof SendTask)) {
	                    res = null;
	                }
	                else {
	                    res = this.generateSendIntEvent((ThrowEvent)a, (MessageEventDefinition)eds[i], seqOrFlowOrScope, gp);
	                }
	            }
	            else if(eds[i] instanceof TerminateEventDefinition) {
	                Exit exit = new ExitImpl(new TExit(), seqOrFlowOrScope);
	                addActivityInParent(exit, seqOrFlowOrScope);
	            }
	            //TODO other events
	        }
	    }
	    // generate next activities
	    this.generateNextActivity(a, seqOrFlowOrScope, gp);
		return res;
	}

	private Wait generateWait(WithName wn, TimerEventDefinition ed, Activity seqOrFlowOrScope, GenerationProperties gp) throws BPELException{
	    Wait wait = new WaitImpl(new TWait(), seqOrFlowOrScope);
	    this.addActivityInParent(wait, (Activity) seqOrFlowOrScope);
	    wait.setName(wn.getName());
	    
	    if(ed.hasTimeDate()){
	        BPELDeadLineExpression exp = new BPELDeadLineExpressionImpl(new TDeadlineExpr(), wait);
	        exp.setContent(ed.getTimeDate().getTextContent());
	        wait.setUntil(exp);
	    }
	    else if(ed.hasTimeDuration()) {
	        BPELDurationExpression exp = new BPELDurationExpressionImpl(new TDurationExpr(), wait);
	        exp.setContent(ed.getTimeDuration().getTextContent());
	        wait.setFor(exp);
	    }
        return wait;
	}
	
	public Activity generateSendIntEvent(ThrowEvent e, MessageEventDefinition ed,
			Activity seqOrFlowOrScope, GenerationProperties gp)
			throws BPELException, BPMNException {
	    if(e instanceof EndEvent) {
	        return generateReply(e.getName(), ed.getOperationRef(), seqOrFlowOrScope, gp);
	    }
	    else if(e instanceof IntermediateThrowEvent) {
	        return generateInvoke(e.getName(), ed.getOperationRef(), false, seqOrFlowOrScope, gp);
	    }
	    throw new BPMNException("Unknown type of event : "+e.getClass());
	}

	public Receive generateReceiveIntEvent(CatchEvent a, MessageEventDefinition ed,
			Activity seqOrFlowOrScope, GenerationProperties gp)
			throws BPELException, BPMNException {
	    assert a!=null;
		return generateReceive(a.getName(),false,ed.getOperationRef(),seqOrFlowOrScope,gp);
	}

	private Activity generateActivityFromGateway(Gateway gateway,
			Activity seqOrFlowOrScope, GenerationProperties gp)
			throws BPELException, BPMNException {
		Activity res = null;
		if(gateway instanceof ExclusiveGateway) {
			ExclusiveGateway eg = (ExclusiveGateway)gateway;
			
			if(gp.getAlreadyGeneratedLoops().contains(gateway)){
				if(gp.getAlreadyGeneratedLoopExit().contains(gateway)){
					return null;
				}
				gp.getAlreadyGeneratedLoopExit().add(eg);
				return generateActivity(gp.getLoopExit(eg), gp.getCurrentWhileParent(), gp);
			}
			
			if(gp.isLoopDiverger(eg)){
				gp.getAlreadyGeneratedLoops().add(eg);
				generateWhile(eg,gp.getLoopingFlows(eg),seqOrFlowOrScope,gp);
			}
			else{
			    List<FlowNode> outgoing = eg.getOutgoingFlowNodes();
			    if(outgoing.size()==1) {
			        res = generateActivity(outgoing.get(0), seqOrFlowOrScope, gp);
			    }
			    else {
			        res = this.generateIf(eg, seqOrFlowOrScope, gp);
			    }
			}
		} 
		else if(gateway instanceof ParallelGateway) {
			res = this.generateFlow((ParallelGateway)gateway, seqOrFlowOrScope, gp);
		}
		else if(gateway instanceof EventBasedGateway) {
		    if(gp.getAlreadyGeneratedLoops().contains(gateway)) {
		        return null;
		    }
		    if(gp.isLoopDiverger(gateway)){
		        gp.getAlreadyGeneratedLoops().add(gateway);
		        return generateWhile((EventBasedGateway)gateway, seqOrFlowOrScope, gp);
		    }
		    else {
		        res = this.generatePick((EventBasedGateway)gateway, seqOrFlowOrScope, gp);
		    }
		}
		else {
			throw new NotImplementedException("Sorry but this element: " + gateway.getClass() + " cannot be generated into a bpel activity");
		}
		return res;
	}
	
	private Activity generateStartingPick(Collection<FlowNode> starts, Activity parent,
            GenerationProperties gp) throws BPELException, BPMNException {
	    Pick pick  = new PickImpl(new TPick(), parent);        
        this.addActivityInParent(pick, parent);
	    for(FlowNode fn: starts) {
            if(fn instanceof CatchEvent) {
                EventDefinition ed = ((CatchEvent)fn).getEventDefinition()[0];
                if(ed instanceof MessageEventDefinition) {
                    generateOnMessage(fn, ((MessageEventDefinition)ed).getOperationRef(),
                            ((CatchEvent)fn).getDataOutputAssociation(), null, pick, gp);
                }
                else {
                	generateOnAlarm(fn, ((TimerEventDefinition)ed), pick, gp);
                }
            }
            else if(fn instanceof ReceiveTask) {
                generateOnMessage((ReceiveTask) fn, pick, gp);
            }
            else if(fn instanceof EventBasedGateway) {
                return generatePick((EventBasedGateway)fn, parent, gp);
            }
            else {
                throw new BPMNException("A BPEL process should start with catch events or receive tasks. "+fn.getClass());
            }
	    }
	    return pick;
	}
	
	private Activity generatePick(EventBasedGateway gateway, Activity parent,
            GenerationProperties gp) throws BPELException, BPMNException {
	    Pick pick  = new PickImpl(new TPick(), parent);        
        this.addActivityInParent(pick, parent);
        FlowElement loopExit = null;
        if(gp.isLoopDiverger(gateway)) {
            loopExit = gp.getLoopExit(gateway);
        }
        for(FlowNode fn: gateway.getOutgoingFlowNodes()) {
            BPELElement e = null;
            if(fn instanceof IntermediateCatchEvent) {
                EventDefinition ed = ((IntermediateCatchEvent)fn).getEventDefinition()[0];
                if(ed instanceof MessageEventDefinition) {
                    e = generateOnMessage(fn, ((MessageEventDefinition)ed).getOperationRef(),
                            ((IntermediateCatchEvent)fn).getDataOutputAssociation(), null, pick, gp);
                }
                else if(ed instanceof TimerEventDefinition) {
                	e = generateOnAlarm(fn, ((TimerEventDefinition)ed), pick, gp);
                }
                else {
                    throw new BPMNException("An event based gateway should be connected to intermediate message catching events or to receive tasks.");
                }
            }
            else if(fn instanceof ReceiveTask) {
                e = generateOnMessage((ReceiveTask) fn, pick, gp);
            }
            else {
                throw new BPMNException("An event based gateway should be connected to intermediate message catching events or to receive tasks.");
            }
            if(loopExit!=null && loopExit.equals(fn)) {
                BPELVariable<?> var = createOrFindBooleanVariable(gateway, gp);
                Assign assign = new AssignImpl(new TAssign(), e);
                Copy copy = new CopyImpl(new TCopy(), assign);
                From from = new FromImpl(new TFrom(), copy);
                from.setContent("true");
                To to = new ToImpl(new TTo(), copy);
                to.setContent("$"+var.getQName().getLocalPart());
                copy.setFrom(from);
                copy.setTo(to);
                assign.addCopy(copy);
                this.addActivityInParent(assign, e);
            }
        }
        return pick;
    }
	
	private Scope generateScope(CallActivity callActi, Activity seqOrFlowOrScope, 
	        GenerationProperties gp) throws BPELException, BPMNException {
	    QName qn = callActi.getCalledElement();
	    Process proc = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), qn, Process.class);
	    
	    Scope scope = new ScopeImpl(new TScope(), seqOrFlowOrScope);
	    this.addActivityInParent(scope, seqOrFlowOrScope);
	    
	    // create main sequence
	    Sequence seqMain = new SequenceImpl(new TSequence(), scope);
	    this.addActivityInParent(seqMain, scope);

	    Definitions defs = gp.getBPMNDefinitions();
	    gp.setBPMNDefinitions(DefinitionsHelper.getParentDefinitions(proc));
	    
	    // create pick
	    List<FlowNode> starts = this.detectAllStartingNodes(proc); 
	    if(starts.size()==1) {
	        generateActivity(starts.iterator().next(), seqMain, gp);
	    }
	    else if(starts.size()>1){
	        generateStartingPick(starts, seqMain, gp);
	    }
	    
	    gp.setBPMNDefinitions(defs);
	    
	    this.generateNextActivity(callActi, seqOrFlowOrScope, gp);
	    
	    return scope;
	}
	
	private While generateWhile(EventBasedGateway ebg, Activity seqOrFlowOrScope,
	        GenerationProperties gp) throws BPELException, BPMNException {
	    TWhile model = new TWhile();
        While while_ = new WhileImpl(model, seqOrFlowOrScope);
        while_.setName(ebg.getName());
        
        gp.setCurrentWhileParent(seqOrFlowOrScope);
        
        BPELVariable<?> var = createOrFindBooleanVariable(ebg, gp);
        Assign assign = new AssignImpl(new TAssign(), seqOrFlowOrScope);
        Copy copy = new CopyImpl(new TCopy(), assign);
        From from = new FromImpl(new TFrom(), copy);
        from.setContent("false");
        To to = new ToImpl(new TTo(), copy);
        to.setContent("$"+var.getQName().getLocalPart());
        copy.setFrom(from);
        copy.setTo(to);
        assign.addCopy(copy);
        this.addActivityInParent(assign, seqOrFlowOrScope);
        
        TBooleanExpr expr = new TBooleanExpr();
        BPELBooleanExpression cond;
        try {
            cond = new BPELBooleanExpressionImpl(expr, while_);
        } catch (XPathExpressionException e) {
            throw new BPELException(e);
        }
        String loopCondition = "not($"+var.getQName().getLocalPart()+")";
        cond.setContent(loopCondition);
        while_.setCondition(cond);
        model.setCondition(expr);
        
        this.addActivityInParent(while_, seqOrFlowOrScope);
        
        this.generatePick(ebg, while_, gp);
        
        return while_;
	}

    private void generateWhile(ExclusiveGateway eg, Set<SequenceFlow> loopingFlows,
			Activity seqOrFlowOrScope, GenerationProperties gp) throws BPELException, BPMNException {
		System.out.println("*************While for : "+eg.getId());
		
		TWhile model = new TWhile();
		While while_ = new WhileImpl(model, seqOrFlowOrScope);
		while_.setName(eg.getName());
		
		gp.setCurrentWhileParent(seqOrFlowOrScope);
		
		TBooleanExpr expr = new TBooleanExpr();
		BPELBooleanExpression cond;
		try {
			cond = new BPELBooleanExpressionImpl(expr, while_);
		} catch (XPathExpressionException e) {
			throw new BPELException(e);
		}
		
		Process p = DefinitionsHelper.getProcess(eg, gp.getBPMNDefinitions());
		SequenceFlow nonlooper = null;
		for(SequenceFlow sf : p.getFlowElementsByClass(SequenceFlow.class)){
			if(sf.getSourceRef().getId().equals(eg.getId()) && !loopingFlows.contains(sf)){
				nonlooper = sf;
				break;
			}
		}
		String loopCondition = "not("+nonlooper.getConditionExpression().getTextContent()+")";
		
		cond.setContent(loopCondition);
		while_.setCondition(cond);
		model.setCondition(expr);
		
		addActivityInParent(while_, seqOrFlowOrScope);
		
		generateWhileInsideIf(eg, loopingFlows, while_ , model, gp);
	}
	
	
	private void generateWhileInsideIf(ExclusiveGateway eg, Collection<SequenceFlow> loopingFlows, While while_ ,
			TWhile model, GenerationProperties gp) throws BPELException, BPMNException {
		TSequence sm = new TSequence();
		Sequence seq = new SequenceImpl(sm, while_);
		while_.setActivity(seq);
		model.setSequence(sm);
		
		List<FlowNode> activities = new ArrayList<FlowNode>();
		for(SequenceFlow sf : loopingFlows){
			activities.add(sf.getTargetRef());
		}
		
		generateIf(activities, seq, eg,  gp);
	}
	
	
	private Flow getCurrentFlow(final BPELElement parent) {
		Flow res = null;
		if(parent != null) {
			if(parent instanceof Flow) {
				res = (Flow) parent;
			} else {
				res = getCurrentFlow((BPELElement) ((AbstractSchemaElementImpl<?>)parent).getParent());
			}
		}
		return res;
	}

	private If getCurrentIf(final BPELElement parent) {
		If res = null;
		if(parent != null) {
			if(parent instanceof If) {
				res = (If) parent;
			} else {
				res = getCurrentIf((BPELElement) ((AbstractSchemaElementImpl<?>)parent).getParent());
			}
		}
		return res;
	}

	private Flow generateFlow(ParallelGateway gateway,
			Activity seqOrFlowOrScope, GenerationProperties gp)
			throws BPELException, BPMNException {
		Flow flowA = null;
			System.out.println("found gateways: "+ gateway.getId());
			List<FlowNode> activities = gateway.getOutgoingFlowNodes();
			if(activities.size() > 0 && gateway.getIncomingFlowNodes().size() == 1) {
				flowA = new FlowImpl(new TFlow(), seqOrFlowOrScope);
				this.addActivityInParent(flowA, (Activity) seqOrFlowOrScope);
				System.out.println("flow out size = " + activities.size());
				for(FlowElement elmt: activities) {
					Sequence seq = new SequenceImpl(new TSequence(), flowA);
					this.addActivityInParent(seq, flowA);
					this.generateActivity(elmt, seq, gp);
				}			
				
			} else {
				// save end of gateway
				if(!gp.getGatewaysAlreadyGenerated().contains(gateway)) {
					gp.getGatewaysAlreadyGenerated().add(gateway);
					this.generateNextActivity(gateway, (BPELElement) ((BPELElementImpl<?>)getCurrentFlow(seqOrFlowOrScope)).getParent(), gp);
				}
			}		
		return flowA;
	}
	
	private If generateIf(List<FlowNode> activities, Activity seqOrFlowOrScope,
			ExclusiveGateway eg, GenerationProperties gp) throws BPELException, BPMNException {
		
		If ifA = new IfImpl(new TIf(), seqOrFlowOrScope);
		this.addActivityInParent(ifA, (Activity) seqOrFlowOrScope);
		System.out.println("if out size = " + activities.size());
		boolean first = true;
		for(FlowElement elmt: activities) {
			Sequence seq = new SequenceImpl(new TSequence(), ifA);
			
			this.generateActivity(elmt, seq, gp);
			
			try{
			    System.out.println("elmt.getId().toString() = " + elmt.getId().toString());
			    SequenceFlow sf = DefinitionsHelper.getSequenceFlow(DefinitionsHelper.getProcess(elmt, gp.getBPMNDefinitions()), eg.getId() , elmt.getId());
			    
			    if(sf.hasConditionExpression()) {
			        if(first) {
	                    BPELBooleanExpression exp = new BPELBooleanExpressionImpl(new TBooleanExpr(), ifA);
	                    exp.setContent(sf.getConditionExpression().getTextContent());
	                    ifA.addIfActivity(exp, seq);
	                    first = false;
	                } else {
	                    BPELBooleanExpression exp = new BPELBooleanExpressionImpl(new TBooleanExpr(), ifA);
	                    exp.setContent(sf.getConditionExpression().getTextContent());
	                    ifA.addElseIfActivity(exp, seq);
	                }
			    }
			    else if(eg.hasDefault() && ((SequenceFlow)eg.getDefault()).getTargetRef().getId().toString().equals(elmt.getId().toString())) {
			        ifA.setElseActivity(seq);
			    }
			    else {
			        throw new BPMNException("A sequence flow going out of an exclusive gateway must either have a condition or be the default of the gateway.");
			    }
			}
			catch(XPathExpressionException e) {
			    throw new BPELException(e);
			}
		}
		
		return ifA;
	}

	private If generateIf(ExclusiveGateway gateway, Activity seqOrFlowOrScope,
	        GenerationProperties gp) throws BPELException, BPMNException {
	    System.out.println("found gateways: "+ gateway.getId());
	    List<FlowNode> activities = gateway.getOutgoingFlowNodes();

	    if(activities.size() > 0 && gateway.getIncomingFlowNodes().size() == 1) {
	        return generateIf(activities, seqOrFlowOrScope, gateway, gp);
	    } else {
	        // save end of gateway
	        if(!gp.getGatewaysAlreadyGenerated().contains(gateway)) {
	            gp.getGatewaysAlreadyGenerated().add(gateway);
	            this.generateNextActivity(gateway, (BPELElement) ((BPELElementImpl<?>)getCurrentIf(seqOrFlowOrScope)).getParent(), gp);
	        }
	    }
	    return null;
	}

	private Activity generateActivityFromTask(Task a, Activity seqOrFlowOrScope,
			GenerationProperties gp) throws BPELException, BPMNException {
		Activity res = null;
		if(a != null) {

		    if(a.hasDataInputAssociation()) {
                this.generateAssign(a.getName(), a.getDataInputAssociation(), seqOrFlowOrScope, gp); 
            }

			if(a instanceof ServiceTask) {
				// generate invoke
				Invoke invoke = this.generateInvoke((ServiceTask)a, seqOrFlowOrScope, gp);
				res = invoke;
			} else if(a instanceof ReceiveTask) {
				// generate receive
				Receive receive = this.generateReceive((ReceiveTask) a, seqOrFlowOrScope, gp);
				res = receive;
			} else if(a instanceof SendTask) {
				// generate invoke or reply
				if(isProcessEndingSendTask((SendTask)a)){
					Reply reply = this.generateReply((SendTask)a, seqOrFlowOrScope, gp);
					res = reply;
				}
				else{
					res = this.generateInvoke((SendTask)a, seqOrFlowOrScope, gp);
				}
			} else {
				// generate empty
				Empty empty = this.generateEmpty(a, seqOrFlowOrScope, gp);
				res = empty;
			}

			if(a.hasDataOutputAssociation()) {
                this.generateAssign(a.getName(), a.getDataOutputAssociation(), seqOrFlowOrScope, gp); 
            }

			// generate next activities
			this.generateNextActivity(a, seqOrFlowOrScope, gp);

		}
		return res;
	}
	
	
	private boolean isProcessEndingSendTask(SendTask st){
		if(st.getOutgoingFlowNodes()==null || st.getOutgoingFlowNodes().isEmpty()){
			return true;
		}
		else{
			boolean allEndEvent = true;
			for(FlowElement a : st.getOutgoingFlowNodes()){
				if(!(a instanceof EndEvent)){
					allEndEvent = false;
					break;
				}
			}
			return allEndEvent;
		}
	}
	

	public Receive generateReceive(ReceiveTask a, 
			Activity seqOrFlowOrScope, GenerationProperties gp)
			throws BPELException, BPMNException {
		assert a != null;
		return generateReceive(a.getName(), a.isInstantiate(), a.getOperationRef(), seqOrFlowOrScope, gp);
	}
	
	private Receive generateReceive(String name, boolean createInstance, QName operationRef,
            Activity seqOrFlowOrScope, GenerationProperties gp)
            throws BPELException, BPMNException {
	    Receive receive = new ReceiveImpl(new TReceive(), seqOrFlowOrScope);
        this.addActivityInParent(receive, (Activity) seqOrFlowOrScope);
        receive.setName(name);
        receive.setCreateInstance(createInstance);
        
        Operation op = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), operationRef, Operation.class);
        Interface itf = (Interface) op.getXmlObjectParent();
        if(itf.hasImplementationRef()) {
            receive.setInterface(itf.getImplementationRef());
            receive.setOperation(op.getImplementationRef().getLocalPart());
        }
        else {
            receive.setOperation(op.getName());
            receive.setInterface(new QName(gp.getBPMNDefinitions().getTargetNamespace(),itf.getName()));
        }
        receive.setPartnerLink(BPELGeneratorHelper.findPartnerLinkFromInterface(
                receive.getInterface(), gp.getBpelProcess(), gp.getArtefactDesc()).getName());
        
        Message msgIn = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), op.getInMessageRef(), Message.class);
        BPELVariable<?> inputVariable = createOrFindVariableFromMessage(msgIn, gp);
        receive.setInputVariable(inputVariable.getQName().getLocalPart());
        
        return receive;
	}

	public Invoke generateInvoke(ServiceTask a, 
			Activity seqOrFlowOrScope, GenerationProperties gp)
			throws BPELException, BPMNException {
	    return generateInvoke(a.getName(), a.getOperationRef(), true, seqOrFlowOrScope, gp);
	}

	public Invoke generateInvoke(SendTask a, 
			Activity seqOrFlowOrScope, GenerationProperties gp)
			throws BPELException, BPMNException {
		return generateInvoke(a.getName(), a.getOperationRef(), false, seqOrFlowOrScope, gp);
	}
	
	private Invoke generateInvoke(String name, QName operationRef, boolean outputVariable,
	        Activity seqOrFlowOrScope, GenerationProperties gp) throws BPELException, BPMNException {
	    Invoke invoke = new InvokeImpl(new TInvoke(), seqOrFlowOrScope);
        this.addActivityInParent(invoke, (Activity) seqOrFlowOrScope);
        invoke.setName(name);
        
        Operation op = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), operationRef, Operation.class);
        Interface itf = (Interface) op.getXmlObjectParent();
        if(itf.hasImplementationRef()) {
            invoke.setInterface(itf.getImplementationRef());
            invoke.setOperation(op.getImplementationRef().getLocalPart());
        }
        else {
            invoke.setOperation(op.getName());
            invoke.setInterface(new QName(gp.getBPMNDefinitions().getTargetNamespace(),itf.getName()));
        }
        invoke.setPartnerLink(BPELGeneratorHelper.findPartnerLinkFromInterface(
                invoke.getInterface(), gp.getBpelProcess(), gp.getArtefactDesc()).getName());
        
        Message msgIn = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), op.getInMessageRef(), Message.class);
        BPELVariable<?> inputVariable = createOrFindVariableFromMessage(msgIn, gp);
        invoke.setInputVariable(inputVariable.getQName().getLocalPart());
        
        if(outputVariable) {
            Message msgOut = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), op.getOutMessageRef(), Message.class);
            BPELVariable<?> outputVar = createOrFindVariableFromMessage(msgOut, gp);
            invoke.setOutputVariable(outputVar.getQName().getLocalPart());
        }
        
        return invoke;
	}

	public Empty generateEmpty(Task a, Activity seqOrFlowOrScope,
			GenerationProperties gp) throws BPELException {
		Empty empty = new EmptyImpl(new TEmpty(), seqOrFlowOrScope);
		this.addActivityInParent(empty, (Activity) seqOrFlowOrScope);
		empty.setName(a.getName());
		return empty;
	}

	public Reply generateReply(SendTask a, Activity seqOrFlowOrScope,
			GenerationProperties gp) throws BPELException, BPMNException {
		return generateReply(a.getName(), a.getOperationRef(), seqOrFlowOrScope, gp);
	}
	
	private Reply generateReply(String name, QName operationRef, Activity seqOrFlowOrScope,
            GenerationProperties gp) throws BPELException, BPMNException {
	    Reply reply = new ReplyImpl(new TReply(), seqOrFlowOrScope);
        this.addActivityInParent(reply, (Activity) seqOrFlowOrScope);
        reply.setName(name);

        Operation op = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), operationRef, Operation.class);
        Interface itf = (Interface) op.getXmlObjectParent();
        if(itf.hasImplementationRef()) {
            reply.setInterface(itf.getImplementationRef());
            reply.setOperation(op.getImplementationRef().getLocalPart());
        }
        else {
            reply.setOperation(op.getName());
            reply.setInterface(new QName(gp.getBPMNDefinitions().getTargetNamespace(),itf.getName()));
        }
        reply.setPartnerLink(BPELGeneratorHelper.findPartnerLinkFromInterface(
                reply.getInterface(), gp.getBpelProcess(), gp.getArtefactDesc()).getName());
        
        Message msg = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), op.getOutMessageRef(), Message.class);
        BPELVariable<?> var = this.createOrFindVariableFromMessage(msg, gp);
        reply.setOutputVariable(var.getQName().getLocalPart());

        return reply;
	}

	private void addActivityInParent(BPELElement child,
			BPELElement seqOrFlowOrScope)
			throws BPELException {
		if (seqOrFlowOrScope instanceof Sequence) {
			((Sequence) seqOrFlowOrScope).addActivity((Activity) child);
		} else if (seqOrFlowOrScope instanceof Flow) {
			((Flow) seqOrFlowOrScope).addActivity((Activity) child);
		} else if (seqOrFlowOrScope instanceof OnMessage) {
			((OnMessage) seqOrFlowOrScope).setActivity((Activity) child);
		} else if (seqOrFlowOrScope instanceof OnAlarm) {
			((OnAlarm) seqOrFlowOrScope).setActivity((Activity) child);
		} else if (seqOrFlowOrScope instanceof Pick) {
			if(child instanceof OnAlarm) {
				((Pick) seqOrFlowOrScope).addOnAlarm((OnAlarm) child);
			}
			else if(child instanceof OnMessage) {
				((Pick) seqOrFlowOrScope).addOnMessage((OnMessage) child);
			}
		} else if(seqOrFlowOrScope instanceof While) {
			((While)seqOrFlowOrScope).setActivity((Activity)child);
		} else if(seqOrFlowOrScope instanceof Scope) {
		    ((Scope)seqOrFlowOrScope).setActivity((Activity)child);
		}
		else {
			throw new BPELException("Sorry but this type of element is not handled yet : "+seqOrFlowOrScope.getClass());
		}
	}

	private Assign generateAssign(String activityName,
			DataAssociation[] dataAssociations, Activity seqOrFlowOrScope,
			GenerationProperties gp) throws BPELException, BPMNException {
		Assign assign = null;
		if(dataAssociations.length > 0) {
			assign = new AssignImpl(new TAssign(), seqOrFlowOrScope);
			for(DataAssociation asso: dataAssociations) {
			    Map<String,String> variableMapping = new HashMap<String,String>();
			    for(ItemAwareElement iae : asso.getSourceRef()) {
			        ItemDefinition itemDef = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(),
			                iae.getItemSubjectRef(), ItemDefinition.class);
			        BPELVariable<?> var = createOrFindVariableFromElement(itemDef.getStructureRef(), gp);
			        variableMapping.put(((BaseElement)iae).getId(), var.getQName().getLocalPart());
			    }
			    ItemDefinition itemDef = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(),
                        asso.getTargetRef().getItemSubjectRef(), ItemDefinition.class);
			    BPELVariable<?> var = createOrFindVariableFromElement(itemDef.getStructureRef(), gp);
			    
				for(Assignment a: asso.getAssignment()) {
					Copy copy = new CopyImpl(new TCopy(), assign);
					
					From from = new FromImpl(new TFrom(), copy);
			        from.setContent(generateExpression(a.getFrom().getTextContent(),variableMapping));
			        copy.setFrom(from);
					
			        variableMapping.clear();
			        variableMapping.put(((BaseElement)asso.getTargetRef()).getId(), var.getQName().getLocalPart());
			        To to = new ToImpl(new TTo(), copy);
			        to.setContent(generateExpression(a.getTo().getTextContent(),variableMapping));
			        copy.setTo(to);

					assign.addCopy(copy);
				}
				
				if(asso.hasTransformation()) {
				    System.out.println("WARNING : the transformations defined in BPMN dataAssociations are not handled yet.");
				}
			}
			this.addActivityInParent(assign, seqOrFlowOrScope);
		}
		return assign;
	}
	
    private String generateExpression(String xpathExpression, Map<String,String> mapping){
        String result = xpathExpression;
        for(String associationExtrmityId : mapping.keySet()) {
            result = result.replace("$"+associationExtrmityId, "$"+mapping.get(associationExtrmityId));
        }
        return result;
    }

	private BPELVariable<?> createOrFindVariableFromElement(QName element,
			GenerationProperties gp) {
	    gp.getElementsToImport().add(element);
	    
		for (BPELVariable<?> var : gp.getBpelProcess().getVariables()) {
			if (var.getElement() != null && var.getElement().equals(element)) {
			    return var;
			}
		}
		BPELVariable<?> var = gp.getBpelProcess().createBPELElementVariable(element.getLocalPart() + "Variable",element, VariableType.ELEMENT);
		return var;
	}

	private OnMessage generateOnMessage(ReceiveTask rt, Pick pick,
			GenerationProperties gp) throws BPMNException, BPELException {
	    return generateOnMessage(rt, rt.getOperationRef(), rt.getDataOutputAssociation(),
	            rt.getDataInputAssociation(), pick, gp);
	}
	
	private OnMessage generateOnMessage(FlowNode fn, QName operationRef, DataOutputAssociation[] doas,
	        DataInputAssociation[] dias, Pick pick, GenerationProperties gp) throws BPELException, BPMNException {
	    OnMessage onMessage = new OnMessageImpl(new TOnMessage(),(BPELElementImpl<?>) pick);
        this.addActivityInParent(onMessage, pick);
        
        Operation op = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), operationRef, Operation.class);
        Interface itf = (Interface) op.getXmlObjectParent();
        if(itf.hasImplementationRef()) {
            onMessage.setInterface(itf.getImplementationRef());
            onMessage.setOperation(op.getImplementationRef().getLocalPart());
        }
        else {
            onMessage.setOperation(op.getName());
            onMessage.setInterface(new QName(gp.getBPMNDefinitions().getTargetNamespace(),itf.getName()));
        }
        
        onMessage.setPartnerLink(BPELGeneratorHelper.findPartnerLinkFromInterface(
                onMessage.getInterface(), gp.getBpelProcess(),
                gp.getArtefactDesc()).getName());
        
        Message msg = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(), op.getInMessageRef(), Message.class);
        BPELVariable<?> var = this.createOrFindVariableFromMessage(msg,gp);
        onMessage.setInputVariable(var.getQName().getLocalPart());

        Sequence seq = new SequenceImpl(new TSequence(), onMessage);

        if(dias!=null) {
            this.generateAssign(fn.getName(), dias, seq, gp);
        }
        if(doas!=null) {
            this.generateAssign(fn.getName(), doas, seq, gp);
        }
        
        this.generateNextActivity(fn, seq, gp);
        
        if(!seq.getActivities().isEmpty()) {
        	this.addActivityInParent(seq, onMessage);
        }

        return onMessage;
	}
	
	private OnAlarm generateOnAlarm(FlowNode fn, TimerEventDefinition ed, Pick pick, GenerationProperties gp) throws BPELException, BPMNException{
		OnAlarm alarm = new OnAlarmImpl(new TOnAlarmPick(), (BPELElementImpl<?>)pick);
	    this.addActivityInParent(alarm, (Activity) pick);
	    
	    if(ed.hasTimeDate()){
	        BPELDeadLineExpression exp = new BPELDeadLineExpressionImpl(new TDeadlineExpr(), alarm);
	        exp.setContent(ed.getTimeDate().getTextContent());
	        alarm.setUntil(exp);
	    }
	    else if(ed.hasTimeDuration()) {
	        BPELDurationExpression exp = new BPELDurationExpressionImpl(new TDurationExpr(), alarm);
	        exp.setContent(ed.getTimeDuration().getTextContent());
	        alarm.setFor(exp);
	    }
	    
	    Sequence seq = new SequenceImpl(new TSequence(), alarm);
        
        this.generateNextActivity(fn, seq, gp);
        
        if(!seq.getActivities().isEmpty()) {
        	this.addActivityInParent(seq, alarm);
        }
	    
        return alarm;
	}

	private BPELVariable<?> createOrFindVariableFromMessage(Message message, GenerationProperties gp)
	        throws BPMNException {
	    ItemDefinition itemDef = DefinitionsHelper.findBPMNObject(gp.getBPMNDefinitions(),
	            message.getItemRef(), ItemDefinition.class);
	    return createOrFindVariableFromElement(itemDef.getStructureRef(), gp);
	}
	
	private BPELVariable<?> createOrFindBooleanVariable(FlowElement fe, GenerationProperties gp)
	    throws BPMNException {
	    for(BPELVariable<?> var : gp.getBpelProcess().getVariables()) {
	        if(var.getQName().getLocalPart().startsWith(fe.getId())) {
	            return var;
	        }
	    }
	    return gp.getBpelProcess().createBPELElementVariable(fe.getId() + "Variable", new QName(com.ebmwebsourcing.easyschema10.api.Constants.XMLSCHEMA_NS_URI, "boolean"), VariableType.TYPE);
	}
	
	private void addDescriptionImport(BPELProcess p, com.ebmwebsourcing.easywsdl11.api.element.Definitions desc){
		Import imp = new ImportImpl(new TImport(), p);
		imp.setImportType(URI.create(Constants.WSDL11_NS_URI));
		imp.setLocation(URI.create(desc.getName()+".wsdl"));
		imp.setNamespace(URI.create(desc.getTargetNamespace()));
		((ImportImpl)imp).setDescription((Description)Factory.getInstance().wrap(desc));
		
		p.addImport(imp);
	}
	
	private void addSchemaImport(BPELProcess p, Schema schema, String location){
        Import imp = new ImportImpl(new TImport(), p);
        imp.setImportType(URI.create(com.ebmwebsourcing.easyschema10.api.Constants.XMLSCHEMA_NS_URI));
        imp.setLocation(URI.create(location));
        imp.setNamespace(URI.create(schema.getTargetNamespace()));
        ((ImportImpl)imp).setSchema2(schema);
        
        p.addImport(imp);
    }
	
}
