package com.ebmwebsourcing.easycommons.logger;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.IllegalFormatException;
import java.util.Locale;
import java.util.logging.LogRecord;

import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

import com.ebmwebsourcing.easycommons.xml.XMLPrettyPrinter;

public class Formatter extends java.util.logging.Formatter {
	
	private static final int INDENT_LEVEL = 2;
	private static final String INDENT_STRING = createIndentString(INDENT_LEVEL, ' ');
	
	private static final String createIndentString(int indentLevel, char indentChar) {
		String indentString = "";
		for (int i = 0 ; i < indentLevel ; ++i) {
			indentString += indentChar; 
		}
		return indentString;
	}
	
    private final DateFormat dateFormat =
        new SimpleDateFormat("yyyy/MM/dd HH:mm:ss.SSS", Locale.getDefault());
    
    
    private final String simplifySourceClassName(String sourceClassName) {
    	return sourceClassName.replaceAll("com.ebmwebsourcing.", "").replaceAll("org.ow2.", "");
    }
    
    
    private final String indent(String s) {
    	// remove multiple new lines and add indent string
    	return s.replaceAll("\r?\n", "\n" + INDENT_STRING).trim();
    }
    
    private final Object formatMessageParameter(Object obj) {
		String res = "";
		if (obj instanceof org.jdom.Element) {
			res = res
					+ new XMLOutputter(Format.getPrettyFormat())
							.outputString((org.jdom.Element) obj);
		} else if (obj instanceof org.jdom.Document) {
			res = res
					+ new XMLOutputter(Format.getPrettyFormat())
							.outputString((org.jdom.Document) obj);
		} else if (obj instanceof org.w3c.dom.Node) {
			res = res
					+ XMLPrettyPrinter.prettyPrint((org.w3c.dom.Node) obj);
		} else {
			res = res + obj;
		}
		return res.trim();
    }

    private final int getFormatSpecifiersCount(String formatString) {
    	int nbOccurences = 0;
    	int currentIndex = -1;
    	while (true) {
        	currentIndex = formatString.indexOf('%', currentIndex + 1);
        	if (currentIndex == -1) break;
        	++nbOccurences;
    	}
    	return nbOccurences;
    }
    
    private final void formatMessageWithParametersUsingSprintfSyntax(StringBuffer sb, String message, Object[] parameters) {
    	try {
    		sb.append(String.format(message, parameters));
    	} catch (IllegalFormatException ife) {
    		formatMessageWithParametersUsingFallbackSyntax(sb, message, parameters);
    	}
    	
    }
    
    private final void formatMessageWithParametersUsingFallbackSyntax(StringBuffer sb, String message, Object[] parameters) {
		sb.append(message);
		if (parameters.length > 0) {
			sb.append(" : ");
    		for (Object parameter : parameters) {
    			sb.append("\n").append(formatMessageParameter(parameter));
    		}
    		sb.append("\n");
		}
    }
    
    
    private final void formatMessageWithParameters(StringBuffer sb, LogRecord logRecord) {
    	Object[] parameters = logRecord.getParameters();
    	if (parameters == null) {
    		parameters = new Object[0];
    	}
    	for (int i = 0 ; i < parameters.length; ++i) {
    		parameters[i] = formatMessageParameter(parameters[i]);
    	}
    	
    	String message = logRecord.getMessage();
    	// we use sprintf syntax only in cases where nb of args is confirm to string format.
    	if (getFormatSpecifiersCount(message) == parameters.length) {
    		formatMessageWithParametersUsingSprintfSyntax(sb, message, parameters);
    	} else {
    		formatMessageWithParametersUsingFallbackSyntax(sb, message, parameters);
    	}
    	if (logRecord.getSourceClassName() != null) {
    		if (sb.charAt(sb.length()-1) != '\n') sb.append(INDENT_STRING); 
    		sb.append("(").append(simplifySourceClassName(logRecord.getSourceClassName())).append(")");
    		sb.append(" [").append(Thread.currentThread().getName()).append("]");
    	}
    	sb.append("\n");
    }
    
    
    protected Date getLogTime(LogRecord logRecord) {
    	return new Date(logRecord.getMillis());    
    }
    
    
    @Override
    public final String format(LogRecord record) {
        StringBuffer sb = new StringBuffer(dateFormat.format(getLogTime(record)));
        sb.append(" - ").append(record.getLevel()).append(" : ");
        
    	formatMessageWithParameters(sb, record);
        if (record.getThrown() != null) {
            sb.append("Caused by ").append(record.getThrown().getClass()).append(" : ");
            sb.append(record.getThrown().toString()).append("\n");

            for (StackTraceElement ste : record.getThrown().getStackTrace()) {
                sb.append("  ").append(ste.toString()).append("\n");
            }
        }

        return indent(sb.toString()) + "\n";
    }

}
