/**
 * EasyCommons - Copyright (c) 2009 PetalsLink, 
 * http://www.petalslink.com/ 
 *  
 * This library is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2.1 of the License, or (at your option) 
 * any later version. This library is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
 * General Public License for more details. 
 *  
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */ 

package com.ebmwebsourcing.easycommons.uri;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;

/**
 * @author Vincent Zurczak - EBM WebSourcing
 */
public class UriManager {

	/**
	 * Builds an URI from an URL (with an handle for URLs not compliant with RFC 2396).
	 * @param url
	 * @return
	 */
	public static URI urlToUri( URL url ) {

		URI uri;
		try {
			// Possible failing step.
			uri = url.toURI();
		} catch( Exception e ) {
			// URL did not comply with RFC 2396 => illegal un-escaped characters.
			try {
				uri = new URI(
					url.getProtocol(), url.getUserInfo(),
					url.getHost(), url.getPort(),
					url.getPath(), url.getQuery(), url.getRef());
			} catch( Exception e1 ) {
				// No automatic repair.
				throw new IllegalArgumentException( "Broken URL: " + url );
			}
		}

		uri = uri.normalize();
		return uri;
	}


	/**
	 * Builds an URI from an URL string (with an handle for URLs not compliant with RFC 2396).
	 * @param urlAsString
	 * @return
	 */
	public static URI urlToUri( String urlAsString ) {

		try {
			URL url = new URL( urlAsString );
			return urlToUri( url );
		} catch( Exception e ) {
			// e.printStackTrace();
		}

		throw new IllegalArgumentException( "Broken URL: " + urlAsString );
	}

	/**
	 * Builds an URL from a string and escape illegal characters.
	 * @param path
	 * @return
	 */
	public static URI filePathToUri( String path ) {

		URI uri;
		try {
			path = path.replaceAll("\\\\", "/");
			uri = new URI( path );
		} catch( URISyntaxException e1 ) {
			uri = new File( path ).toURI();
			uri.normalize();
		}

		return uri;
	}

	/**
	 * Builds an URL from a string and escape illegal characters.
	 * @param path
	 * @return
	 */
	public static URL filePathToUrl( String path ) {

		URL url;
		try {
			url = new URL( path );
		} catch( MalformedURLException e1 ) {
			try {
				URI uri = filePathToUri(path);
				url = uri.toURL();
			} catch( Exception e ) {
				throw new IllegalArgumentException( "Broken URL: " + path );
			}
		}

		return url;
	}


	/**
	 * Builds an URI from an URI and a suffix.
	 * 
	 * <p>
	 * This suffix can be an absolute URL, or a relative path
	 * with respect to the first URI. In this case, the suffix is resolved
	 * with respect to the URI.
	 * </p>
	 * <p>
	 * If the suffix is already an URL, its is returned.<br />
	 * If the suffix is a relative file path and cannot be resolved, an exception is thrown.
	 * </p>
	 * <p>
	 * The returned URI is normalized.
	 * </p>
	 * 
	 * @param referenceUri the reference URI
	 * @param uriSuffix the URI suffix (not null)
	 * @return the new URI
	 * @throws Exception if the resolution failed
	 */
	public static URI buildNewURI( URI referenceUri, String uriSuffix ) throws Exception {

		if( uriSuffix == null )
			throw new NullPointerException( "The URI suffix cannot be null." );

		URI importUri = null;
		try {	// Absolute URL ?
			importUri = urlToUri( new URL( uriSuffix ));

		} catch( Exception e ) {
			try {	// Relative URL ?
				importUri = referenceUri.resolve( new URI( null, uriSuffix, null ));

			} catch( Exception e2 ) {
				String msg =
					"An URI could not be built from the URI " + referenceUri.toString()
					+ " and the suffix " + uriSuffix + ".";
				throw new Exception( msg, e2 );
			}
		}

		return importUri.normalize();
	}


	/**
	 * Determines whether a string is an absolute URI.
	 * @param uriAsString the potential URI, as a string
	 * @return true if it is an absolute URI, false otherwise or if it is not a valid URI
	 */
	public static boolean isAbsoluteUri( String uriAsString ) {

		boolean result;
		try {
			URI uri = urlToUri( uriAsString );
			result = uri.isAbsolute();

		} catch( Exception e ) {
			result = false;
		}

		return result;
	}

}
