/**
 * easy VIPER software - Copyright (c) 2009 PetalsLink, 
 * http://www.petalslink.com/ 
 *  
 * This library is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2.1 of the License, or (at your option) 
 * any later version. This library is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
 * General Public License for more details. 
 *  
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 *  
 * ------------------------------------------------------------------------- 
 * $Id$ 
 * ------------------------------------------------------------------------- 
 */ 
package com.ebmwebsourcing.easycommons.xml;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStream;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Source;
import javax.xml.transform.Templates;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * An util class to prettify XML document
 * 
 * @author ofabre - eBM WebSourcing
 * 
 */
public class XMLPrettyPrinter {


	public final static ThreadLocal<TransformerFactory> transformerFactoryThreadLocal = new ThreadLocal<TransformerFactory>() {

		@Override
		protected TransformerFactory initialValue() {
			return TransformerFactory.newInstance();
		}
	};


	/**
	 * parse the xml String and return it pretty-printed (with correct
	 * indentations, etc..)
	 * 
	 * @param xmlDocument
	 *            the xml document to pretty print. Must be non null
	 * @param encoding
	 *            the encoding to use
	 *   
	 * @returnpretty printed string if no error occurs. If an error occurs,
	 *               return an empty String
	 */
	public static String prettyPrint(final Node xmlDocument,String encoding) {
		String result = "";
		try {
			ByteArrayOutputStream outStream = new ByteArrayOutputStream();
			XMLPrettyPrinter.prettify(xmlDocument, outStream,encoding);
			result = outStream.toString(encoding);
		} catch (final Exception e) {
			System.err.println("write_dom failed:" + e);
			// if an error occurs, the result will be the original string
		}
		return result;

	}

	/**
	 * parse the xml Document and return it pretty-printed (with correct
	 * indentations, etc..).
	 * Use the encoding defined at the parsing or in the document 
	 * (utf8 is used if no encoding is defined)
	 * @param xmlDocument
	 *            the xml document to pretty print. Must be non null
	 * @returnpretty printed string if no error occurs. If an error occurs,
	 *               return an empty String
	 */
	public static String prettyPrint(final Document xmlDocument) {

		return prettyPrint(xmlDocument,getEncoding(xmlDocument));
	}

	/**
	 * parse the xml Document and return it pretty-printed (with correct
	 * indentations, etc..).
	 * Use the encoding defined at the parsing or in the document 
	 * (utf8 is used if no encoding is defined)
	 * @param xmlDocument
	 *            the xml document to pretty print. Must be non null
	 * @returnpretty printed string if no error occurs. If an error occurs,
	 *               return an empty String
	 */
	public static String prettyPrint(final Element node) {
		String res = null;
		if(node != null) {
			if(node instanceof Document) {
				res = prettyPrint(node,getEncoding((Document) node));
			} else {
				res = prettyPrint(node,getEncoding(node.getOwnerDocument()));
			}
		}
		return res;
	}

	/**
	 * parse the xml Document and return it pretty-printed (with correct
	 * indentations, etc..).
	 * Use the encoding defined at the parsing or in the document 
	 * (utf8 is used if no encoding is defined)
	 * @param xmlDocument
	 *            the xml document to pretty print. Must be non null
	 * @returnpretty printed string if no error occurs. If an error occurs,
	 *               return an empty String
	 */
	public static String prettyPrint(final Node node) {
		String res = null;
		if(node != null) {
			if(node instanceof Document) {
				res = prettyPrint(node,getEncoding((Document) node));
			} else {
				res = prettyPrint(node,getEncoding(node.getOwnerDocument()));
			}
		}
		return res;
	}

	/**
	 * Prettify the node into the output stream.
	 *
	 * @param node
	 * @param out
	 * @throws Exception
	 */
	public static void prettify(Node node, OutputStream out,String encoding) throws Exception {
		Source source = new DOMSource(node);
		Source stylesheetSource = getStyleSheetSource();

		TransformerFactory tf = transformerFactoryThreadLocal.get();
		Templates templates = tf.newTemplates(stylesheetSource);
		Transformer transformer = templates.newTransformer();
		transformer.setOutputProperty(OutputKeys.ENCODING,encoding);
		transformer.transform(source, new StreamResult(out));
	}

	/**
	 * Return the encoding of the document.
	 * @param xmlDocument
	 * @return InputEncoding or the XmlEncoding of the document, UTF-8 if not found
	 */
	public static String getEncoding(Document xmlDocument){
		String encoding = xmlDocument.getInputEncoding();
		if(encoding == null){
			encoding = xmlDocument.getXmlEncoding();
		}
		if(encoding == null){
			encoding = "UTF-8";
		}
		return encoding;
	}

	/**
	 * Prettify the node into the output stream.
	 * 
	 * @param node
	 * @param out
	 * @throws Exception
	 */
	public static void prettify(Node node, OutputStream out) throws Exception {
		Source source = new DOMSource(node);
		Source stylesheetSource = getStyleSheetSource();

		TransformerFactory tf = transformerFactoryThreadLocal.get();
		Templates templates = tf.newTemplates(stylesheetSource);
		Transformer transformer = templates.newTransformer();
		transformer.transform(source, new StreamResult(out));
	}

	/**
	 * Prettify the xml input stream into the output stream.
	 * 
	 * @param node
	 * @param out
	 * @throws Exception
	 */
	public static void prettify(InputStream in, OutputStream out) throws Exception {
		StreamSource source = new StreamSource(in);
		Source stylesheetSource = getStyleSheetSource();

		TransformerFactory tf = transformerFactoryThreadLocal.get();
		Templates templates = tf.newTemplates(stylesheetSource);
		Transformer transformer = templates.newTransformer();
		transformer.transform(source, new StreamResult(out));
	}

	private static Source getStyleSheetSource() {
		Source stylesheetSource = new StreamSource(XMLPrettyPrinter.class
				.getResourceAsStream("/prettyPrint.xsl"));
		return stylesheetSource;
	}
}
