package org.petalslink.abslayer;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.ServiceLoader;

import com.ebmwebsourcing.easybox.api.ClassMetadataConstants;
import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;


public final class Factory {

    private static final Factory INSTANCE;
    
    private final Map<Class<?>, Constructor<?>> constructorsFromModelMap; // model class => constructor
    
    
    static {
        ServiceLoader<FactoryProvider> sl = ServiceLoader.load(FactoryProvider.class);
        INSTANCE = new Factory(sl);
    }
    
    Factory(Iterable<FactoryProvider> providers) {
        this.constructorsFromModelMap = new HashMap<Class<?>, Constructor<?>>();
        Iterator<FactoryProvider> providersIt = providers.iterator();
        while (providersIt.hasNext()) {
            FactoryProvider provider = providersIt.next();
            constructorsFromModelMap.putAll(provider.getConstructorsFromModelMap());
        }
    }
    
    public static Factory getInstance()  {
        return INSTANCE;
    }
    
    
    public Object wrap(XmlObject obj) {
        if (obj.hasUserData()) {
            return obj.getUserData();
        }
        Class<?> xmlObjectInterfaceClass = 
            obj.getXmlContext().getClassMetadata().get(obj.getClass(), ClassMetadataConstants.IMPLEMENTATION_CLASS_INTERFACE_CLASS);
        assert xmlObjectInterfaceClass != null;
        Constructor<?> constructor = constructorsFromModelMap.get(xmlObjectInterfaceClass);
        assert constructor != null : String.format("Cannot wrap instances of class '%s'.", xmlObjectInterfaceClass);
        try {
            Object o = constructor.newInstance(obj);
            obj.setUserData(o);
            return o;
        } catch (IllegalArgumentException e) {
            throw new UncheckedException(e);
        } catch (InstantiationException e) {
            throw new UncheckedException(e);
        } catch (IllegalAccessException e) {
            throw new UncheckedException(e);
        } catch (InvocationTargetException e) {
            throw new UncheckedException(e);
        }
    }
    
}
