/**
 * easy VIPER software - Copyright (c) 2009 PetalsLink, 
 * http://www.petalslink.com/ 
 *  
 * This library is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2.1 of the License, or (at your option) 
 * any later version. This library is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
 * General Public License for more details. 
 *  
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 *  
 * ------------------------------------------------------------------------- 
 * $Id$ 
 * ------------------------------------------------------------------------- 
 */
package com.ebmwebsourcing.easyviper.core.impl.engine;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.logging.Logger;

import org.oasisopen.sca.ServiceReference;
import org.oasisopen.sca.annotation.PolicySets;
import org.oasisopen.sca.annotation.Reference;
import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.fraclet.annotations.Lifecycle;
import org.objectweb.fractal.fraclet.types.Step;
import org.objectweb.fractal.julia.ComponentInterface;
import org.objectweb.fractal.julia.Interceptor;
import org.osoa.sca.annotations.Property;
import org.ow2.frascati.tinfi.oasis.ServiceReferenceImpl;

import com.ebmwebsourcing.easycommons.sca.helper.api.SCAComponent;
import com.ebmwebsourcing.easycommons.sca.helper.api.SCAException;
import com.ebmwebsourcing.easycommons.sca.helper.impl.Binding;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAComponentImpl;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAHelper;
import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.Node;
import com.ebmwebsourcing.easyviper.core.api.engine.Process;
import com.ebmwebsourcing.easyviper.core.api.engine.Scope;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.functionnal.ExclusiveBehaviour;
import com.ebmwebsourcing.easyviper.core.impl.engine.thread.ThreadExecution;

/**
 * @author Nicolas Salatge - eBM WebSourcing
 */
@org.oasisopen.sca.annotation.Scope("COMPOSITE")
@org.oasisopen.sca.annotation.Service(value=Execution.class,names="service")
@PolicySets("frascati:scaEasyPrimitive")
public class ExecutionImpl extends SCAComponentImpl implements Execution {

	protected BlockingQueue<ExecutionMessage> executionMessages = 
		new LinkedBlockingQueue<ExecutionMessage>();


	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private Logger log = Logger.getLogger(ExecutionImpl.class.getName());

	/**
	 * variable for debugging
	 */
	@Property(name="stepByStep", required=true)
	private boolean stepByStep = false;

	/**
	 * {@link Execution#STATE_ACTIVE}, {@link Execution#STATE_CANCELLED},
	 * {@link Execution#STATE_ENDED}, {@link Execution#STATE_INACTIVE},
	 * {@link Execution#STATE_SUSPENDED} or any other state defined by a
	 * specific processDefinition language.
	 */
	@Property(name="currentState", required=true)
	private State currentState = State.INACTIVE;

	@Reference(name="currentTarget",required=false)
	private Node currentTarget;

	@Reference(name="parentExec",required=false)
	private Execution parent = null;

	/** are concurrent childExecutions that related to this execution. */
	@Reference(name="childs",required=false)
	private final List<Execution> childExecutions = new ArrayList<Execution>();

	private long speedTime;

	public Node getCurrentTarget() {

		/*
		 * The solution implemented below is linked to a difference between
		 * Fraclet/Julia and Tinfi.
		 * 
		 * With Fraclet/Julia, when injecting a reference in a @Required
		 * annotated field, the reference of the target bound interface is
		 * injected.
		 * 
		 * With Tinfi, the ServiceReference of the source component is injected.
		 * 
		 * Let getCurrentTarget return the target bound interface.
		 */
		ServiceReference<?> sr = (ServiceReference<?>) currentTarget;
		if(sr != null){
			ServiceReferenceImpl<?> cri = (ServiceReferenceImpl<?>) sr;
			Object service = cri._getDelegate();
			ComponentInterface ci = (ComponentInterface) service;
			Interceptor intercep = (Interceptor) ci.getFcItfImpl();
			Node itf = (Node) intercep.getFcItfDelegate();
			return itf;
		}else return null;
	}

	public Scope getParentScope() throws CoreException {
		Scope res = null;
		try {
			final Component scopeComponent = SCAHelper.getSCAHelper()
			.getParent(getComponent());

			res = (Scope) scopeComponent.getFcInterface("service");
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		} catch (final SCAException e) {
			throw new CoreException(e);
		}
		return res;
	}

	public synchronized void run() throws CoreException {
		assert this.currentTarget != null;
		log.finest("execution run begins for "+getName());
		
		init();

		if (this.currentTarget != null) {
			// ancestor execution, sender is null.
			setNextExecutableElements(null, this.currentTarget);

			while((getState() != State.ENDED) && (getState() != State.CANCELLED) 
					&& (getState() != State.SUSPENDED)) {
				step();
				log.finest("End of step for "+getName()+" "+getState());
			}
			log.finest("END OF EXECUTION "+this.getName()+" "+this.getState());
			
		} else {
			throw new CoreException("Error initialization exception");
		}
		
	}

	public void removeChildExecution(final Execution execution)
	throws CoreException {
		if (execution != null) {
			Component exeComp;
			try {
				exeComp = SCAHelper.getSCAHelper()
				.getComponentByInterface(
						this.getParentScope().getComponent(),
						(org.objectweb.fractal.api.Interface) execution.getComponent().getFcInterface("service"),
				"service");
				
				if (exeComp != null) {
					SCAHelper.getSCAHelper().deleteComponent(exeComp);
				}
				
			} catch (NoSuchInterfaceException e) {
				throw new CoreException(e);
			} catch (SCAException e) {
				throw new CoreException(e);
			}

		}
	}

	/**
	 * @throws CoreException
	 * @see Execution#end(String, boolean)
	 */
	public void end() throws CoreException {
		try {
			if (SCAHelper.getSCAHelper().isBinded(getComponent(),
			"currentTarget")) {
				// delete binding between node and execution
				final List<Binding> listOfBindings = new ArrayList<Binding>();
				listOfBindings
				.add(new Binding(
						"currentTarget",
						(org.objectweb.fractal.api.Interface) ((Node) this.currentTarget)
						.getComponent().getFcInterface(
						"service")));
				SCAHelper.getSCAHelper().deleteBindings(getComponent(),
						listOfBindings);
			}
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		} catch (final SCAException e) {
			throw new CoreException(e);
		}
	}

	public List<Execution> getChildExecutions() {
		final List<Execution> res = new ArrayList<Execution>();
		//		res.addAll(this.childExecutions.values());
		res.addAll(this.childExecutions);
		return res;
	}

	public Execution getParentExecution() {
		return this.parent;
	}

	public State getState() {
		return this.currentState;
	}

	public void setStateRecursively(final State state) {
		// TODO : in the case we set recursively SUSPENDED state,
		// there is a chance that a parent execution becomes SUSPENDED
		// whereas some child execution are still ACTIVE.
		// Instead of that, we should mark a parent execution as SUSPENDED
		// if and only if all child executions are ENDED or SUSPENDED.
		this.setState(state);
		try {
			if (!this.getParentScope().getName().equals(
					this.getParentScope().getProcess().getName())) {
				Execution parentScopeExecution = this.getParentScope().getExecution();
				parentScopeExecution.setStateRecursively(state);
			} else if (getParentExecution() != null) {
				getParentExecution().setStateRecursively(state);
			}
		} catch (CoreException e) {
			log.severe("error: " + e.getMessage());
			// do nothing
		}
	}

	public void setState(final State state) {
		this.currentState = state;
	}

	/**
	 * @throws CoreException
	 * @see Execution#createChildExecution(String)
	 */
	public Execution createChildExecution()
	throws CoreException {

		// create child execution
		final Execution childExecution = this.getParentScope().createExecution(
				this.getName() + "_child_" + this.childExecutions.size());
		try {
			final Component exeChildComp = SCAHelper.getSCAHelper()
			.getComponentByInterface(this.getParentScope().getComponent(),
					(org.objectweb.fractal.api.Interface) childExecution,
			"service");

			SCAHelper.getSCAHelper().startComponent(exeChildComp);

			childExecution.setStepByStep(this.stepByStep);
			childExecution.setSpeedTime(this.speedTime);

			this.log.finest("creating " + childExecution.getName());

			// add it to this execution
			this.addExecution(childExecution);
		} catch(SCAException e) {
			throw new CoreException(e);
		}
		return childExecution;
	}

	public void addExecution(final Execution childExecution)
	throws CoreException {
		try {
			// link child to execution
			List<Binding> listOfBindings = new ArrayList<Binding>();
			listOfBindings.add(new Binding("childs_"
					+ String.valueOf(this.childExecutions.size()),
					(org.objectweb.fractal.api.Interface) (childExecution)
					.getComponent().getFcInterface("service")));
			SCAHelper.getSCAHelper().addBindings(this.getComponent(),
					listOfBindings);

			// link execution to child
			listOfBindings = new ArrayList<Binding>();
			listOfBindings.add(new Binding("parentExec",
					(org.objectweb.fractal.api.Interface) this.getComponent()
					.getFcInterface("service")));
			SCAHelper.getSCAHelper().addBindings(
					childExecution.getComponent(), listOfBindings);

		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		} catch (final SCAException e) {
			throw new CoreException(e);
		}
	}

	public void removeChildExecutions() throws CoreException {
		Execution child = null;
		//		while ((this.childExecutions.values() != null)
		//				&& (this.childExecutions.values().size() > 0)) {
		//			child = this.childExecutions.values().iterator().next();
		while ((this.childExecutions != null)
				&& (this.childExecutions.size() > 0)) {
			child = this.childExecutions.iterator().next();
			log.finest("remove execution: " + child.getName());
			this.removeChildExecution(child);
		}
	}

	public void setParentExecution(final Execution parent) {
		this.parent = parent;
	}

	public final boolean hasNextExecutableElement() {
		return !executionMessages.isEmpty();
	}

	public final void setNextExecutableElements(SCAComponent sender,
			Collection<? extends Node> nextExecutableElements) {
		assert nextExecutableElements != null;
		executionMessages.add(new ExecutionMessage(sender, nextExecutableElements));
	}

	public final void setNextExecutableElements(SCAComponent sender,
			Node nextExecutableElement) {
		setNextExecutableElements(sender, Collections
				.singletonList(nextExecutableElement));
	}

	@Override
	public final void notifyParentExecution() throws CoreException {
		getParentExecution().setNextExecutableElements(this,
				getParentExecution().getCurrentTarget());
	}



	public final synchronized void step() throws CoreException {
		// TODO : what do we want to do exactly when we step in a parent
		// execution which is waiting for child execution to end ?
		// for now it is a blocking call : seems right for unit testing in
		// a multithreaded context.
		if (getState().equals(State.ENDED)) return;

		try {
			ExecutionMessage executionMessage = executionMessages.take();
			Collection<? extends Node> nextExecutableElements = 
				executionMessage.getNextExecutableElements();

			SCAComponent sender = executionMessage.getSender();
			if ((sender != null) && (sender instanceof Execution)) {
				removeChildExecution((Execution) sender); 
			}

			if (nextExecutableElements.size() == 1) {
				nextSequentially(nextExecutableElements.iterator().next());
			} else if (nextExecutableElements.size() > 1) {
				try {
					nextConcurrently(nextExecutableElements);
					//FIXME not pretty at all !!!
					if(getCurrentTarget().getBehaviour() instanceof ExclusiveBehaviour) {
						setState(State.SUSPENDED);
					}
				} catch (NoSuchInterfaceException e) {
					throw new CoreException(e);
				}
			}

			if (!hasNextExecutableElement() 
					&& getChildExecutions().isEmpty() 
					&& (getState() != State.SUSPENDED)
					&& (getState() != State.CANCELLED)) {
				
				currentTarget.selectDefaultNextTarget();
				
			}
			

			// check for termination (empty list is the termination token)
			ExecutionMessage nextExecutionMessage = executionMessages.peek();
			
			if (nextExecutionMessage == null) return;
			Collection<? extends Node> after = nextExecutionMessage.getNextExecutableElements();
			
			if (!after.isEmpty()) return;

			// we found termination token. throw it away ...
			executionMessages.clear();

			setState(State.ENDED);
			
			if (getParentExecution() != null) {
				notifyParentExecution();
			} else {
				Process p = null;
				try {
					//					p = (Process) this.getParentScope().getComponent().getFcInterface("process");
					Scope parentScope = this.getParentScope();
					Component parentScopeComp = parentScope.getComponent();
					p = (Process) parentScopeComp.getFcInterface("service");
					p = (Process) this.getParentScope().getComponent().getFcInterface("service");
				} catch (Exception e) {
					// do nothing
				}
				if ((p != null) && (p.getState().equals(Process.State.PROCESS_ACTIVE))) {
					p.end(false);
				}
			}
			
		} catch (InterruptedException ie) {
			throw new CoreException(ie);
		}
	}


	private final void nextSequentially(Node nextExecutableElement)
	throws CoreException {
		//nextExecutableElement.getExecution().setParentExecution(this);
		execute(nextExecutableElement);
	}

	private final void nextConcurrently(

			Collection<? extends Node> nextExecutableElements)
	throws CoreException, NoSuchInterfaceException {
		try {
			for (Node nextExecutableElement : nextExecutableElements) {
				final Execution childExecution = createChildExecution();

				final Component targetComp = SCAHelper
				.getSCAHelper()
				.getComponentByInterface(
						SCAHelper.getSCAHelper().getParent(
								this.getComponent()),
								//							(org.objectweb.fractal.api.Interface) nextExecutableElement,
								(ServiceReference<?>) nextExecutableElement,
				"service");

				// link the execution to the currentTarget
				if (!SCAHelper
						.getSCAHelper()
						.isAlreadyBind(
								childExecution.getComponent(),
								"currentTarget",
								//							(org.objectweb.fractal.api.Interface) nextExecutableElement)) {
								(ServiceReference<?>) nextExecutableElement)) {
					this.getParentScope().linkedExecution2ExecutableElement(
							(Execution) childExecution.getComponent()
							.getFcInterface("service"),
							nextExecutableElement);
				}

				// link the currentTarget to the child execution
				if (!SCAHelper
						.getSCAHelper()
						.isAlreadyBind(
								targetComp,
								"execution",
								(org.objectweb.fractal.api.Interface) (Execution) childExecution
								.getComponent().getFcInterface("service"))) {
					this.getParentScope().linkedExecutableElement2Execution(
							(Node) targetComp
							.getFcInterface("service"),
							(Execution) childExecution.getComponent()
							.getFcInterface("service"));
				}

				childExecution.setStepByStep(isStepByStep());
				
				final ThreadExecution threadChildExecution = new ThreadExecution(childExecution);
				threadChildExecution.start();
			}
		} catch(SCAException e) {
			throw new CoreException(e);
		}
	}


	/**
	 * @throws CoreException
	 * @see Execution#execute(Node)
	 */
	public synchronized void execute(final Node execElmt)
	throws CoreException {
		this.log.finest("Entering in execute method");
		try {
			if (execElmt != null) {
				Node target = null;
				if( execElmt instanceof ServiceReference<?> ) {
					ServiceReference<?> sr = (ServiceReference<?>) execElmt;
					target = (Node)
					SCAHelper.getSCAHelper().getBoundInterface(sr);
				}
				else {
					target = execElmt;
				}

				final Component targetComp = SCAHelper.getSCAHelper()
				.getComponentByInterface(
						SCAHelper.getSCAHelper().getParent(
								this.getComponent()),
								(org.objectweb.fractal.api.Interface) target,
				"service");

				// unbind execution
				SCAHelper.getSCAHelper()
				.deleteLinkWithAnItfClientOfComponent(
						this.getComponent(), "currentTarget");

				// start the target component
				if ((targetComp != null)
						&& (!SCAHelper.getSCAHelper().isStarted(
								targetComp))) {
					SCAHelper.getSCAHelper().startComponent(targetComp);
				}

				this.log.finest("run the next elmt for execution: "
						+ this.getName() + " => " + target.getName());

				// link the execution to the currentTarget
				if (!SCAHelper.getSCAHelper().isAlreadyBind(
						this.getComponent(), "currentTarget",
						(org.objectweb.fractal.api.Interface) target)) {
					this.getParentScope().linkedExecution2ExecutableElement(
							(Execution) this.getComponent().getFcInterface(
							"service"), target);
				}

				// link the currentTarget to the execution
				if (!SCAHelper.getSCAHelper().isAlreadyBind(
						targetComp,
						"execution",
						(org.objectweb.fractal.api.Interface) (Execution) this
						.getComponent().getFcInterface("service"))) {
					this.getParentScope().linkedExecutableElement2Execution(
							(Node) targetComp
							.getFcInterface("service"),
							(Execution) this.getComponent().getFcInterface(
							"service"));
				}
				//				// TODO: Not necessary => Fix fractal bug
				//				try {
				//					if (targetComp.getFcInterface("scope") != null) {
				//						Node scope = (Node) targetComp
				//								.getFcInterface("service");
				//						scope.setExecution(this);
				//					}
				//				} catch (NoSuchInterfaceException e) {
				//					// do nothing
				//				}
				
				execElmt.execute();
			}
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		} catch (final SCAException e) {
			throw new CoreException(e);
		}


		this.log.finest("Exit of execute method");
	}

	public long getSpeedTime() {
		return this.speedTime;
	}

	public void setSpeedTime(final long time) {
		this.speedTime = time;
	}

	public void runSlowly(final long time) throws CoreException {
		this.setSpeedTime(time);
		this.run();
	}



	public void init() throws CoreException {
		// start the current target component
		try {
			final Component targetComp = SCAHelper
			.getSCAHelper()
			.getComponentByInterface(
					SCAHelper.getSCAHelper().getParent(
							this.getComponent()),
							//						(org.objectweb.fractal.api.Interface) currentTarget, "service");
							(ServiceReference<?>) currentTarget, "service");
			if ((targetComp != null)
					&& (!SCAHelper.getSCAHelper().isStarted(targetComp))) {
				SCAHelper.getSCAHelper().startComponent(targetComp);
			}
			this.setState(Execution.State.ACTIVE);
		} catch(SCAException e) {
			throw new CoreException(e);
		}
	}

	public void runStepByStep() throws CoreException {
		this.stepByStep = true;
		init();
		if (this.currentTarget != null) {
			// ancestor execution
			setNextExecutableElements(null, this.currentTarget);

		} else {
			throw new CoreException("Error initialization exception");
		}
	}

	public synchronized void signal() throws CoreException {
		step();

	}

	public boolean isStepByStep() {
		return this.stepByStep;
	}

	public void setStepByStep(final boolean stepByStep) {
		this.stepByStep = stepByStep;
	}

	public void setLog(final Logger logger) {
		this.log = logger;
	}
	
	/**
	 * Destroy the ExecutionImpl behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = Step.DESTROY)
	public void destroySCAComponent() throws SCAException {
		this.log.fine("Fractal execution destroyed: " + this.getName());
		this.setState(Execution.State.CANCELLED);
	}
	
	@Override
	public String toString() {
		String res = super.toString();

		if (this.getName() != null) {
			res = this.getName();
		}

		return res;
	}

}
