/**
* easy VIPER software - Copyright (c) 2009 PetalsLink, 
* http://www.petalslink.com/ 
*  
* This library is free software; you can redistribute it and/or modify it under 
* the terms of the GNU Lesser General Public License as published by the Free 
* Software Foundation; either version 2.1 of the License, or (at your option) 
* any later version. This library is distributed in the hope that it will be 
* useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
* General Public License for more details. 
*  
* You should have received a copy of the GNU Lesser General Public License 
* along with this library; if not, write to the Free Software Foundation, Inc., 
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 *  
 * ------------------------------------------------------------------------- 
 * $Id$ 
 * ------------------------------------------------------------------------- 
 */ 
package com.ebmwebsourcing.easyviper.core.api.engine.behaviour;

import static org.objectweb.fractal.fraclet.types.Step.CREATE;
import static org.objectweb.fractal.fraclet.types.Step.DESTROY;
import static org.objectweb.fractal.fraclet.types.Step.START;
import static org.objectweb.fractal.fraclet.types.Step.STOP;

import java.util.logging.Logger;

import javax.xml.namespace.QName;

import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.fraclet.annotations.Controller;
import org.objectweb.fractal.fraclet.annotations.Interface;
import org.objectweb.fractal.fraclet.annotations.Lifecycle;
import org.objectweb.fractal.fraclet.extensions.Membrane;

import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Node;
import com.ebmwebsourcing.easyviper.core.fractal.FractalHelper;

/**
 * @author Nicolas Salatge - eBM WebSourcing
 */
@org.objectweb.fractal.fraclet.annotations.Component(provides = @Interface(name = "abstract_service", signature = Behaviour.class))
@Membrane(controller = "primitive")
public abstract class AbstractBehaviourImpl implements Behaviour {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private Logger log = Logger.getLogger(AbstractBehaviourImpl.class.getName());

	protected State state = State.ACTIVITY_INACTIVE;

	/**
	 * The component.
	 */
	@Controller
	protected Component component;

	/**
	 * Default constructor
	 * 
	 * @throws CoreException
	 */
	public AbstractBehaviourImpl() {
		super();
	}

	public Component getComponent() {
		return this.component;
	}

	/**
	 * Create the AbstractBehaviourImpl behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = CREATE)
	public void create() throws CoreException {
		this.log.fine("Fractal behaviour created: "
				+ FractalHelper.getFractalHelper().getName(this.component));
	}

	public void init(final Component fractalNode) throws CoreException {
		this.component = fractalNode;
		this.log.fine("Fractal behaviour initiated: "
				+ FractalHelper.getFractalHelper().getName(this.component));
	}

	/**
	 * Start the AbstractBehaviourImpl behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = START)
	public void start() throws CoreException {
		this.log.fine("Fractal behaviour started: " + this.getName());
	}

	/**
	 * Stop the AbstractBehaviourImpl behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = STOP)
	public void stop() throws CoreException {
		this.log.finest("behaviour stopped: " + this.getName());
	}

	/**
	 * Destroy the AbstractBehaviourImpl behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = DESTROY)
	public void destroy() throws CoreException {
		this.log.fine("Fractal behaviour destroyed: " + this.getName());
	}

	public String getName() throws CoreException {
		return FractalHelper.getFractalHelper().getName(this.component);
	}

	public void setName(final String name) throws CoreException {
		if (name != null) {
			FractalHelper.getFractalHelper().changeName(this.component, name);
		}
	}
	
	public QName getQName() throws CoreException {
		QName res = QName.valueOf(this.getName());
		return res;
	}

	public void setQName(QName name) throws CoreException {
		this.setName(name.toString());
	}
	
	public void execute() throws CoreException {
		if (this.state == State.ACTIVITY_ENDED) {
			this.log.finest("restart the " + this.getName() + " activity");
			this.state = State.ACTIVITY_INACTIVE;
		}

		this.log.finest("execute the " + this.getName() + " activity on node ("
				+ this.getNode().getName() + ") by execution ("
				+ this.getNode().getExecution().getName() + ")");
		this.executeActivity();

		if (this.getNode().getNextSelectedExecutableElements().size() == 0) {
			this.log.finest("end of the " + this.getName() + " activity");
			this.state = State.ACTIVITY_ENDED;
		}
	}

	public abstract void executeActivity() throws CoreException;

	public abstract Node getNode();

	@Override
	public String toString() {
		String res = "";
		try {
			res = "behaviour " + this.getName() + " on node "
					+ this.getNode().getName();
		} catch (final CoreException e) {
			res = e.getMessage();
		}
		return res;
	}

	public State getState() {
		return this.state;
	}

	public void setState(final State state) {
		this.state = state;
	}

	public void setLog(final Logger logger) {
		this.log = logger;
	}
}
