/**
* easy VIPER software - Copyright (c) 2009 PetalsLink, 
* http://www.petalslink.com/ 
*  
* This library is free software; you can redistribute it and/or modify it under 
* the terms of the GNU Lesser General Public License as published by the Free 
* Software Foundation; either version 2.1 of the License, or (at your option) 
* any later version. This library is distributed in the hope that it will be 
* useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
* General Public License for more details. 
*  
* You should have received a copy of the GNU Lesser General Public License 
* along with this library; if not, write to the Free Software Foundation, Inc., 
* 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 *  
 * ------------------------------------------------------------------------- 
 * $Id$ 
 * ------------------------------------------------------------------------- 
 */ 
package com.ebmwebsourcing.easyviper.core.impl.engine;

import static org.objectweb.fractal.fraclet.types.Step.CREATE;
import static org.objectweb.fractal.fraclet.types.Step.DESTROY;
import static org.objectweb.fractal.fraclet.types.Step.START;
import static org.objectweb.fractal.fraclet.types.Step.STOP;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.Map.Entry;
import java.util.logging.Logger;

import javax.xml.namespace.QName;

import org.objectweb.fractal.adl.ADLException;
import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.Interface;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.fraclet.annotations.Controller;
import org.objectweb.fractal.fraclet.annotations.Lifecycle;
import org.objectweb.fractal.fraclet.annotations.Requires;
import org.objectweb.fractal.fraclet.extensions.Membrane;
import org.objectweb.fractal.fraclet.types.Cardinality;
import org.objectweb.fractal.fraclet.types.Contingency;

import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Event;
import com.ebmwebsourcing.easyviper.core.api.engine.ExecutableElement;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.Node;
import com.ebmwebsourcing.easyviper.core.api.engine.Scope;
import com.ebmwebsourcing.easyviper.core.api.engine.Transition;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.Behaviour;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.BreakpointBehaviour;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.ReceiverBehaviour;
import com.ebmwebsourcing.easyviper.core.api.engine.fault.Fault;
import com.ebmwebsourcing.easyviper.core.api.engine.handler.TerminationHandler;
import com.ebmwebsourcing.easyviper.core.api.engine.variable.Variable;
import com.ebmwebsourcing.easyviper.core.api.soa.Endpoint;
import com.ebmwebsourcing.easyviper.core.api.soa.correlation.CorrelationGroup;
import com.ebmwebsourcing.easyviper.core.fractal.Binding;
import com.ebmwebsourcing.easyviper.core.fractal.FractalHelper;
import com.ebmwebsourcing.easyviper.core.impl.engine.behaviour.BreakpointBehaviourImpl;

/**
 * @author Nicolas Salatge - eBM WebSourcing
 */
@org.objectweb.fractal.fraclet.annotations.Component(provides = @org.objectweb.fractal.fraclet.annotations.Interface(name = "scope", signature = Scope.class))
@Membrane(controller = "composite")
public class ScopeImpl extends NodeImpl implements Scope {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private static int cpt_childsNode = 0;
	private static int cpt_transitions = 0;
	private static int cpt_breakpoints = 0;

	private Logger log = Logger.getLogger(ScopeImpl.class.getName());

	protected Component initialNode;

	protected Component parentExecution;

	private boolean debug;

	@Requires(name = "childNodes", cardinality = Cardinality.COLLECTION, contingency = Contingency.OPTIONAL)
	protected Map<String, Node> childNodes = Collections.synchronizedMap(new TreeMap<String, Node>());

	@Requires(name = "incomingTransitions", cardinality = Cardinality.COLLECTION, contingency = Contingency.OPTIONAL)
	protected Map<String, Transition> incomingTransitions = Collections.synchronizedMap(new TreeMap<String, Transition>());

	@Requires(name = "outgoingTransitions", cardinality = Cardinality.COLLECTION, contingency = Contingency.OPTIONAL)
	protected Map<String, Transition> outgoingTransitions = Collections.synchronizedMap(new TreeMap<String, Transition>());

	@Requires(name = "behaviour", contingency = Contingency.OPTIONAL)
	protected Behaviour behaviour;

	@Requires(name = "parentNode", contingency = Contingency.OPTIONAL)
	protected Node parentNode;

	@Requires(name = "execution", contingency = Contingency.OPTIONAL)
	protected Execution execution;


	protected Map<Fault, Scope> exceptions = new HashMap<Fault, Scope>();

	protected Map<Object, Endpoint> endpoints = new HashMap<Object, Endpoint>();

	protected List<CorrelationGroup> correlationGroups = new ArrayList<CorrelationGroup>(); 

	protected Exception faultScope = null;

	protected TerminationHandler th = null;

	/**
	 * The Fractal scope composite component-
	 */
	@Controller
	private Component component;

	private final Map<QName, Variable> variables = new HashMap<QName, Variable>();

	/**
	 * Default Constructor
	 * 
	 * @throws ADLException
	 */
	public ScopeImpl() throws CoreException {
		super();
	}

	/**
	 * Start the scope behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = CREATE)
	public void create() throws CoreException {
		this.log.fine("Fractal scope created: "
				+ FractalHelper.getFractalHelper().getName(this.component));
	}

	public void init(final Component fractalNode) throws CoreException {
		this.component = fractalNode;
		this.log.fine("Fractal scope initiated: "
				+ FractalHelper.getFractalHelper().getName(this.component));
	}

	/**
	 * Start the scope behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = START)
	public void start() throws CoreException {
		this.log.fine("Fractal scope started: "
				+ FractalHelper.getFractalHelper().getName(this.component));
	}

	/**
	 * Stop the scope behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = STOP)
	public void stop() throws CoreException {
		this.log.fine("Fractal scope stopped: "
				+ FractalHelper.getFractalHelper().getName(this.component));
	}

	/**
	 * Start the scope behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = DESTROY)
	public void destroy() throws CoreException {
		this.log.fine("Fractal scope destroyed: "
				+ FractalHelper.getFractalHelper().getName(this.component));
	}

	public void startFractalComponent() throws CoreException {
		FractalHelper.getFractalHelper().startComponent(this.component);
		this.log.fine("Start the scope: " + this.getName());
	}

	public Node createNode(String nodeName, final Behaviour behaviourContext)
	throws CoreException {
		this.log.finest("start create node");
		Node node = null;
		try {

			if ((nodeName == null)
					&& (behaviourContext instanceof BreakpointBehaviour)) {
				nodeName = "breakpoint_"
					+ this.getAllBreakPoints(this.getComponent()).size();
			}

			// create node
			final Component fractalNode = FractalHelper.getFractalHelper()
			.createNewComponent(NodeImpl.class.getName(), null);
			FractalHelper.getFractalHelper().addComponent(fractalNode,
					this.component, null);
			node = (Node) fractalNode.getFcInterface("/content");
			((Node) node).init(fractalNode);
			((Node) node).setName(nodeName);
			if (!this.log.getName().equals(ScopeImpl.class.getName())) {
				node.setLog(this.log);
			}

			node = (Node) fractalNode.getFcInterface("service");

			// create behaviour
			final Component fractalActivity = FractalHelper.getFractalHelper()
			.createNewComponent(behaviourContext.getClass().getName(),
					null);
			FractalHelper.getFractalHelper().changeName(
					fractalActivity,
					behaviourContext.getClass().getSimpleName()
					+ "4"
					+ FractalHelper.getFractalHelper().getName(
							fractalNode));
			FractalHelper.getFractalHelper().addComponent(fractalActivity,
					this.component, null);
			Behaviour behaviour = (Behaviour) fractalActivity
			.getFcInterface("/content");
			((Behaviour) behaviour).init(fractalActivity);
			((Behaviour) behaviour).setInitializationContext(behaviourContext
					.getInitializationContext());
			if (!this.log.getName().equals(ScopeImpl.class.getName())) {
				behaviour.setLog(this.log);
			}
			behaviour = (Behaviour) fractalActivity.getFcInterface("service");

			this.linkedNodeAndActivity(node, behaviour);

			this.log.fine("Creation of the fractal node: " + nodeName);

		} catch (final NoSuchInterfaceException e) {
			this.log.severe(e.getMessage());
			throw new CoreException(e);
		}
		this.log.finest("end of create node");
		return node;
	}

	public Node getInitialNode() throws CoreException {
		Node res = null;
		try {
			if(this.initialNode != null) {
				res = (Node) this.initialNode.getFcInterface("service");
			}
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
		return res;
	}

	public Node getNode(final String name) throws CoreException {
		Node res = null;
		try {
			final List<Component> comps = FractalHelper.getFractalHelper()
			.getComponentsByName(this.component, name);

			final List<Node> nodes = new ArrayList<Node>();

			if (comps != null) {
				for (final Component comp : comps) {
					if (comp.getFcInterface("service") instanceof Node) {
						nodes.add((Node) comp.getFcInterface("service"));
					}
				}
			}

			if (nodes.size() > 1) {
				throw new CoreException(
				"Several node components are the same name");
			}

			if (nodes.size() > 0) {
				res = nodes.get(0);
			}

		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
		return res;
	}

	public void deleteExecution(final Execution exec) throws CoreException {
		// delete binding between scope and execution
		final List<Binding> listOfBindings = new ArrayList<Binding>();
		try {
			listOfBindings.add(new Binding("service",
					(org.objectweb.fractal.api.Interface) exec.getComponent()
					.getFcInterface("service")));
			FractalHelper.getFractalHelper().deleteBindings(
					this.getComponent(), listOfBindings);
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException("impossible to delete binding", e);
		}

		FractalHelper.getFractalHelper().deleteComponent(exec.getComponent());
	}

	public void stopAllExecutions() throws CoreException {
		for (final Execution execution : this.getParentExecution()
				.getChildExecutions()) {
			((Execution) execution).stop();
		}
		this.getParentExecution().stop();
	}

	public void end(boolean clean) throws CoreException {

		// apply termination handler
		if(this.th != null) {
			th.end(this);
		}

		if(clean) {
			this.log.finest("clean all scopes");
			this.cleanScopes(this);

			this.log.finest("stop all sub components");
			FractalHelper.getFractalHelper().stopAllSubComponents(this.component);
		}
	}

	protected void cleanScopes(Scope scope) throws CoreException {
		for(Scope faultScope: scope.getExceptions().values()) {
			cleanScopes(faultScope);
		}

		//scope.setExceptions(new HashMap<Fault, Scope>());
		scope.setFaultScope(null);

		// clean variables
		cleanVariables(scope);

		// end all behaviours
		cleanBehaviours(scope);


		// remove all executions
		this.log.finest("remove all executions");
		if (scope.getParentExecution() != null) {
			scope.getParentExecution().removeChildExecutions();
			final Component exeComp = FractalHelper.getFractalHelper()
			.getComponentByInterface(
					scope.getComponent(),
					(org.objectweb.fractal.api.Interface) scope
					.getParentExecution(), "service");
			if (exeComp != null) {
				FractalHelper.getFractalHelper().deleteComponent(exeComp);
			}
		}
		scope.setParentExecution(null);


		// find under scopes
		try {
			List<Component> components = FractalHelper.getFractalHelper().getComponents(scope.getComponent());
			for(Component component: components) {
				if(FractalHelper.getFractalHelper().isStarted(component)) {
					try {
						Scope underScope = (Scope) component.getFcInterface("scope");
						cleanScopes(underScope);
					} catch (NoSuchInterfaceException e) {
						// do nothing
					}
				}
			}
		} catch (CoreException e) {
			throw new CoreException(e);
		}
	}

	private void cleanBehaviours(Scope scope) throws CoreException {
		// find behaviour
		try {
			List<Component> components = FractalHelper.getFractalHelper().getComponents(scope.getComponent());
			for(Component component: components) {
				if(FractalHelper.getFractalHelper().isStarted(component)) {
					try {
						Behaviour behaviour = (Behaviour) component.getFcInterface("service");
						behaviour.setState(Behaviour.State.ACTIVITY_ENDED);

						if(behaviour instanceof ReceiverBehaviour) {
							((ReceiverBehaviour)behaviour).setMessage(null);
						}

					} catch (NoSuchInterfaceException e) {
						// do nothing
					} catch (ClassCastException e) {
						// do nothing
					}
				}
			}
		} catch (CoreException e) {
			throw new CoreException(e);
		}
	}

	private void cleanVariables(Scope scope) throws CoreException {
		this.log.finest("Clean all variables of scope: " + scope.getName());
		for(Variable var: scope.getVariables().values()) {
			var.setValue(null);
		}
	}

	private List<Component> getAllBreakPoints(final Component parent)
	throws CoreException {
		final List<Component> behaviours = new ArrayList<Component>();
		final List<Component> comps = FractalHelper.getFractalHelper().getComponents(
				parent);
		try {
			for (final Component component : comps) {
				if (component.getFcInterface("service") instanceof BreakpointBehaviour) {
					behaviours.add(component);
				} else if (component.getFcInterface("service") instanceof Scope) {
					behaviours.addAll(this.getAllBreakPoints(component));
				}
			}
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
		return behaviours;
	}

	public boolean removeNode(final Node node) throws CoreException {
		final boolean res = true;
		try {
			final Component nodeComp = FractalHelper.getFractalHelper()
			.getComponentByInterface(this.getComponent(),
					(org.objectweb.fractal.api.Interface) node,
			"service");

			final String name = FractalHelper.getFractalHelper().getName(nodeComp);

			// delete all child nodes
			Node child = null;
			final String clientItfName = "childNodes_";

			Map<String, org.objectweb.fractal.api.Interface> map = FractalHelper
			.getFractalHelper()
			.getServerInterfacesLinkedToClientInterfacesOfComponent(
					nodeComp);

			Iterator<Entry<String, org.objectweb.fractal.api.Interface>> entries = map
			.entrySet().iterator();
			Entry<String, org.objectweb.fractal.api.Interface> entry = null;
			while ((entries != null) && (entries.hasNext())) {
				entry = entries.next();
				if (entry.getKey().startsWith(clientItfName)) {
					child = (Node) entry.getValue();
					this.removeNode(child);
					// FractalHelper.getFractalHelper().getComponentByInterface(this.getComponent(),
					// (org.objectweb.fractal.api.Interface) child, "service");

					map = FractalHelper
					.getFractalHelper()
					.getServerInterfacesLinkedToClientInterfacesOfComponent(
							nodeComp);
					entries = map.entrySet().iterator();
				}
			}

			// delete all outgoing transitions
			List<Component> clientsOfComp = FractalHelper.getFractalHelper()
			.getClientComponentsLinkedToServerInterfacesOfComponent(
					this.getComponent(),
					(org.objectweb.fractal.api.Interface) nodeComp
					.getFcInterface("service"));
			Iterator<Component> components = clientsOfComp.iterator();
			Component outTransition = null;
			while ((components != null) && (components.hasNext())) {
				outTransition = components.next();
				if (outTransition.getFcInterface("service") instanceof Transition) {
					FractalHelper.getFractalHelper().deleteComponent(
							outTransition);
					clientsOfComp = FractalHelper
					.getFractalHelper()
					.getClientComponentsLinkedToServerInterfacesOfComponent(
							this.getComponent(),
							(org.objectweb.fractal.api.Interface) nodeComp
							.getFcInterface("node"));
					components = clientsOfComp.iterator();
				}
			}

			// delete the component
			FractalHelper.getFractalHelper().deleteComponent(nodeComp);
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
		return res;
	}

	public void setInitialNode(final Node initial) throws CoreException {
		this.initialNode = FractalHelper.getFractalHelper()
		.getComponentByInterface(this.getComponent(),
				(Interface) initial, "service");
	}

	public String getName() throws CoreException {
		return FractalHelper.getFractalHelper().getName(this.component);
	}

	public void setName(final String name) throws CoreException {
		FractalHelper.getFractalHelper().changeName(this.component, name);
	}

	public Component getComponent() {
		return this.component;
	}

	public Execution getParentExecution() throws CoreException {
		Execution res = null;
		try {
			if (this.parentExecution != null) {
				res = (Execution) this.parentExecution
				.getFcInterface("service");
			}
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
		return res;
	}

	public List<Transition> getAllTransitions() throws CoreException {
		final List<Transition> transitions = new ArrayList<Transition>();
		final List<Component> comps = FractalHelper.getFractalHelper().getComponents(
				this.component);

		for (final Component component : comps) {
			try {
				if (component.getFcInterface("service") instanceof Transition) {
					transitions.add((Transition) component
							.getFcInterface("service"));
				}
			} catch (final NoSuchInterfaceException e) {
				// do nothing
			}
		}

		return transitions;
	}

	public List<Node> getAllNodes() throws CoreException {
		final List<Node> nodes = new ArrayList<Node>();
		final List<Component> comps = FractalHelper.getFractalHelper().getComponents(
				this.component);
		for (final Component component : comps) {
			try {
				if (component.getFcInterface("service") instanceof Node) {
					nodes.add((Node) component.getFcInterface("service"));
				}
			} catch (final NoSuchInterfaceException e) {
				// do nothing
			}
		}
		return nodes;
	}

	public List<Scope> getAllScopes() throws CoreException {
		final List<Scope> scope = new ArrayList<Scope>();
		final List<Component> comps = FractalHelper.getFractalHelper().getComponents(
				this.component);
		for (final Component component : comps) {
			try {
				if (component.getFcInterface("scope") instanceof Scope) {
					scope.add((Scope) component.getFcInterface("scope"));
				}
			} catch (final NoSuchInterfaceException e) {
				// do nothing
			}
		}

		return scope;
	}

	public Transition getTransitionByName(final String name) throws CoreException {
		Transition res = null;
		try {
			final List<Component> comps = FractalHelper.getFractalHelper()
			.getComponentsByName(this.component, name);

			final List<Transition> interfaces = new ArrayList<Transition>();

			if (comps != null) {
				for (final Component comp : comps) {
					if (comp.getFcInterface("service") instanceof Transition) {
						interfaces.add((Transition) comp
								.getFcInterface("service"));
					}
				}
			}

			if (interfaces.size() > 1) {
				throw new CoreException(
				"Several interface components are the same name");
			}

			if (interfaces.size() > 0) {
				res = interfaces.get(0);
			}

		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
		return res;

	}

	public Node getNodeByName(final String name) throws CoreException {
		Node res = null;
		try {
			final List<Component> comps = FractalHelper.getFractalHelper()
			.getComponentsByName(this.component, name);

			final List<Node> nodes = new ArrayList<Node>();

			if (comps != null) {
				for (final Component comp : comps) {
					if (comp.getFcInterface("service") instanceof Node) {
						nodes.add((Node) comp.getFcInterface("service"));
					}
				}
			}

			if (nodes.size() > 1) {
				throw new CoreException(
				"Several node components are the same name");
			}

			if (nodes.size() > 0) {
				res = nodes.get(0);
			}

		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
		return res;
	}

	public void linkedTransition2Node(final Transition transition, final Node sourceNode)
	throws CoreException {
		try {

			final Component transitionComp = FractalHelper.getFractalHelper()
			.getComponentByInterface(this.getComponent(),
					(org.objectweb.fractal.api.Interface) transition,
			"service");
			final Component sourceComp = FractalHelper.getFractalHelper()
			.getComponentByInterface(this.getComponent(),
					(org.objectweb.fractal.api.Interface) sourceNode,
			"service");

			if ((transitionComp == null) || (sourceComp == null)) {
				throw new CoreException(
				"Impossible to link transition to a node");
			}

			// Add binding between transition and source node
			List<Binding> listOfBindings = new ArrayList<Binding>();
			listOfBindings = new ArrayList<Binding>();
			listOfBindings.add(new Binding("outgoingTransitions_"
					+ this.formatCounter(ScopeImpl.cpt_transitions),
					(org.objectweb.fractal.api.Interface) transitionComp
					.getFcInterface("service")));
			FractalHelper.getFractalHelper().addBindings(sourceComp,
					listOfBindings);
			// TODO: Not necessary => Fix fractal bug 
			try {
				if(sourceComp.getFcInterface("scope") != null) {
					Node scope = (Node) sourceComp.getFcInterface("service");
					scope.getMapOutgoingTransitions().put("outgoingTransitions_"
							+ this.formatCounter(ScopeImpl.cpt_transitions), (Transition) transitionComp
							.getFcInterface("service"));
				}
			} catch(NoSuchInterfaceException e) {
				// do nothing
			}

			// Add binding between sourceNode and transition
			listOfBindings = new ArrayList<Binding>();
			listOfBindings.add(new Binding("source",
					(org.objectweb.fractal.api.Interface) sourceComp
					.getFcInterface("service")));
			FractalHelper.getFractalHelper().addBindings(transitionComp,
					listOfBindings);

		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
	}

	public void linkedNode2Transition(final Node targetNode, final Transition transition)
	throws CoreException {
		try {

			final Component targetComp = FractalHelper.getFractalHelper()
			.getComponentByInterface(this.getComponent(),
					(org.objectweb.fractal.api.Interface) targetNode,
			"service");
			final Component transitionComp = FractalHelper.getFractalHelper()
			.getComponentByInterface(this.getComponent(),
					(org.objectweb.fractal.api.Interface) transition,
			"service");

			if ((targetComp == null) || (transitionComp == null)) {
				throw new CoreException(
				"Impossible to link node to a transition");
			}

			// Add binding between transition and target node
			List<Binding> listOfBindings = new ArrayList<Binding>();
			listOfBindings = new ArrayList<Binding>();
			listOfBindings.add(new Binding("incomingTransitions_"
					+ this.formatCounter(ScopeImpl.cpt_transitions),
					(org.objectweb.fractal.api.Interface) transitionComp
					.getFcInterface("service")));
			FractalHelper.getFractalHelper().addBindings(targetComp,
					listOfBindings);
			// TODO: Not necessary => Fix fractal bug 
			try {
				if(targetComp.getFcInterface("scope") != null) {
					Node scope = (Node) targetComp.getFcInterface("service");
					scope.getMapIncomingTransitions().put("incomingTransitions_"
							+ this.formatCounter(ScopeImpl.cpt_transitions), (Transition) transitionComp
							.getFcInterface("service"));
				}
			} catch(NoSuchInterfaceException e) {
				// do nothing
			}

			// Add binding between transition and target node
			listOfBindings = new ArrayList<Binding>();
			listOfBindings = new ArrayList<Binding>();
			listOfBindings.add(new Binding("destination",
					(org.objectweb.fractal.api.Interface) targetComp
					.getFcInterface("service")));
			FractalHelper.getFractalHelper().addBindings(transitionComp,
					listOfBindings);

		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
	}

	public void linkedNodeAndActivity(final Node targetNode, final Behaviour behaviour)
	throws CoreException {
		try {
			if ((behaviour != null) && (targetNode != null)) {
				final Component targetComp = FractalHelper
				.getFractalHelper()
				.getComponentByInterface(
						this.getComponent(),
						(org.objectweb.fractal.api.Interface) targetNode,
				"service");

				final Component activityComp = FractalHelper
				.getFractalHelper()
				.getComponentByInterface(
						this.getComponent(),
						(org.objectweb.fractal.api.Interface) behaviour,
				"service");

				if ((targetComp == null) || (activityComp == null)) {
					throw new CoreException(
					"Impossible to link node to a transition");
				}

				// Add binding between target node and behaviour
				List<Binding> listOfBindings = new ArrayList<Binding>();
				listOfBindings.add(new Binding("behaviour",
						(org.objectweb.fractal.api.Interface) activityComp
						.getFcInterface("service")));
				FractalHelper.getFractalHelper().addBindings(targetComp,
						listOfBindings);

				// Add binding between behaviour and target node
				listOfBindings = new ArrayList<Binding>();
				listOfBindings.add(new Binding("node",
						(org.objectweb.fractal.api.Interface) targetComp
						.getFcInterface("service")));
				FractalHelper.getFractalHelper().addBindings(activityComp,
						listOfBindings);
			}
		} catch (final NoSuchInterfaceException e) {
			this.log.finest(e.getMessage());
			throw new CoreException(e);
		}
	}

	public void unlinkedNodeAndActivity(final Node targetNode, final Behaviour behaviour)
	throws CoreException {
		try {
			if ((behaviour != null) && (targetNode != null)) {
				final Component targetComp = FractalHelper
				.getFractalHelper()
				.getComponentByInterface(
						this.getComponent(),
						(org.objectweb.fractal.api.Interface) targetNode,
				"service");

				final Component activityComp = FractalHelper
				.getFractalHelper()
				.getComponentByInterface(
						this.getComponent(),
						(org.objectweb.fractal.api.Interface) behaviour,
				"service");

				if ((targetComp == null) || (activityComp == null)) {
					throw new CoreException(
					"Impossible to link node to a transition");
				}

				// Add binding between target node and behaviour
				List<Binding> listOfBindings = new ArrayList<Binding>();
				listOfBindings.add(new Binding("behaviour",
						(org.objectweb.fractal.api.Interface) activityComp
						.getFcInterface("service")));
				FractalHelper.getFractalHelper().deleteBindings(targetComp,
						listOfBindings);

				// delete binding between behaviour and target node
				listOfBindings = new ArrayList<Binding>();
				listOfBindings.add(new Binding("node",
						(org.objectweb.fractal.api.Interface) targetComp
						.getFcInterface("service")));
				FractalHelper.getFractalHelper().deleteBindings(activityComp,
						listOfBindings);
			}
		} catch (final NoSuchInterfaceException e) {
			this.log.finest(e.getMessage());
			throw new CoreException(e);
		}
	}


	public void unlinkedNodeAndExecution(final Node targetNode, final Execution execution)
	throws CoreException {
		try {
			if ((execution != null) && (targetNode != null)) {


				Component targetComp = null;

				if(targetNode instanceof Scope) {
					targetComp = FractalHelper
					.getFractalHelper()
					.getComponentByInterface(
							this.getComponent(),
							(org.objectweb.fractal.api.Interface) targetNode,
					"scope");
				} else if(targetNode instanceof Node) {
					targetComp = FractalHelper
					.getFractalHelper()
					.getComponentByInterface(
							this.getComponent(),
							(org.objectweb.fractal.api.Interface) targetNode,
					"service");
				} 


				final Component execComp = FractalHelper
				.getFractalHelper()
				.getComponentByInterface(
						this.getComponent(),
						(org.objectweb.fractal.api.Interface) execution,
				"service");

				if ((targetComp == null) || (execComp == null)) {
					throw new CoreException(
					"Impossible to unlink node to the execution ");
				}

				// delete binding between target node and execution
				List<Binding> listOfBindings = new ArrayList<Binding>();
				listOfBindings.add(new Binding("execution",
						(org.objectweb.fractal.api.Interface) execComp
						.getFcInterface("service")));
				FractalHelper.getFractalHelper().deleteBindings(targetComp,
						listOfBindings);

				// delete binding between execution and target node
				listOfBindings = new ArrayList<Binding>();
				listOfBindings.add(new Binding("currentTarget",
						(org.objectweb.fractal.api.Interface) targetComp
						.getFcInterface("service")));
				FractalHelper.getFractalHelper().deleteBindings(execComp,
						listOfBindings);
			}
		} catch (final NoSuchInterfaceException e) {
			this.log.finest(e.getMessage());
			throw new CoreException(e);
		}
	}

	public void linked(final Node sourceNode, final Transition transition, final Node targetNode)
	throws CoreException {
		this.linkedTransition2Node(transition, sourceNode);
		this.linkedNode2Transition(targetNode, transition);
		ScopeImpl.cpt_transitions++;
	}

	public void linked(final Node parentNode, final Node childrenNode)
	throws CoreException {
		try {
			final Component parentComp = FractalHelper.getFractalHelper()
			.getComponentByInterface(this.getComponent(),
					(org.objectweb.fractal.api.Interface) parentNode,
			"service");
			final Component childrenComp = FractalHelper.getFractalHelper()
			.getComponentByInterface(this.getComponent(),
					(org.objectweb.fractal.api.Interface) childrenNode,
			"service");

			if ((parentComp == null) || (childrenComp == null)) {
				throw new CoreException(
				"Impossible to link parent and children node");
			}

			// Add binding between parent and child
			List<Binding> listOfBindings = new ArrayList<Binding>();
			listOfBindings.add(new Binding("childNodes_"
					+ this.formatCounter(ScopeImpl.cpt_childsNode),
					(org.objectweb.fractal.api.Interface) childrenComp
					.getFcInterface("service")));
			FractalHelper.getFractalHelper().addBindings(parentComp,
					listOfBindings);
			ScopeImpl.cpt_childsNode++;

			// Add binding between child and parent
			listOfBindings = new ArrayList<Binding>();
			listOfBindings.add(new Binding("parentNode",
					(org.objectweb.fractal.api.Interface) parentComp
					.getFcInterface("service")));
			FractalHelper.getFractalHelper().addBindings(childrenComp,
					listOfBindings);


			// TODO: Fix bugs
			NodeImpl nodeContent = (NodeImpl)childrenComp.getFcInterface("/content");
			if(nodeContent.getParentNode() == null) {
				nodeContent.setParentNode(parentNode);
			}
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
	}

	public void linkedExecution2ExecutableElement(final Execution exe,
			final ExecutableElement target) throws CoreException {
		try {
			Component targetComp = FractalHelper.getFractalHelper()
			.getComponentByInterface(this.getComponent(),
					(org.objectweb.fractal.api.Interface) target,
			"service");
			Component exeComp = FractalHelper.getFractalHelper()
			.getComponentByInterface(this.getComponent(),
					(org.objectweb.fractal.api.Interface) exe,
			"service");


			if ((exeComp == null) || (targetComp == null)) {
				throw new CoreException(
						"Impossible to link the execution and the target node: exeComp = " + exeComp + " or targetComp = " + targetComp);
			} 

			// Add binding between execution and target node
			List<Binding> listOfBindings = new ArrayList<Binding>();
			listOfBindings = new ArrayList<Binding>();
			listOfBindings.add(new Binding("currentTarget",
					(org.objectweb.fractal.api.Interface) targetComp
					.getFcInterface("service")));

			if (FractalHelper.getFractalHelper().isBinded(exeComp,
			"currentTarget")) {
				throw new CoreException("Execution already bind");
			}
			FractalHelper.getFractalHelper().addBindings(exeComp,
					listOfBindings);
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
	}

	public void linkedExecutableElement2Execution(final ExecutableElement source,
			final Execution exe) throws CoreException {
		try {
			final Component sourceComp = FractalHelper.getFractalHelper()
			.getComponentByInterface(this.getComponent(),
					(org.objectweb.fractal.api.Interface) source,
			"service");

			final Component exeComp = FractalHelper.getFractalHelper()
			.getComponentByInterface(this.getComponent(),
					(org.objectweb.fractal.api.Interface) exe,
			"service");

			if ((exeComp == null) || (sourceComp == null)) {
				throw new CoreException(
				"Impossible to link the axecution and the target node");
			}

			// Add binding between source element and execution
			if (!FractalHelper.getFractalHelper().isBinded(sourceComp,
			"execution")) {
				List<Binding> listOfBindings = new ArrayList<Binding>();
				listOfBindings = new ArrayList<Binding>();
				listOfBindings.add(new Binding("execution",
						(org.objectweb.fractal.api.Interface) exeComp
						.getFcInterface("service")));

				FractalHelper.getFractalHelper().addBindings(sourceComp,
						listOfBindings);
			}
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
	}

	public Transition createTransition(final String transitionName, final Event event)
	throws CoreException {
		Transition transition = null;
		Component fractalTransition = null;
		try {
			// create a new transition
			fractalTransition = FractalHelper.getFractalHelper()
			.createNewComponent(TransitionImpl.class.getName(), null);
			FractalHelper.getFractalHelper().addComponent(fractalTransition,
					this.getComponent(), null);
			transition = (Transition) fractalTransition
			.getFcInterface("/content");

			((TransitionImpl) transition).init(fractalTransition);
			((TransitionImpl) transition).setName(transitionName);
			if (!this.log.getName().equals(ScopeImpl.class.getName())) {
				transition.setLog(this.log);
			}
			transition = (Transition) fractalTransition
			.getFcInterface("service");
			this.log.fine("Creation of the fractal transition: " + transitionName);

		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
		return transition;
	}

	/**
	 * constructor to feed in subclasses of processes.
	 * 
	 * @throws CoreException
	 */
	public synchronized Execution createExecution(final String execName)
	throws CoreException {
		this.log.finest("start the creation of execution: " + execName);
		Execution execution = null;
		Component fractalExecution = null;

		try {

			fractalExecution = FractalHelper.getFractalHelper()
			.createNewComponent(ExecutionImpl.class.getName(), null);

			FractalHelper.getFractalHelper().addComponent(fractalExecution,
					this.getComponent(), null);
			FractalHelper.getFractalHelper().changeName(fractalExecution,
					execName);
			execution = (Execution) fractalExecution.getFcInterface("/content");
			if (!this.log.getName().equals(ScopeImpl.class.getName())) {
				execution.setLog(this.log);
			}

			execution = (Execution) fractalExecution.getFcInterface("service");


			this.log.finest("execution created: " + execName);

			if (this.getParentExecution() == null) {
				this.log.finest(execName + " is a parent execution");
				this.parentExecution = fractalExecution;
			}

		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}

		return execution;
	}

	public void addBreakpoint(final Node node) throws CoreException {
		final Component nodeComp = FractalHelper.getFractalHelper()
		.getComponentByInterface(this.component, (Interface) node,
		"service");

		// create Breakpoint
		final Node breakpointNode = this.createNode("breakPoint_" + this.formatCounter(ScopeImpl.cpt_breakpoints),
				new BreakpointBehaviourImpl());
		final Transition breakpointTransition = this.createTransition(
				"breakTransition_" + this.formatCounter(ScopeImpl.cpt_breakpoints), null);
		this.linkedNode2Transition(breakpointNode, breakpointTransition);

		// TODO: unlink node with all previous previous

		ScopeImpl.cpt_breakpoints++;
	}

	public boolean isDebug() {
		return this.debug;
	}

	public void setDebug(final boolean debug) {
		this.debug = debug;
	}

	public Map<QName, Variable> getVariables() {
		return this.variables;
	}

	public Variable findVariable(final QName name) throws CoreException {
		Variable v = null;
		v = this.variables.get(name);

		if (v == null) {
			Scope current = this.getParentScope(this);
			while ((v == null) && (current != null)) {
				v = current.getVariables().get(name);
				current = this.getParentScope(current);
			}			
		}

		return v;
	}

	public List<CorrelationGroup> findCorrelationGroups(final String name) throws CoreException {
		List<CorrelationGroup> res = new ArrayList<CorrelationGroup>();

		for(CorrelationGroup correlationGroup: correlationGroups) {
			if(correlationGroup.getCorrelations().get(name) != null) {
				res.add(correlationGroup);
			}
		}

		Scope current = this.getParentScope(this);
		if(current != null) {
			res.addAll(current.findCorrelationGroups(name));
			current = this.getParentScope(current);
		}			

		return res;
	}

	public Endpoint findEndpoint(final Object key) throws CoreException {
		Endpoint ep = null;
		ep = this.endpoints.get(key);

		if (ep == null) {
			Scope current = this.getParentScope(this);
			while ((ep == null) && (current != null)) {
				ep = current.getEndpoints().get(key);
				current = this.getParentScope(current);
			}			
		}

		return ep;
	}

	public void setVariable(Variable v) throws CoreException {
		Scope current = this;
		v = current.getVariables().get(v.getQName());
		if (v == null) {
			while ((v == null) && (current != null)) {
				v = current.getVariables().get(v.getQName());
				current = this.getParentScope();
			}
		}

		if (current != null) {
			current.getVariables().put(v.getQName(), v);
		} else {
			this.getVariables().put(v.getQName(), v);
		}
	}

	public Scope getParentScope() throws CoreException {
		Scope parent = null;
		try {
			if (!(this instanceof com.ebmwebsourcing.easyviper.core.api.engine.Process)) {
				final Component parentComp = FractalHelper.getFractalHelper()
				.getParent(this.getComponent());
				if (parentComp != null) {
					try {
						parent = (com.ebmwebsourcing.easyviper.core.api.engine.Process) parentComp.getFcInterface("service");
					} catch (final NoSuchInterfaceException e) {
						parent = (Scope) parentComp.getFcInterface("scope");
					} catch (final ClassCastException e) {
						parent = (Scope) parentComp.getFcInterface("scope");
					}
				}
			}
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
		return parent;
	}

	private Scope getParentScope(Scope child) throws CoreException {
		Scope parent = null;
		try {
			if (!(child instanceof com.ebmwebsourcing.easyviper.core.api.engine.Process)) {
				final Component parentComp = FractalHelper.getFractalHelper()
				.getParent(child.getComponent());
				if (parentComp != null) {
					parent = (Scope) parentComp.getFcInterface("scope");
				}
			}
		} catch (final NoSuchInterfaceException e) {
			// do nothing
		}
		return parent;
	}

	public com.ebmwebsourcing.easyviper.core.api.engine.Process getProcess()
	throws CoreException {
		com.ebmwebsourcing.easyviper.core.api.engine.Process res = null;
		Scope current = this;
		Scope topScope = this;
		while (current != null) {
			current = this.getParentScope(current);
			if (current != null) {
				topScope = current;
			}
		}

		try {
			res = (com.ebmwebsourcing.easyviper.core.api.engine.Process) topScope.getComponent().getFcInterface("process");
		} catch (final NoSuchInterfaceException e) {
			// do nothing
		}

		return res;
	}

	public void linkInitialNodeAndExecution() throws CoreException {
		this.log.finest("start linkInitialNodeAndExecution");
		try {
			if (this.initialNode != null) {
				// Add binding between execution and initial node
				if (!FractalHelper.getFractalHelper().isBinded(
						this.parentExecution, "currentTarget")) {

					this.log.finest("bind execution to initial node");
					final List<Binding> listOfBindings = new ArrayList<Binding>();
					listOfBindings.add(new Binding("currentTarget",
							(org.objectweb.fractal.api.Interface) this
							.getInitialNode()));
					FractalHelper.getFractalHelper().addBindings(
							this.parentExecution, listOfBindings);
					this.log.finest("execution and initial node binded");

				}

				// link the initial node to the execution
				if (!FractalHelper
						.getFractalHelper()
						.isAlreadyBind(
								this.initialNode,
								"execution",
								(org.objectweb.fractal.api.Interface) this.parentExecution)) {
					this.linkedExecutableElement2Execution(
							(ExecutableElement) this.initialNode
							.getFcInterface("service"),
							(Execution) this.parentExecution
							.getFcInterface("service"));
					this.log.finest("initial and execution linked");
				}
			} else {
				log.finest("Initial node not setted");
			}
		} catch (final NoSuchInterfaceException e) {
			this.log.severe(e.getMessage());
			throw new CoreException(e);
		}
		this.log.finest("end of linkInitialNodeAndExecution");
	}

	@Override
	public void setLog(final Logger logger) {
		super.setLog(logger);
		this.log = logger;
	}

	private String formatCounter(int cpt) {
		String res = "";
		if(cpt < 10) {
			res = "000" + cpt;
		} else if(cpt < 100) {
			res = "00" + cpt;
		} else if(cpt < 1000) {
			res = "0" + cpt;
		} else {
			res = String.valueOf(cpt);
		}
		return res;
	}

	public Map<Object, Endpoint> getEndpoints() {
		return endpoints;
	}

	public List<CorrelationGroup> getCorrelationGroups() {
		return correlationGroups;
	}

	public Map<Fault, Scope> getExceptions() {
		return this.exceptions;
	}

	public void setExceptions(Map<Fault, Scope> exceptions) {
		this.exceptions = exceptions;
	}

	public Exception isFaultScope() {
		return this.faultScope;
	}

	@Override
	public Node getParentNode() {
		return this.parentNode;
	}

	@Override
	public void setParentNode(Node parentNode) {
		this.parentNode = parentNode;
	}

	public void setFaultScope(Exception faultScope) {
		this.faultScope = faultScope;
	}

	public void setParentExecution(Component parentExecution) {
		this.parentExecution = parentExecution;
	}

	public void setTerminationHandler(TerminationHandler th) {
		this.th = th;
	}

	public TerminationHandler getTerminationHandler() {
		return this.th;
	}
}
