/**
 * easy VIPER software - Copyright (c) 2009 PetalsLink, 
 * http://www.petalslink.com/ 
 *  
 * This library is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2.1 of the License, or (at your option) 
 * any later version. This library is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
 * General Public License for more details. 
 *  
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 *  
 * ------------------------------------------------------------------------- 
 * $Id$ 
 * ------------------------------------------------------------------------- 
 */ 
package com.ebmwebsourcing.easyviper.core.impl.engine;

import static org.objectweb.fractal.fraclet.types.Step.CREATE;
import static org.objectweb.fractal.fraclet.types.Step.DESTROY;
import static org.objectweb.fractal.fraclet.types.Step.START;
import static org.objectweb.fractal.fraclet.types.Step.STOP;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import javax.xml.namespace.QName;

import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.fraclet.annotations.Controller;
import org.objectweb.fractal.fraclet.annotations.Interface;
import org.objectweb.fractal.fraclet.annotations.Lifecycle;
import org.objectweb.fractal.fraclet.annotations.Requires;
import org.objectweb.fractal.fraclet.extensions.Membrane;
import org.objectweb.fractal.fraclet.types.Contingency;

import com.ebmwebsourcing.easyviper.core.api.Core;
import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Engine;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.Node;
import com.ebmwebsourcing.easyviper.core.api.engine.Process;
import com.ebmwebsourcing.easyviper.core.api.engine.Process.State;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.ReceiverBehaviour;
import com.ebmwebsourcing.easyviper.core.api.engine.registry.ProcessInstanceRegistry;
import com.ebmwebsourcing.easyviper.core.api.engine.thread.service.AutoTrashProcessService;
import com.ebmwebsourcing.easyviper.core.api.engine.thread.service.ServiceManager;
import com.ebmwebsourcing.easyviper.core.api.env.ExternalContext;
import com.ebmwebsourcing.easyviper.core.api.env.ExternalEnvironment;
import com.ebmwebsourcing.easyviper.core.api.model.Model;
import com.ebmwebsourcing.easyviper.core.api.model.registry.ProcessKey;
import com.ebmwebsourcing.easyviper.core.api.model.registry.definition.ProcessDefinition;
import com.ebmwebsourcing.easyviper.core.api.soa.Endpoint;
import com.ebmwebsourcing.easyviper.core.api.soa.message.InternalMessage;
import com.ebmwebsourcing.easyviper.core.fractal.FractalHelper;
import com.ebmwebsourcing.easyviper.core.impl.engine.registry.MemoryProcessInstanceRegistryImpl;
import com.ebmwebsourcing.easyviper.core.impl.engine.thread.service.AutoFlushMessageServiceImpl;
import com.ebmwebsourcing.easyviper.core.impl.engine.thread.service.AutoTrashProcessServiceImpl;
import com.ebmwebsourcing.easyviper.core.impl.engine.thread.service.ServiceManagerImpl;
import com.ebmwebsourcing.easyviper.core.impl.model.registry.ProcessKeyImpl;

/**
 * @author Nicolas Salatge - eBM WebSourcing
 */
@org.objectweb.fractal.fraclet.annotations.Component(provides = @Interface(name = "service", signature = Engine.class))
@Membrane(controller = "composite")
public class EngineImpl implements Engine {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	@Controller
	private Component component = null;

	private Logger log = Logger.getLogger(EngineImpl.class.getName());

	@Requires(name = "externalEnv", contingency = Contingency.OPTIONAL)
	protected ExternalEnvironment externalEnvironment;

	@Requires(name = "model", contingency = Contingency.OPTIONAL)
	private Model model;

	private ProcessInstanceRegistry processInstanceRegistry;


	private Class internalMessageType;

	private ServiceManager serviceManager = new ServiceManagerImpl(this);

	/**
	 * Creates a new instance of {@link PetalsServer}
	 * 
	 * @throws PvmException
	 */
	public EngineImpl() throws CoreException {
	}

	public Component getComponent() {
		return this.component;
	}

	/**
	 * Create the scope behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = CREATE)
	public void create() throws CoreException {
		this.log.fine("Fractal core created: "
				+ FractalHelper.getFractalHelper().getName(this.component));
	}

	public void init(final Component fractalNode) throws CoreException {
		this.component = fractalNode;
		this.processInstanceRegistry = new MemoryProcessInstanceRegistryImpl(
				this);
		this.log.fine("Fractal core initiated: "
				+ FractalHelper.getFractalHelper().getName(this.component));
	}

	/**
	 * Start the NodeImpl behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = START)
	public void start() throws CoreException {
		this.log.fine("Fractal core started: " + this.getName());
	}

	/**
	 * Stop the NodeImpl behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = STOP)
	public void stop() throws CoreException {
		this.log.finest("Fractal core stopped: " + this.getName());
	}

	/**
	 * Destroy the NodeImpl behaviour
	 * 
	 * @throws CoreException
	 */
	@Lifecycle(step = DESTROY)
	public void destroy() throws CoreException {
		this.log.fine("Fractal core destroyed: " + this.getName());
	}

	public String getName() throws CoreException {
		return FractalHelper.getFractalHelper().getName(this.component);
	}

	public void setName(final String name) throws CoreException {
		if (name != null) {
			FractalHelper.getFractalHelper().changeName(this.component, name);
		}
	}

	/**
	 * constructor to feed in subclasses of processes.
	 * 
	 * @throws CoreException
	 */
	private Component createEmptyProcess(final String processName)
	throws CoreException {
		final Component process = FractalHelper.getFractalHelper()
		.createNewComponent(ProcessImpl.class.getName(), null);
		FractalHelper.getFractalHelper().addComponent(process,
				this.getComponent(), null);
		try {
			Process processDefinition = (Process) process
			.getFcInterface("/content");
			(processDefinition).init(process);
			(processDefinition).setName(processName);
			if (!this.log.getName().equals(EngineImpl.class.getName())) {
				processDefinition.setLog(this.log);
			}

			this.log.fine("Creation of the fractal process: "
					+ (processDefinition).getName());

			// Start the process
			FractalHelper.getFractalHelper().startComponent(process);
			processDefinition = (Process) process.getFcInterface("process");

		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
		return process;
	}

	public synchronized Process createNewEmptyProcessInstance(
			final QName uniqueProcessName, ProcessDefinition processDefinition) throws CoreException {
		this.log.finest("start createNewEmptyProcessInstance");
		Process process = null;

		this.checkModel();

		// create an empty process
		final Component processDefinitionComp = this
		.createEmptyProcess(uniqueProcessName.getLocalPart());
		try {
			process = (Process) processDefinitionComp.getFcInterface("process");
		} catch (final NoSuchInterfaceException e) {
			this.log.severe(e.getMessage());
			e.printStackTrace();
			throw new CoreException(e);
		}



		if (this.model == null) {
			this.log
			.finest("store the process instance in registry in instance registry");
			// store in instance registry
			final ProcessKey key = new ProcessKeyImpl(uniqueProcessName, null, null);
			List<ProcessKey> keys = new ArrayList<ProcessKey>(1);
			keys.add(key);
			process.setProcessKeys(keys);

			this.processInstanceRegistry.storeProcessInstance(key, process);
		} else {
			List<ProcessKey> keys = this.model.getRegistry().createKeys(processDefinition);
			this.log.finest("Number of keys: "+keys.size());
			for(final ProcessKey key: keys){
				this.log.finest("key put in process instances map: "+key);

				this.getProcessInstanceRegistry().storeProcessInstance(key, process);
			}
			process.setProcessKeys(keys);
		}
		this.log.finest("end of createNewEmptyProcessInstance");
		return process;
	}

	// TODO: Fix bug: unnessary methods
	private void checkModel() throws CoreException {
		try {
			if (this.model == null) {
				final Component coreComp = FractalHelper.getFractalHelper()
				.getParent(this.getComponent());
				final Core core = (Core) coreComp
				.getFcInterface("service");
				this.model = core.getModel();
			}
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
	}

	// TODO: Fix bug: unnessary methods
	private void checkExternalEnvironment() throws CoreException {
		try {
			if (this.externalEnvironment == null) {
				final Component coreComp = FractalHelper.getFractalHelper()
				.getParent(this.getComponent());
				final Core core = (Core) coreComp
				.getFcInterface("service");
				this.externalEnvironment = core.getExternalEnvironment();
			}
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
	}

	public synchronized void accept(final InternalMessage<?> internalMessage,
			final ExternalContext context) throws CoreException {


		if (this.internalMessageType == null) {
			throw new CoreException(
			"Configuration error: The type of internal message must be setted");
		}

		this.checkModel();

		// load the good instance
		Execution validProcessOrExecution = this.acceptInternalMessage(internalMessage, context);

		if (validProcessOrExecution == null) {
			this.log.finest("no process found => store the received message");
			AutoFlushMessageServiceImpl autoflush = this.serviceManager.getService(AutoFlushMessageServiceImpl.class);
			autoflush.addMessagesInRegistry(internalMessage, context);
		} else {
			// set the context
			if(validProcessOrExecution.getParentScope().getProcess().getExternalContext() == null) {
				this.log.finest("set the context: " + context + " into process instance " + validProcessOrExecution.getParentScope().getProcess().getName());
				validProcessOrExecution.getParentScope().getProcess().setExternalContext(
						context);
			}

			// run the process
			ThreadRunProcess processThread = new ThreadRunProcess(validProcessOrExecution, context);
			processThread.start();
		}
	}

	private Execution acceptInternalMessage(final InternalMessage<?> internalMessage,
			final ExternalContext context) throws CoreException {



		Execution validExecution = null;

		

		// get the good instance of process
		if (this.processInstanceRegistry != null) {
			List<Process> processes = new ArrayList<Process>();

			// find the good instances able to handle the message
			this.log.finest("try to find an available instance");
			processes = this.processInstanceRegistry
			.getProcessInstances(internalMessage);

			if(processes != null) {
				this.log.finest("number of potential process: " + processes.size());
				String proc = null;
				for(Process p: processes) {
					if(proc == null) {
						proc = p.getName();
					} else {
						proc = proc + ", " + p.getName();
					}
				}
				this.log.finest("potential process list : " + proc);
			} else {
				this.log.finest("number of potential process: 0");
			}
			if ((processes != null) && (processes.size() > 0)) {

				List<Execution> possibleExecution = new ArrayList<Execution>();
				boolean reinitOrSuspended = false;

				//Parsing all processes corresponding to the message
				Iterator<Process> it = processes.iterator();
				while(it.hasNext()) {
					Process process = it.next();

					reinitOrSuspended = false;

					try {
						log.finest("test process: " + process.getName());

						// reinit all ended process
						if ((process.getParentExecution() != null)
								&& (process.getParentExecution().getState() == Execution.State.ENDED)) {
							this.log.finest("reinit and restart an older instance: " + process.getName());
							process.end(true);
							process.setExternalContext(null);
							process.run();
						} else if (process.getParentExecution() == null) {
							this.log.finest("reinit and restart an older instance: " + process.getName());
							process.setExternalContext(null);
							process.run();
						}
						reinitOrSuspended = true;
					} catch(CoreException e) {
						// error to renit process
						process.setState(State.PROCESS_UNSTABLE);
						processes.remove(process);
						it = processes.iterator();
						AutoTrashProcessService autoTrash = this.serviceManager.getService(AutoTrashProcessServiceImpl.class);
						autoTrash.addUninstableProcessInstance(process);
					}
					if(reinitOrSuspended) {
						// give the message to each suspended receiver behaviour (either it accept or not)
						try {

							final Execution exec = this.injectMessageInProcessInstance(process,
									internalMessage, context);

							if (exec != null) {

								this.log
								.finest("available instance found => restart the process " + process.getName());
								process.setState(Process.State.PROCESS_ACTIVE);

								// active the execution
								exec.setStateRecursively(Execution.State.ACTIVE);

								possibleExecution.add(exec);

								// CURRENT STRATEGY: stop when an execution is found.
								break;
							}
						} catch (final CoreException e) {
							this.log.severe("Error injecting a message in process " + process.getName() + ": " + e.getMessage());
							// do nothing
							e.printStackTrace();
						}
					}

				}


				if(possibleExecution.size() > 0) {
					validExecution = possibleExecution.get(0);
				}
			}
		}

		if (validExecution == null) {
			this.log
			.finest("no available instance found");


			// no instance process found => create a new instance IF receiver is "createInstance=YES"
			if (this.model != null) {


				final ProcessKey key = new ProcessKeyImpl();
				key.setEndpoint(internalMessage.getEndpoint());
				key.setService(internalMessage.getService());

				// find definition
				final ProcessDefinition definition = this.model.getRegistry()
				.getProcessDefinition(key);
				if (definition == null) {
					throw new CoreException(
							"Impossible to find definition corresponding to this key: "
							+ key);
				}

				//Check if create instance is set to yes or not
				if(isCreateInstance(definition, internalMessage)){

					// compile the definition
					final Process process = this.model.getCompiler().compile(definition);


					if (process != null) {

						// run the instance process
						this.log.finest("run the process...");
						process.run();


						// inject the message in process instance
						this.log.finest("inject the message in process");
						final Execution exec = this.injectMessageInProcessInstance(process,
								internalMessage, context);


						if(exec != null) {
							// active the execution
							exec.setState(Execution.State.ACTIVE);
							validExecution = exec;
						} 
					} else {
						this.log
						.finest("no process definition found concerned by the received message: "
								+ key);
					}
				}else{
					//do nothing, message will be stored
				}
			}
		}
		return validExecution;
	}


	private Execution injectMessageInProcessInstance(final Process process,
			final InternalMessage<?> internalMessage, ExternalContext context) throws CoreException {
		Execution res = null;
		final Map<String, Execution> execs = process.getSuspendedExecutions();

		this.log.finest("list of execution to try: " + execs);

		//List<CoreException> exceptions = new ArrayList<CoreException>();
		for(Execution exec: execs.values()) {
			try {
				this.log.finest("try to inject message in process " + process.getName() + " using execution " + exec.getName());
				if (exec.getCurrentTarget() instanceof Node) {
					final Node node = (Node) exec.getCurrentTarget();
					if (node.getBehaviour() instanceof ReceiverBehaviour) {
						final ReceiverBehaviour receiverBehaviour = (ReceiverBehaviour) node
						.getBehaviour();
						if (receiverBehaviour.getMessage() == null) {
							receiverBehaviour.setMessage(internalMessage);
							this.log.finest("message setted in receive behaviour: "
									+ receiverBehaviour.getName());
							boolean accept = receiverBehaviour.accept(internalMessage, context);
							this.log.finest("accepted by receiver: " + receiverBehaviour.getName() + " ? " + accept);
							if(accept) {
								res = exec;
								break;
							} else {
								receiverBehaviour.setMessage(null);
							}
						} else {
							this.log.finest("the receiver " + receiverBehaviour.getName() + " already contains a message");
						}
					} else {
						throw new CoreException(
								"Error: the node selected is not concerned by the received message "+node.getBehaviour().getName()+"::"+node.getClass().getCanonicalName());
					}
				} else {
					throw new CoreException(
					"Error: the execution selected is not concerned by the received message");
				}
			} catch(CoreException e) {
				this.log.finest("Error in injectMessageInProcessInstance: " + e.getMessage());
				//exceptions.add(e);
			}
		}

//		if(res == null && exceptions.size() > 0) {
//			throw exceptions.get(0);
//		}

		return res;
	}

	public void flushMessagesInRegistry() throws CoreException {
		this.log.finest("flush stored messages");
		AutoFlushMessageServiceImpl autoflush = this.serviceManager.getService(AutoFlushMessageServiceImpl.class);
		autoflush.flushMessagesInRegistry();
	}

	public void setInternalMessageType(final Class clazz) {
		this.internalMessageType = clazz;
	}

	public Class getInternalMessageType() {
		return this.internalMessageType;
	}

	public void sendTo(final InternalMessage<?> message, final Endpoint providerEndpoint,
			final ExternalContext context) throws CoreException {

		this.checkExternalEnvironment();
		if ((this.externalEnvironment.getSenders() != null)
				&& (this.externalEnvironment.getSenders().size() > 0)) {

			// TODO: Set strategy to select the good senders
			this.externalEnvironment.getSenders().get(0).sendTo(message,
					providerEndpoint, context);
		} else {
			throw new CoreException("Senders does not exist!!!");
		}
	}

	public InternalMessage<?> sendSyncTo(final InternalMessage<?> request,
			final Endpoint providerEndpoint, final ExternalContext context)
			throws CoreException {
		InternalMessage<?> res = null;
		this.checkExternalEnvironment();
		if ((this.externalEnvironment.getSenders() != null)
				&& (this.externalEnvironment.getSenders().size() > 0)) {

			// TODO: Set strategy to select the good senders
			res = this.externalEnvironment.getSenders().get(0).sendSyncTo(
					request, providerEndpoint, context);
		} else {
			throw new CoreException("Senders does not exist!!!");
		}
		return res;
	}

	public ProcessInstanceRegistry getProcessInstanceRegistry() {
		return this.processInstanceRegistry;
	}

	public void setLog(final Logger logger) {
		this.log = logger;
	}

	public void deleteProcess(final Process process) throws CoreException {
		FractalHelper.getFractalHelper()
		.deleteComponent(process.getComponent());
	}

	public Core getCore() throws CoreException {
		Core res = null;
		try {
			final Component coreComp = FractalHelper.getFractalHelper().getParent(
					this.getComponent());
			res = (Core) coreComp.getFcInterface("service");
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		}
		return res;
	}

	private class ThreadRunProcess extends Thread {

		private Execution validProcessOrExecution;

		private ExternalContext context;

		public ThreadRunProcess(Execution validProcessOrExecution, ExternalContext context) {
			super();
			this.validProcessOrExecution = validProcessOrExecution;
			this.context = context;
		}

		@Override
		public void run() {
			try {
				((Execution)validProcessOrExecution).next();
			} catch (Exception e) {
				if(e instanceof CoreException) {
					try {
						this.validProcessOrExecution.getParentScope().getProcess().getEngine().getCore().getExternalEnvironment().getSenders().get(0).sendTo((CoreException)e, context);
					} catch (CoreException e1) {
						this.validProcessOrExecution.setState(Execution.State.ENDED);
						try {
							this.validProcessOrExecution.end();
						} catch (CoreException e2) {
							e1.printStackTrace();
							log.severe(e1.getMessage());
						}
						e1.printStackTrace();
						log.severe(e1.getMessage());
					}
				} else {
					e.printStackTrace();
					log.severe(e.getMessage());
				}
			} 
		}



	}

	public ServiceManager getServiceManager() {
		return this.serviceManager;
	}

	public ClassLoader getClassLoader() {
		return Thread.currentThread().getContextClassLoader();
	}


	private boolean isCreateInstance(ProcessDefinition def,
			InternalMessage<?> mess) throws CoreException {

		return this.model.getRegistry().isCreateInstance(def, mess);

	}


}
