/**
 * easy VIPER software - Copyright (c) 2009 PetalsLink, 
 * http://www.petalslink.com/ 
 *  
 * This library is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2.1 of the License, or (at your option) 
 * any later version. This library is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
 * General Public License for more details. 
 *  
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 *  
 * ------------------------------------------------------------------------- 
 * $Id$ 
 * ------------------------------------------------------------------------- 
 */ 
package com.ebmwebsourcing.easyviper.core.impl.engine;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import javax.xml.namespace.QName;

import org.oasisopen.sca.annotation.PolicySets;
import org.oasisopen.sca.annotation.Reference;
import org.oasisopen.sca.annotation.Scope;
import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.ow2.frascati.tinfi.api.control.ContentInstantiationException;
import org.ow2.frascati.tinfi.api.control.SCAContentController;
import org.ow2.frascati.tinfi.control.content.SCAExtendedContentController;

import com.ebmwebsourcing.easycommons.sca.helper.api.SCAException;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAComponentImpl;
import com.ebmwebsourcing.easycommons.sca.helper.impl.SCAHelper;
import com.ebmwebsourcing.easyviper.core.api.Core;
import com.ebmwebsourcing.easyviper.core.api.CoreException;
import com.ebmwebsourcing.easyviper.core.api.engine.Engine;
import com.ebmwebsourcing.easyviper.core.api.engine.Execution;
import com.ebmwebsourcing.easyviper.core.api.engine.Node;
import com.ebmwebsourcing.easyviper.core.api.engine.Process;
import com.ebmwebsourcing.easyviper.core.api.engine.Process.State;
import com.ebmwebsourcing.easyviper.core.api.engine.behaviour.functionnal.ReceiverBehaviour;
import com.ebmwebsourcing.easyviper.core.api.engine.registry.ProcessInstanceRegistry;
import com.ebmwebsourcing.easyviper.core.api.engine.thread.service.ServiceManager;
import com.ebmwebsourcing.easyviper.core.api.env.ExternalContext;
import com.ebmwebsourcing.easyviper.core.api.env.ExternalEnvironment;
import com.ebmwebsourcing.easyviper.core.api.model.Model;
import com.ebmwebsourcing.easyviper.core.api.model.registry.ProcessKey;
import com.ebmwebsourcing.easyviper.core.api.model.registry.definition.ProcessDefinition;
import com.ebmwebsourcing.easyviper.core.api.soa.Endpoint;
import com.ebmwebsourcing.easyviper.core.api.soa.message.InternalMessage;
import com.ebmwebsourcing.easyviper.core.api.soa.message.Message;
import com.ebmwebsourcing.easyviper.core.api.tools.PoolOfProcessForkers;
import com.ebmwebsourcing.easyviper.core.impl.engine.registry.MemoryProcessInstanceRegistryImpl;
import com.ebmwebsourcing.easyviper.core.impl.engine.thread.service.ServiceManagerImpl;
import com.ebmwebsourcing.easyviper.core.impl.model.registry.ProcessKeyImpl;
import com.ebmwebsourcing.easyviper.extended.service.autoflush.impl.AutoFlushMessageServiceImpl;
import com.ebmwebsourcing.easyviper.extended.service.autotrash.api.AutoTrashProcessService;
import com.ebmwebsourcing.easyviper.extended.service.autotrash.impl.AutoTrashProcessServiceImpl;

/**
 * @author Nicolas Salatge - eBM WebSourcing
 */
@Scope("COMPOSITE")
@org.oasisopen.sca.annotation.Service(value=Engine.class,names="service")
@PolicySets("frascati:scaEasyCompositeWithContent")
public class EngineImpl extends SCAComponentImpl implements Engine {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private Logger log = Logger.getLogger(EngineImpl.class.getName());

	@Reference(name="externalEnv",required=false)
	protected ExternalEnvironment externalEnvironment;

	@Reference(name="model",required=false)
	private Model model;

	private ProcessInstanceRegistry processInstanceRegistry =
		new MemoryProcessInstanceRegistryImpl(this);


	private Class internalMessageType;

	private ServiceManager serviceManager = new ServiceManagerImpl(this);

	private PoolOfProcessForkers poolOfProcessForkers = null;

	/**
	 * Creates a new instance
	 * 
	 * @throws CoreException
	 */
	public EngineImpl() throws CoreException {

		poolOfProcessForkers = new PoolOfProcessForkers();
	}


	/**
	 * constructor to feed in subclasses of processes.
	 * 
	 * @throws CoreException
	 */
	private Component createEmptyProcess(final String processName)
	throws CoreException {
		Component process = null;
		try {
			process = SCAHelper.getSCAHelper()
			.createNewComponent(ProcessImpl.class.getName(), null);
			SCAHelper.getSCAHelper().addComponent(process,
					this.getComponent(), null);

			SCAContentController scacc = (SCAContentController)
			process.getFcInterface(SCAContentController.NAME);
			Process processDefinition;
			try {
				processDefinition = (Process) scacc.getFcContent();
			}
			catch (ContentInstantiationException e) {
				throw new CoreException(e);
			}
			(processDefinition).setName(processName);
			if (!this.log.getName().equals(EngineImpl.class.getName())) {
				processDefinition.setLog(this.log);
			}

			this.log.fine("Creation of the fractal process: "
					+ (processDefinition).getName());

			// Start the process
			SCAHelper.getSCAHelper().startComponent(process);
			processDefinition = (Process) process.getFcInterface("service");

		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		} catch (SCAException e) {
			throw new CoreException(e);
		}
		return process;
	}

	public synchronized Process createNewEmptyProcessInstance(
			final QName uniqueProcessName, ProcessDefinition processDefinition) throws CoreException {
		this.log.finest("start createNewEmptyProcessInstance");
		Process process = null;

		this.checkModel();

		// create an empty process
		final Component processDefinitionComp = this
		.createEmptyProcess(uniqueProcessName.getLocalPart());
		try {
			process = (Process) processDefinitionComp.getFcInterface("service");
		} catch (final NoSuchInterfaceException e) {
			this.log.severe(e.getMessage());
			e.printStackTrace();
			throw new CoreException(e);
		}

		if (this.model == null) {
			this.log
			.finest("store the process instance in registry in instance registry");
			// store in instance registry
			final ProcessKey key = new ProcessKeyImpl(uniqueProcessName, null, null);
			List<ProcessKey> keys = new ArrayList<ProcessKey>(1);
			keys.add(key);
			process.setProcessKeys(keys);

			this.processInstanceRegistry.storeProcessInstance(key, process);
		} else {
			List<ProcessKey> keys = this.model.getRegistry().createKeys(processDefinition);
			this.log.finest("Number of keys: "+keys.size());
			for(final ProcessKey key: keys){
				this.log.finest("key put in process instances map: "+key);

				this.getProcessInstanceRegistry().storeProcessInstance(key, process);
			}
			process.setProcessKeys(keys);
		}
		this.log.finest("end of createNewEmptyProcessInstance");
		return process;
	}

	// TODO: Fix bug: unnessary methods
	private void checkModel() throws CoreException {
		try {
			if (this.model == null) {
				final Component coreComp = SCAHelper.getSCAHelper()
				.getParent(this.getComponent());
				final Core core = (Core) coreComp
				.getFcInterface("service");
				this.model = core.getModel();
			}
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		} catch (SCAException e) {
			throw new CoreException(e);
		}
	}

	// TODO: Fix bug: unnessary methods
	private void checkExternalEnvironment() throws CoreException {
		try {
			if (this.externalEnvironment == null) {
				final Component coreComp = SCAHelper.getSCAHelper()
				.getParent(this.getComponent());
				final Core core = (Core) coreComp
				.getFcInterface("service");
				this.externalEnvironment = core.getExternalEnvironment();
			}
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		} catch (SCAException e) {
			throw new CoreException(e);
		}
	}

	public synchronized void accept(final InternalMessage<?> internalMessage,
			final ExternalContext context) throws CoreException {


		if (this.internalMessageType == null) {
			throw new CoreException(
			"Configuration error: The type of internal message must be setted");
		}

		this.checkModel();

		// load the good instance
		Execution validExecution = this.acceptInternalMessage(internalMessage, context);

		if (validExecution == null) {
			this.log.finest("no process found => store the received message");
			AutoFlushMessageServiceImpl autoflush = this.serviceManager.getService(AutoFlushMessageServiceImpl.class);
			autoflush.addMessagesInRegistry(internalMessage, context);
		} else {
			// set the context
			//TODO Check Reflexion on what is a context (implements equality ...)

			Endpoint endpoint = findEndpointFromMessage(internalMessage, validExecution.getParentScope().getProcess());//this.getProcessInstanceRegistry());

			if(validExecution.getParentScope().getProcess().getExternalContext(endpoint, internalMessage.getOperationName()) == null){ 

				this.log.finest("set the context: " + context + " into process instance " + validExecution.getParentScope().getProcess().getName());

				validExecution.getParentScope().getProcess().addExternalContext(endpoint, internalMessage.getOperationName(),
						context);
			}

			// restart threads for all necessary executions.
			Execution execution = validExecution;
			Collection<String> threadNames = getAllActiveThreadNames();
			while (execution != null) {
				// check if there is already a thread running this execution.
				// we know it because we created each execution thread using execution
				// hashCode.
				if (!threadNames.contains(execution.hashCode())) {
					log.finest("Spawning (or respawning) thread for execution " + execution.getName());
					ThreadRunProcess processThread = new ThreadRunProcess(execution, context);//validExecution.getParentScope().getProcess().getExternalContexts());
					processThread.start();
				}
				execution = execution.getParentExecution();
			}
		}
	}

	private final Collection<String> getAllActiveThreadNames() {
		Collection<String> result = new HashSet<String>();
		// TODO : not very nice, should be reimplemented with activeCount and enumerate
		// caring about races...
		Map<Thread,StackTraceElement[]> threads = Thread.getAllStackTraces();
		for (Thread t : threads.keySet()) {
			result.add(t.getName());
		}
		return result;
	}



	private final Execution findFirstAvailableExecution(final InternalMessage<?> internalMessage,
			final ExternalContext context) throws CoreException {
		Execution validExecution = null;

		// get the good instance of process
		if (this.processInstanceRegistry != null) {
			List<Process> processes = new ArrayList<Process>();

			// find the good instances able to handle the message
			this.log.finest("try to find an available instance");
			processes = this.processInstanceRegistry
			.getProcessInstances(internalMessage);

			if(processes != null) {
				this.log.finest("number of potential process: " + processes.size());
				String proc = null;
				for(Process p: processes) {
					if(proc == null) {
						proc = p.getName();
					} else {
						proc = proc + ", " + p.getName();
					}
				}
				this.log.finest("potential process list : " + proc);
			} else {
				this.log.finest("number of potential process: 0");
			}
			if ((processes != null) && (processes.size() > 0)) {

				List<Execution> possibleExecution = new ArrayList<Execution>();
				boolean reinitOrSuspended = false;

				//Parsing all processes corresponding to the message
				Iterator<Process> it = processes.iterator();
				while(it.hasNext()) {
					Process process = it.next();

					reinitOrSuspended = false;

					try {
						log.finest("test process: " + process.getName());

						Endpoint endpoint = findEndpointFromMessage(internalMessage, process);//this.processInstanceRegistry);
						// reinit all ended process
						if ((process.getParentExecution() != null)
								&& (process.getParentExecution().getState() == Execution.State.ENDED)) {
							this.log.finest("reinit and restart an older instance: " + process.getName());
							process.end(true);

							process.removeExternalContext(endpoint, internalMessage.getOperationName());//setExternalContext(null);

							process.run();

						} else if (process.getParentExecution() == null) {
							this.log.finest("reinit and restart an older instance: " + process.getName());
							process.removeExternalContext(endpoint, internalMessage.getOperationName());//setExternalContext(null);

							process.run();
						}
						reinitOrSuspended = true;
					} catch(CoreException e) {
						// error to renit process
						process.setState(State.PROCESS_UNSTABLE);
						processes.remove(process);
						it = processes.iterator();
						AutoTrashProcessService autoTrash = this.serviceManager.getService(AutoTrashProcessServiceImpl.class);
						autoTrash.addUninstableProcessInstance(process);
					}
					if(reinitOrSuspended) {
						// give the message to each suspended receiver behaviour (either it accept or not)
						try {

							// this does not really feed with the message, this is just to execute matchers to see
							// if the execution is capable of handling internal message.
							final Execution exec = this.injectMessageInProcessInstance(process,
									internalMessage, context);

							if (exec != null) {

								this.log
								.finest("available instance found => restart the process " + process.getName());
								process.setState(Process.State.PROCESS_ACTIVE);

								// active the execution
								exec.setStateRecursively(Execution.State.ACTIVE);

								possibleExecution.add(exec);

								// CURRENT STRATEGY: stop when an execution is found.
								break;
							}
						} catch (final CoreException e) {
							this.log.severe("Error injecting a message in process " + process.getName() + ": " + e.getMessage());
							// do nothing
							e.printStackTrace();
						}
					}

				}

				if(possibleExecution.size() > 0) {
					validExecution = possibleExecution.get(0);
				}
			}
		}
		return validExecution;

	}


	private Execution acceptInternalMessage(final InternalMessage<?> internalMessage,
			final ExternalContext context) throws CoreException {

		Execution validExecution = findFirstAvailableExecution(internalMessage, context);

		if (validExecution == null) {
			this.log
			.finest("no available instance found");


			// no instance process found => create a new instance IF receiver is "createInstance=YES"
			if (this.model != null) {


				final ProcessKey key = new ProcessKeyImpl();
				key.setEndpoint(internalMessage.getEndpoint());
				key.setService(internalMessage.getService());

				// find definition
				final ProcessDefinition definition = this.model.getRegistry()
				.getProcessDefinition(key);
				if (definition == null) {
					throw new CoreException(
							"Impossible to find definition corresponding to this key: "
							+ key);
				}

				//Check if create instance is set to yes or not
				if(isCreateInstance(definition, internalMessage)){

					// compile the definition
					final Process process = this.model.getCompiler().compile(definition);


					if (process != null) {

						// run the instance process
						this.log.finest("run the process...");
						process.run();


						// inject the message in process instance
						this.log.finest("inject the message in process");
						final Execution exec = this.injectMessageInProcessInstance(process,
								internalMessage, context);


						if(exec != null) {
							// active the execution
							exec.setStateRecursively(Execution.State.ACTIVE);
							validExecution = exec;
						} 
					} else {
						this.log
						.finest("no process definition found concerned by the received message: "
								+ key);
					}
				}else{
					//do nothing, message will be stored
				}
			}
		}
		return validExecution;
	}


	private Execution injectMessageInProcessInstance(final Process process,
			final InternalMessage<?> internalMessage, ExternalContext context) throws CoreException {
		Execution res = null;
		final Map<String, Execution> execs = process.getSuspendedExecutions();

		this.log.finest("list of execution to try: " + execs);

		for(Execution exec: execs.values()) {
			try {
				this.log.finest("try to inject message in process " + process.getName() + " using execution " + exec.getName());
				if (exec.getCurrentTarget() instanceof Node) {
					final Node node = (Node) exec.getCurrentTarget();
					if (node.getBehaviour() instanceof ReceiverBehaviour) {
						final ReceiverBehaviour receiverBehaviour = (ReceiverBehaviour) node
						.getBehaviour();
						if (receiverBehaviour.getMessage() == null) {
							receiverBehaviour.setMessage(internalMessage);
							this.log.finest("message setted in receive behaviour: "
									+ receiverBehaviour.getName());
							boolean accept = receiverBehaviour.accept(internalMessage, context);
							this.log.finest("accepted by receiver: " + receiverBehaviour.getName() + " ? " + accept);
							if(accept) {
								res = exec;
								break;
							} else {
								receiverBehaviour.setMessage(null);
							}
						} else {
							this.log.finest("the receiver " + receiverBehaviour.getName() + " already contains a message");
						}
					} else {
						throw new CoreException(
								"Error: the node selected is not concerned by the received message "+node.getBehaviour().getName()+"::"+node.getClass().getCanonicalName());
					}
				} else {
					throw new CoreException(
					"Error: the execution selected is not concerned by the received message");
				}
			} catch(CoreException e) {
				this.log.finest("Error in injectMessageInProcessInstance: " + e.getMessage());
			}
		}

		return res;
	}

	public void flushMessagesInRegistry() throws CoreException {
		this.log.finest("flush stored messages");
		AutoFlushMessageServiceImpl autoflush = this.serviceManager.getService(AutoFlushMessageServiceImpl.class);
		autoflush.flushMessagesInRegistry();
	}

	public void setInternalMessageType(final Class clazz) {
		this.internalMessageType = clazz;
	}

	public Class getInternalMessageType() {
		return this.internalMessageType;
	}

	public void sendTo(final InternalMessage<?> message, final Endpoint providerEndpoint,
			final Map<Endpoint, Map<String, ExternalContext>> context) throws CoreException {

		this.checkExternalEnvironment();
		if ((this.externalEnvironment.getSenders() != null)
				&& (this.externalEnvironment.getSenders().size() > 0)) {

			// TODO: Set strategy to select the good senders
			log.finest("Context map : "+context);
			Map<String, ExternalContext> map = context.get(providerEndpoint);
			ExternalContext ec = null;
			if(map != null){
				ec = map.get(message.getOperationName());
			}

			if(ec == null){

				assert !context.isEmpty();

				Collection<Map<String, ExternalContext>> coll = context.values();


				if(!coll.isEmpty()){

					Collection<ExternalContext> coll2 = coll.iterator().next().values();

					if(!coll2.isEmpty()) {

						ec = coll2.iterator().next();
					}
				}
			}
			assert ec != null;
			this.externalEnvironment.getSenders().get(0).sendTo(message,
					providerEndpoint, ec);
		} else {
			throw new CoreException("Senders does not exist!!!");
		}
	}

	public InternalMessage<?> sendSyncTo(final InternalMessage<?> request,
			final Endpoint providerEndpoint,  Map<Endpoint, Map<String, ExternalContext>> context)
			throws CoreException {
		InternalMessage<?> res = null;
		this.checkExternalEnvironment();
		if ((this.externalEnvironment.getSenders() != null)
				&& (this.externalEnvironment.getSenders().size() > 0)) {

			// TODO: Set strategy to select the good senders
			Map<String, ExternalContext> map = context.get(providerEndpoint);
			ExternalContext ec = null;
			if(map != null){
				ec = map.get(request.getOperationName());
			}
			if(ec == null){
				Collection<Map<String, ExternalContext>> coll = context.values();
				if(coll != null && !coll.isEmpty()){
					Collection<ExternalContext> coll2 = coll.iterator().next().values();
					if(coll2 != null && !coll2.isEmpty()){
						ec = coll2.iterator().next();
					}
				}
			}

			res = this.externalEnvironment.getSenders().get(0).sendSyncTo(
					request, providerEndpoint, ec);
		} else {
			throw new CoreException("Senders does not exist!!!");
		}
		return res;
	}

	public ProcessInstanceRegistry getProcessInstanceRegistry() {
		return this.processInstanceRegistry;
	}

	public void setLog(final Logger logger) {
		this.log = logger;
	}

	public void deleteProcess(final Process process) throws CoreException {
		try {
			SCAExtendedContentController parentContentController = (SCAExtendedContentController)
			process.getParentExecution().getComponent().getFcInterface(SCAContentController.NAME);

			parentContentController.releaseFcContent(process.getComponent(), true);

		} catch (NoSuchInterfaceException e) {
			throw new CoreException("Delete process "+process.getName()+ "failed. Caused by "+e.getMessage());
		}
	}

	public Core getCore() throws CoreException {
		Core res = null;
		try {
			final Component coreComp = SCAHelper.getSCAHelper().getParent(
					this.getComponent());
			res = (Core) coreComp.getFcInterface("service");
		} catch (final NoSuchInterfaceException e) {
			throw new CoreException(e);
		} catch (SCAException e) {
			throw new CoreException(e);
		}
		return res;
	}

	private class ThreadRunProcess extends Thread {

		private Execution validProcessOrExecution;

		private  ExternalContext context;

		public ThreadRunProcess(Execution validProcessOrExecution, ExternalContext context) {
			super(String.valueOf(validProcessOrExecution.hashCode()));
			this.validProcessOrExecution = validProcessOrExecution;
			this.context = context;
		}

		@Override
		public void run() {
			try {
				if (validProcessOrExecution.isStepByStep()) {
					validProcessOrExecution.runStepByStep();
				} else {
					validProcessOrExecution.run();
				}
			} catch (Exception e) {
				if(e instanceof CoreException) {
					try {
						this.validProcessOrExecution.getParentScope().getProcess().getEngine()
						.getCore().getExternalEnvironment().getSenders().get(0).sendTo((CoreException)e, 
								this.context);
					} catch (CoreException e1) {
						this.validProcessOrExecution.setState(Execution.State.ENDED);
						try {
							this.validProcessOrExecution.end();
						} catch (CoreException e2) {
							e1.printStackTrace();
							log.severe(e1.getMessage());
						}
						e1.printStackTrace();
						log.severe(e1.getMessage());
					}
				} else {
					e.printStackTrace();
					log.severe(e.getMessage());
				}
			} 
		}



	}

	public ServiceManager getServiceManager() {
		return this.serviceManager;
	}

	public ClassLoader getClassLoader() {
		return Thread.currentThread().getContextClassLoader();
	}


	private boolean isCreateInstance(ProcessDefinition def,
			InternalMessage<?> mess) throws CoreException {

		return this.model.getRegistry().isCreateInstance(def, mess);

	}

	public PoolOfProcessForkers getPoolOfProcessForkers(){
		return this.poolOfProcessForkers;
	}


	public Endpoint findEndpointFromMessage(Message<?> mess, Process process) {

		Collection<Endpoint> endpoints = process.getEndpoints().values();
		Iterator<Endpoint> itEndpoints = endpoints.iterator();
		Endpoint endpoint = null;
		
		while(itEndpoints.hasNext()){
			Endpoint current = itEndpoints.next();


			if(current.getEndpointName() != null && current.getServiceName() != null){
		
				
				if(current.getEndpointName().equalsIgnoreCase(mess.getEndpoint()) 
						&&
						current.getServiceName().equals(mess.getService())){
					endpoint = current;
					break;
				}
			}
		}

		return endpoint;
	}
	
	public Process findProcessFromName(String processInstanceName){
		Process p = null;

		ProcessKey key = new ProcessKeyImpl();

		QName processDefinitionName = extractProcessDefinitionName(processInstanceName);

		try {
			key = this.getCore().getModel().getRegistry().findProcessKey(processDefinitionName);
		
			List<Process> processInstances = this.getProcessInstanceRegistry().getProcessInstances(key);

			for(Process process : processInstances){
				
				if(process.getName().equals(processInstanceName)){
					p = process;
					break;
				}
			}
		}catch (CoreException e) {
			e.printStackTrace();
		}
		return p;
	}

	static private QName extractProcessDefinitionName(String processInstanceName) {
		QName res = null;

		int indexOf_ = processInstanceName.lastIndexOf("_");

		String valueOfQName = processInstanceName.substring(0, indexOf_);

		res = QName.valueOf(valueOfQName);
		return res;
	}
	

}
