
package com.ebmwebsourcing.enforcerrules;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.apache.maven.enforcer.rule.api.EnforcerRuleException;
import org.apache.maven.enforcer.rule.api.EnforcerRuleHelper;
import org.apache.maven.model.Plugin;
import org.apache.maven.model.PluginManagement;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;

public class NoPluginVersionOverride extends AbstractRule {

    @Override
    protected void doExecute(EnforcerRuleHelper helper) throws Exception {
        MavenProject mavenProject = (MavenProject) helper.evaluate("${project}");

        HashMap<MavenProject, List<Plugin>> parentPluginsByParentProject = getAllParentPluginsFromPluginMgtByProject(mavenProject);
        List<Plugin> plugins = getNewPlugins(mavenProject);

        for (Plugin plugin : plugins) {

            for (MavenProject parentProject : parentPluginsByParentProject.keySet()) {
                List<Plugin> parentPlugins = parentPluginsByParentProject.get(parentProject);
                for (Plugin parentPlugin : parentPlugins) {

                    if (isOverridingPluginVersion(parentPlugin, plugin)) {
                        String parentGroupId = parentProject.getGroupId();
                        String parentArtifactId = parentProject.getArtifactId();
                        String message = String
                                .format("The plugin %s:%s:%s override is not allowed : plugin %s:%s:%s"
                                        + " is defined in the pluginManagement section of the ancestor project %s:%s.",
                                        plugin.getGroupId(), plugin.getArtifactId(),
                                        plugin.getVersion(), parentPlugin.getGroupId(),
                                        parentPlugin.getArtifactId(), parentPlugin.getVersion(),
                                        parentGroupId, parentArtifactId);
                        throw new EnforcerRuleException(message);
                    }
                }
            }
        }
    }

    private static boolean isOverridingPluginVersion(Plugin parentPlugin, Plugin plugin) {
        String parentPluginGroupId = parentPlugin.getGroupId();
        String parentPluginArtifactId = parentPlugin.getArtifactId();
        String parentPluginVersion = parentPlugin.getVersion();

        String pluginGroupId = plugin.getGroupId();
        String pluginArtifactId = plugin.getArtifactId();
        String pluginVersion = plugin.getVersion();

        if (pluginGroupId.equals(parentPluginGroupId)
                && pluginArtifactId.equals(parentPluginArtifactId)) {
            if (pluginVersion == null)
                return false;
            return (!pluginVersion.equals(parentPluginVersion));
        } else {
            return false;
        }

    }

    /**
     * Gets the new plugins added by the current maven project (without plugins
     * got from the parent project)
     * 
     * @param mavenProject
     *            a maven project
     * @return the new plugins added by the current maven project (without
     *         plugins got from the parent project)
     */
    @SuppressWarnings("unchecked")
    private static final List<Plugin> getNewPlugins(MavenProject mavenProject) {
        List<String> allParentPluginId = getAllParentPlugins(mavenProject);
        List<Plugin> allPlugins = mavenProject.getBuildPlugins();
        return getNewPlugin(allParentPluginId, allPlugins);
    }

    /**
     * Get the plugin ID (the string groupId:artifactId) from all the ancestors
     * of the specified maven project
     * 
     * @param mavenProject
     *            a maven project
     * @return the plugin ID (the string groupId:artifactId) from all the
     *         ancestors
     */
    @SuppressWarnings("unchecked")
    private static final List<String> getAllParentPlugins(MavenProject mavenProject) {
        MavenProject parentProject = mavenProject.getParent();

        List<String> allParentDepId = new ArrayList<String>();
        ;
        if (parentProject != null) {
            List<Plugin> allParentPlugins = parentProject.getBuildPlugins();
            for (Plugin plugin : allParentPlugins) {
                String groupId = plugin.getGroupId();
                String artifactId = plugin.getArtifactId();
                String version = plugin.getVersion();
                String depParId = groupId + ":" + artifactId + ":" + version;

                allParentDepId.add(depParId);
            }
        }

        return allParentDepId;
    }

    /**
     * Get the plugins in the pluginManagement section for each ancestor of the
     * specified maven project
     * 
     * @param mavenProject
     *            a maven project
     * @return the plugins in the pluginManagement section for each ancestor. It
     *         returns a hashmap where the key is the maven project ancestor and
     *         the value the list of plugins in the pluginManagement section for
     *         the ancestor
     */
    private static final HashMap<MavenProject, List<Plugin>> getAllParentPluginsFromPluginMgtByProject(
            MavenProject mavenProject) {
        return getAllParentPluginsFromPluginMgtByProject(mavenProject, new ArrayList<String>());
    }

    @SuppressWarnings("unchecked")
    private static final HashMap<MavenProject, List<Plugin>> getAllParentPluginsFromPluginMgtByProject(
            MavenProject mavenProject, List<String> allParentDepFromDepMgt) {
        HashMap<MavenProject, List<Plugin>> parentPluginFromPluginMgt = new HashMap<MavenProject, List<Plugin>>();

        if (mavenProject.hasParent()) {
            MavenProject parentProject = mavenProject.getParent();
            parentPluginFromPluginMgt.putAll(getAllParentPluginsFromPluginMgtByProject(
                    parentProject, allParentDepFromDepMgt));
            PluginManagement pluginManagement = parentProject.getPluginManagement();
            if (pluginManagement != null) {
                List<Plugin> allPlugins = pluginManagement.getPlugins();
                List<Plugin> newDep = getNewPlugin(allParentDepFromDepMgt, allPlugins);

                parentPluginFromPluginMgt.put(parentProject, newDep);
            }
        }

        return parentPluginFromPluginMgt;
    }

    private static final List<Plugin> getNewPlugin(List<String> allParentPluginId,
            List<Plugin> allPlugins) {
        List<Plugin> newPlugins = new ArrayList<Plugin>();
        for (Plugin plugin : allPlugins) {
            String groupId = plugin.getGroupId();
            String artifactId = plugin.getArtifactId();
            String version = plugin.getVersion();
            String depParId = groupId + ":" + artifactId + ":" + version;

            if (!allParentPluginId.contains(depParId)) {
                newPlugins.add(plugin);
                allParentPluginId.add(depParId);
            }
        }
        return newPlugins;
    }

}
