
package com.ebmwebsourcing.enforcerrules;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.apache.maven.enforcer.rule.api.EnforcerRuleException;
import org.apache.maven.enforcer.rule.api.EnforcerRuleHelper;
import org.apache.maven.model.Dependency;
import org.apache.maven.model.DependencyManagement;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;

public class NoDependencyVersionOverride extends AbstractRule {

    private boolean mustFail;

    public NoDependencyVersionOverride() {
        this.mustFail = true;
    }

    @Override
    protected void doExecute(EnforcerRuleHelper helper) throws Exception {
        MavenProject mavenProject = (MavenProject) helper.evaluate("${project}");

        if (!isJbiSharedLibraryProject(mavenProject)) {
            HashMap<MavenProject, List<Dependency>> parentDependenciesByParentProject = getAllParentDepFromDepMgtByProject(mavenProject);
            List<Dependency> dependencies = getNewDependencies(mavenProject);

            for (Dependency dependency : dependencies) {
                String depGroupId = dependency.getGroupId();
                String depArtifactId = dependency.getArtifactId();
                String depVersion = dependency.getVersion();

                for (MavenProject parentProject : parentDependenciesByParentProject.keySet()) {
                    List<Dependency> parentDependencies = parentDependenciesByParentProject
                            .get(parentProject);
                    for (Dependency parentDependency : parentDependencies) {
                        String parentDepGroupId = parentDependency.getGroupId();
                        String parentDepArtifactId = parentDependency.getArtifactId();
                        String parentDepVersion = parentDependency.getVersion();

                        if (depGroupId.equals(parentDepGroupId)
                                && depArtifactId.equals(parentDepArtifactId)
                                && !depVersion.equals(parentDepVersion)) {
                            String parentGroupId = parentProject.getGroupId();
                            String parentArtifactId = parentProject.getArtifactId();
                            String message = String.format(
                                    "The dependency %s:%s:%s override is not allowed : dependency %s:%s:%s"
                                            + " is defined in the dependencyManagement section of the ancestor project %s:%s.",
                                    depGroupId, depArtifactId, depVersion,
                                    parentDepGroupId, parentDepArtifactId,
                                    parentDepVersion, parentGroupId, parentArtifactId);
                            if(mustFail == true) {
                                throw new EnforcerRuleException(
                                        message);
                            } else {
                                Log log = helper.getLog();
                                log.warn(message);
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * Gets the new dependencies added by the current maven project (without
     * dependency got from the parent project)
     *
     * @param mavenProject
     *            a maven project
     * @return the new dependencies added by the current maven project (without
     *         dependency got from the parent project)
     */
    @SuppressWarnings("unchecked")
    private static final List<Dependency> getNewDependencies(MavenProject mavenProject) {
        List<String> allParentDepId = getAllParentDep(mavenProject);
        List<Dependency> allDep = mavenProject.getDependencies();
        return getNewDep(allParentDepId, allDep);
    }

    /**
     * Get the dependency ID (the string groupId:artifactId) from all the
     * ancestors of the specified maven project
     *
     * @param mavenProject
     *            a maven project
     * @return the dependencies ID (the string groupId:artifactId) from all the
     *         ancestors
     */
    @SuppressWarnings("unchecked")
    private static final List<String> getAllParentDep(MavenProject mavenProject) {
        MavenProject parentProject = mavenProject.getParent();

        List<String> allParentDepId = new ArrayList<String>();
        ;
        if (parentProject != null) {
            List<Dependency> allParentDep = parentProject.getDependencies();
            for (Dependency dependency : allParentDep) {
                String groupId = dependency.getGroupId();
                String artifactId = dependency.getArtifactId();
                String version = dependency.getVersion();
                String depParId = groupId + ":" + artifactId + ":" + version;

                allParentDepId.add(depParId);
            }
        }

        return allParentDepId;
    }

    /**
     * Get the dependencies in the dependencyManagement section for each
     * ancestor of the specified maven project
     *
     * @param mavenProject
     *            a maven project
     * @return the dependencies in the dependencyManagement section for each
     *         ancestor. It returns a hashmap where the key is the maven project
     *         ancestor and the value the list of dependencies in the
     *         dependencyManagement section for the ancestor
     */
    private static final HashMap<MavenProject, List<Dependency>> getAllParentDepFromDepMgtByProject(
            MavenProject mavenProject) {
        return getAllParentDepFromDepMgtByProject(mavenProject, new ArrayList<String>());
    }

    @SuppressWarnings("unchecked")
    private static final HashMap<MavenProject, List<Dependency>> getAllParentDepFromDepMgtByProject(
            MavenProject mavenProject, List<String> allParentDepFromDepMgt) {
        HashMap<MavenProject, List<Dependency>> parentDepFromDepMgt = new HashMap<MavenProject, List<Dependency>>();

        if (mavenProject.hasParent()) {
            MavenProject parentProject = mavenProject.getParent();
            parentDepFromDepMgt.putAll(getAllParentDepFromDepMgtByProject(parentProject,
                    allParentDepFromDepMgt));
            DependencyManagement dependencyManagement = parentProject.getDependencyManagement();
            if (dependencyManagement != null) {
                List<Dependency> allDep = dependencyManagement.getDependencies();

                List<Dependency> newDep = getNewDep(allParentDepFromDepMgt, allDep);

                parentDepFromDepMgt.put(parentProject, newDep);
            }
        }

        return parentDepFromDepMgt;
    }

    private static final List<Dependency> getNewDep(List<String> allParentDepId,
            List<Dependency> allDep) {
        List<Dependency> newDep = new ArrayList<Dependency>();
        for (Dependency dependency : allDep) {
            String groupId = dependency.getGroupId();
            String artifactId = dependency.getArtifactId();
            String version = dependency.getVersion();
            String depParId = groupId + ":" + artifactId + ":" + version;

            if (!allParentDepId.contains(depParId)) {
                newDep.add(dependency);
                allParentDepId.add(depParId);
            }
        }
        return newDep;
    }

}
