/**
 * Raphael Diagram - A crossbrowser SVG/VML library creating diagrams - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.gwt.raphael.client.diagram.element;

import java.util.ArrayList;
import java.util.List;

import com.ebmwebsourcing.gwt.jquery.client.core.J4GUI;
import com.ebmwebsourcing.gwt.jquery.client.core.JqueryUI;
import com.ebmwebsourcing.gwt.raphael.client.core.Path;
import com.ebmwebsourcing.gwt.raphael.client.core.SVGElement;
import com.ebmwebsourcing.gwt.raphael.client.core.Text;
import com.ebmwebsourcing.gwt.raphael.client.diagram.DiagramPanel;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementDragListener;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementDragListenerAdapter;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementDragListenerBinder;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementDropListener;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementDropListenerBinder;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementListener;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementMouseListener;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementMouseListenerAdapter;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementMouseListenerBinder;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementResizableListener;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementResizableListenerAdapter;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementResizableListenerBinder;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementSortableListener;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.DiagramElementSortableListenerBinder;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.HasHideHandlers;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.HasShowHandlers;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.HideEvent;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.HideHandler;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.ShowEvent;
import com.ebmwebsourcing.gwt.raphael.client.diagram.event.ShowHandler;
import com.ebmwebsourcing.gwt.raphael.client.diagram.type.DiagramDefaultType;
import com.ebmwebsourcing.gwt.raphael.client.diagram.type.DiagramElementType;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.ContextMenuEvent;
import com.google.gwt.event.dom.client.ContextMenuHandler;
import com.google.gwt.event.dom.client.DoubleClickEvent;
import com.google.gwt.event.dom.client.DoubleClickHandler;
import com.google.gwt.event.dom.client.HasClickHandlers;
import com.google.gwt.event.dom.client.HasContextMenuHandlers;
import com.google.gwt.event.dom.client.HasDoubleClickHandlers;
import com.google.gwt.event.dom.client.HasMouseOutHandlers;
import com.google.gwt.event.dom.client.HasMouseOverHandlers;
import com.google.gwt.event.dom.client.MouseOutEvent;
import com.google.gwt.event.dom.client.MouseOutHandler;
import com.google.gwt.event.dom.client.MouseOverEvent;
import com.google.gwt.event.dom.client.MouseOverHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.ui.AbsolutePanel;

/**
 * 
 * @author nfleury
 * 
 */
public class DiagramElement extends AbsolutePanel implements HasHideHandlers,
		HasShowHandlers, HasMouseOutHandlers, HasMouseOverHandlers,
		HasContextMenuHandlers, HasDoubleClickHandlers, HasClickHandlers {

	private SVGElement svgElement;

	private J4GUI jqueryObject;

	private DiagramPanel diagramPanel;

	protected DiagramElementType diagramElementType;

	// As an element can be part of another element we keep the parent reference
	private DiagramElement parentElement;

	// Listeners
	private ArrayList<DiagramElementDragListener> dragListeners = new ArrayList<DiagramElementDragListener>();;
	private ArrayList<DiagramElementDropListener> dropListeners = new ArrayList<DiagramElementDropListener>();
	private ArrayList<DiagramElementMouseListener> mouseListeners = new ArrayList<DiagramElementMouseListener>();
	private ArrayList<DiagramElementSortableListener> sortableListerners = new ArrayList<DiagramElementSortableListener>();
	private ArrayList<DiagramElementResizableListener> resizableListerners = new ArrayList<DiagramElementResizableListener>();
	private ArrayList<DiagramElementListener> elementListeners = new ArrayList<DiagramElementListener>();

	// Default Mouse Listeners
	private DiagramElementMouseListener defaultMouseListener;

	// Context allows to save the state of the element at a specific moment
	protected DiagramElementContext context = new DiagramElementContext();

	private boolean isDraggable;

	private boolean isDroppable;

	private boolean isSortable;

	private boolean isResizable;

	protected boolean isLoaded;

	protected int width;
	protected int height;
	
	protected int x;
	protected int y;
	
	// boolean state of collections...
	private boolean iteratingDragListeners = false;
	private List<DiagramElementDragListener> dragListenersToRemove = new ArrayList<DiagramElementDragListener>();

	public DiagramElement(SVGElement svgElement) {
		this(svgElement, new DiagramDefaultType());
	}

	public DiagramElementType getDiagramElementType() {
		return diagramElementType;
	}

	public DiagramElement(SVGElement svgElement, DiagramElementType type) {

		this.svgElement = svgElement;

		this.diagramElementType = type;

		this.getElement().setId(svgElement.getId());

		this.addStyleName("zindexFront");

		// event sink
		sinkEvents(Event.ONCLICK);
		sinkEvents(Event.MOUSEEVENTS);
		sinkEvents(Event.ONCONTEXTMENU);
		sinkEvents(Event.ONDBLCLICK);
		
		//initialize width / heigth / X / Y
		this.setWidth(getSvgElement().getWidth());
		this.setHeight(getSvgElement().getHeight());
		this.setX(getSvgElement().getX());
		this.setY(getSvgElement().getY());
		
		
	}

	public boolean isLoaded() {
		return isLoaded;
	}

	@Override
	protected void onLoad() {

		JqueryUI jui = new JqueryUI();

		this.jqueryObject = jui.$("#" + this.getId());

//		 this.jqueryObject.css("border","1px solid green");

		this.getSvgElement().attr("id", this.getSvgElement().getId());


		// //////////
		// Event handling
		// ///////////////////

		// Bind mouse listeners
		new DiagramElementMouseListenerBinder(jqueryObject, mouseListeners);

		this.attachDefaultListeners();

		// /Load all effects as the DOM is now loaded
		if (isDraggable) {
			isDraggable();
		}

		if (isDroppable) {
			isDroppable();
		}

		if (isResizable) {
			isResizable();
		}

		if (isSortable) {
			isSortable();
		}


		int width;
		int height;
		
		if (svgElement instanceof Path || svgElement instanceof Text){
			width = this.getSvgElement().getWidth();
			height = this.getSvgElement().getHeight();
		}else {
			width = this.width;
			height = this.height;
		}
		
		
		//initialize width / heigth / X / Y
		this.setWidth(width);
		this.setHeight(height);
		this.setX(x);
		this.setY(y);
		
		
		// initialize some context stuffs
		this.context.setBorderColor(this.getSvgElement().attr("stroke"));
		this.context.setBackgroundColor(this.getSvgElement().attr("fill"));
		this.context.setWidth(width);
		this.context.setHeight(height);
		
		super.onLoad();
		this.isLoaded = true;
		
		refreshSVGsize();
		
		//As paths doesnt have an initial width and height
		if (svgElement instanceof Path || svgElement instanceof Text){
			this.setWidth(width);
			this.setHeight(height);			
		}

		
		// trigger the onLoad event
		for (DiagramElementListener listener : elementListeners) {
			listener.onLoad();
		}
	}
	
	
	
	
	/**
	 * Initializes some data that can only be initialized after the element have
	 * been rendered
	 * 
	 */
	private void refreshHTMLElementPosition() {

		if (isLoaded){
	
			this.context.setSize(width, height);
	
			this.setWidth(String.valueOf(width));
			this.setHeight(String.valueOf(height));
	
			this.context.setLastXPosition(this.getAbsoluteLeft());
			this.context.setLastYPosition(this.getAbsoluteTop());
			
		}		
		
	}

	public void setParentElement(DiagramElement parentElement) {
		this.parentElement = parentElement;
	}

	/**
	 * Return the parent element diagram element
	 * 
	 * @return
	 */
	public DiagramElement getParentElement() {
		return parentElement;
	}

	public void setDiagramPanel(DiagramPanel diagramPanel) {
		this.diagramPanel = diagramPanel;
	}

	public DiagramPanel getDiagramPanel() {
		return diagramPanel;
	}

	public int getX() {
		return this.svgElement.getX();
	}

	public int getY() {
		return this.svgElement.getY();
	}

	public void setX(int x) {

		String s = String.valueOf(x);
		this.x = x;
		
		if (isLoaded){
			this.jqueryObject.css("left", s);
			this.refreshSVGPosition();
		}
		 
		 
		 
	}

	public void setY(int y) {

		String s = String.valueOf(y);
		this.y = y;
		
		if (isLoaded){
			this.jqueryObject.css("top", s);
			this.refreshSVGPosition();
		}
		 
		 

	}

	public SVGElement getSvgElement() {
		return svgElement;
	}

	public void isDraggable(boolean draggable) {

		this.isDraggable = draggable;
		
		if (isDraggable && isLoaded) {
			isDraggable();
			
			this.getJqueryObject().destroyDraggable();
			
		}

	}

	public void isDroppable(boolean droppable) {

		this.isDroppable = droppable;

		if (isDroppable && isLoaded) {
			isDroppable();
		}

	}

	public void isSortable(boolean sortable) {

		this.isSortable = sortable;

		if (isSortable && isLoaded) {
			isSortable();
		}

	}

	public void isResizable(boolean resizable) {

		this.isResizable = resizable;

		if (isResizable && isLoaded) {
			isResizable();
		}

	}

	/**
	 * Makes current element draggable
	 */
	private void isDraggable() {
		new DiagramElementDragListenerBinder(this);
	}

	/**
	 * Makes current element droppable
	 */
	private void isDroppable() {

		new DiagramElementDropListenerBinder(this);

	}

	private void isSortable() {
		new DiagramElementSortableListenerBinder(this);
	}

	private void isResizable() {
		new DiagramElementResizableListenerBinder(this);
	}

	public ArrayList<DiagramElementDragListener> getDragListeners() {
		return dragListeners;
	}

	public ArrayList<DiagramElementDropListener> getDropListeners() {
		return dropListeners;
	}

	public ArrayList<DiagramElementResizableListener> getResizableListerners() {
		return resizableListerners;
	}

	public ArrayList<DiagramElementSortableListener> getSortableListerners() {
		return sortableListerners;
	}

	public ArrayList<DiagramElementMouseListener> getMouseListeners() {
		return mouseListeners;
	}
	
	public void refresh(){
		if (isLoaded){
			refreshSVGPosition();
			refreshHTMLElementPosition();			
		}
	}
	


	// reimpl to do whatever you want there
	protected void refreshSpecific(){
		
	}
	
	
	public void setBorderColor(String borderColor) {
		this.getSvgElement().attr("stroke", borderColor);
	}

	public void setBackgroundColor(String backgrounColor) {
		this.getSvgElement().attr("fill", backgrounColor);
	}

	public String getBorderColor() {
		return this.getSvgElement().attr("stroke");
	}

	public String getBackgroundColor() {
		return this.getSvgElement().attr("fill");
	}

	public String getOpacity() {
		return this.getSvgElement().attr("opacity");
	}

	public void setOpacity(String opacity) {
		this.getSvgElement().attr("opacity", opacity);
	}

	public void toFront() {
		this.getSvgElement().toFront();
	}

	public void toBack() {
		this.getSvgElement().toBack();
	}

	public void restorInitialBorderAndBackGroundColor() {
		this.setBackgroundColor(context.getBackgroundColor());
		this.setBorderColor(context.getBorderColor());
		this.setOpacity("1");
	}

	/**
	 * Resizes the svgElement when the HTML artifact is being resized
	 * 
	 * @param event
	 */
	protected void refreshSVGsize() {

		if (isLoaded){

			this.getSvgElement().setWidth(width);
			this.getSvgElement().setHeight(height);
			

			for (DiagramElementListener listener : elementListeners) {
				listener.onRefreshSVGSize();
			}
			
		}

	}

	/**
	 * Drags the svgElement along with the HTML artifact
	 * 
	 * @param event
	 */
	public void refreshSVGPosition() {

		if (isLoaded && diagramPanel!=null){
			svgElement.setX(diagramPanel.getRelativeX(this));
			svgElement.setY(diagramPanel.getRelativeY(this));

			for (DiagramElementListener listener : elementListeners) {
				listener.onRefreshSVGPosition();
			}			
		}

				
	}

	public void addDragListener(DiagramElementDragListener listener) {
		this.dragListeners.add(listener);
	}

	public void addDropListener(DiagramElementDropListener listener) {
		this.dropListeners.add(listener);
	}

	public void addMouseListener(DiagramElementMouseListener listener) {
		this.mouseListeners.add(listener);
	}

	public void addSortableListerner(DiagramElementSortableListener listener) {
		this.sortableListerners.add(listener);
	}

	public void addResizableListener(DiagramElementResizableListener listener) {
		this.resizableListerners.add(listener);
	}

	public void addDiagramElementListener(DiagramElementListener listener) {
		this.elementListeners.add(listener);
	}

	public void removeAllDragListeners() {
		this.dragListeners.clear();
	}

	public void removeAllDropListeners() {
		this.dropListeners.clear();
	}

	public void removeAllMouseListeners() {
		this.mouseListeners.clear();
	}

	public void removeAllSortableListeners() {
		this.sortableListerners.clear();
	}

	public void removeAllResizableListerners() {
		this.resizableListerners.clear();
	}

	public String getId() {

		if (this.svgElement != null) {
			return svgElement.getId();
		} else {
			return this.getElement().getId();
		}

	}

	public DiagramElementContext getContext() {
		return context;
	}

	public J4GUI getJqueryObject() {
		return jqueryObject;
	}

	/**
	 * Remove current diagramElement from DOM
	 */
	public void remove() {
		this.removeFromParent();
		this.getSvgElement().remove();

		for (DiagramElementListener listener : elementListeners) {
			listener.onRemove();
		}

	}

	public DiagramElementMouseListener getDefaultMouseListener() {

		return defaultMouseListener;
	}

	public void setDefaultMouseListener(
			DiagramElementMouseListener defaultMouseListener) {
		this.defaultMouseListener = defaultMouseListener;
	}

	@Override
	public boolean equals(Object obj) {

		DiagramElement elToCompare = (DiagramElement) obj;
		return this.getId().equals(elToCompare.getId());
	}

	public void removeDiagramDragListener(DiagramElementDragListener listener) {
		if (!iteratingDragListeners) {
			dragListeners.remove(listener);
		} else {
			dragListenersToRemove.add(listener);
		}
	}

	@Override
	public int hashCode() {
		return this.getId().hashCode();
	}



	public void setWidth(final int width) {
	
		this.setWidth(String.valueOf(width));
		this.width = width;

		if (isLoaded){
			this.refreshSVGsize();
		}
		
	}



	public void setHeight(final int height) {
		
		
		this.setHeight(String.valueOf(height));
		this.height = height;
		
		if (isLoaded){
			this.refreshSVGsize();
		}
		
	}

	public int getWidth() {
		return width;
	}

	public int getHeight() {
		return height;
	}

	protected void attachDefaultListeners() {

		this.addResizableListener(new DiagramElementResizableListenerAdapter() {

			@Override
			public void onStop(){
				
				DiagramElement.this.height = getOffsetHeight();
				DiagramElement.this.width = getOffsetWidth();
			
				refreshSVGsize();
				 DiagramElement.this.jqueryObject.css("border","none");
			}
			
			@Override
			public void onResize() {

				 DiagramElement.this.jqueryObject.css("border","1px dotted blue");
				//refreshSVGsize();

			}

		});

		// ////////////
		// Element is selected when it is being dragged or clicked
		// /////////////////

		// When we click on current element or drag it : this is the selected
		// element
		// the element is deselected on 'selectstop' event on diagram panel

		if (getDefaultMouseListener() != null) {

			this.addMouseListener(getDefaultMouseListener());

		} else {

			this
					.setDefaultMouseListener(new DiagramElementMouseListenerAdapter() {
						@Override
						public void onMouseEnter() {

						}
					});

		}

		this.addDragListener(new DiagramElementDragListenerAdapter() {

			@Override
			public void onDrag(DiagramElement diagramElement) {
				 DiagramElement.this.jqueryObject.css("border","1px dotted blue");
//				refreshSVGPosition();
				// isSelected();
				// update();refreshSVGPosition
//				DiagramElement.this.setX(Integer.valueOf(DiagramElement.this.getAbsoluteLeft()));
//				DiagramElement.this.setY(Integer.valueOf(DiagramElement.this.getAbsoluteTop()));
			}

			// When drag stop :
			// - initialize svg position
			// - set context position
			@Override
			public void onStop(DiagramElement diagramElement) {

				DiagramElement.this.jqueryObject.css("border","none");
				refreshSVGPosition();
				// update();

				getContext().setLastXPosition(getX());
				getContext().setLastYPosition(getY());

				// if this is a group element refresh the last positions for all
				// childs
				if (diagramElement instanceof DiagramGroupElement) {

					DiagramGroupElement group = (DiagramGroupElement) diagramElement;

					group.getContext().setLastXPosition(group.getX());
					group.getContext().setLastYPosition(group.getY());

					for (DiagramElement element : group.getChilds()) {

						element.getContext().setLastXPosition(element.getX());
						element.getContext().setLastYPosition(element.getY());
					}

				}

				// if this is a complex element refresh the last positions for
				// all childs
				// that are group elements
				if (diagramElement instanceof DiagramComplexElement) {

					DiagramComplexElement complexElement = (DiagramComplexElement) diagramElement;

					complexElement.getContext().setLastXPosition(
							complexElement.getX());
					complexElement.getContext().setLastYPosition(
							complexElement.getY());

					for (DiagramElementChildData element : complexElement
							.getDiagramElementChildren()) {

						if (element.getDiagramElement() instanceof DiagramGroupElement) {

							DiagramGroupElement group = (DiagramGroupElement) element
									.getDiagramElement();

							group.getContext().setLastXPosition(group.getX());
							group.getContext().setLastYPosition(group.getY());

							for (DiagramElement el : group.getChilds()) {

								el.getContext().setLastXPosition(el.getX());
								el.getContext().setLastYPosition(el.getY());
							}

						} else if (element.getDiagramElement() instanceof DiagramSortableElement) {

							DiagramSortableElement sortable = (DiagramSortableElement) element
									.getDiagramElement();

							sortable.getContext().setLastXPosition(
									sortable.getX());
							sortable.getContext().setLastYPosition(
									sortable.getY());

							for (DiagramElementChildData el : sortable
									.getDiagramElementChildren()) {

								el.getDiagramElement().getContext()
										.setLastXPosition(
												el.getDiagramElement().getX());
								el.getDiagramElement().getContext()
										.setLastYPosition(
												el.getDiagramElement().getY());

							}

						}

					}

				}
			}

		});
	}

	public final void hide() {
		hideSpecific();
		// notify
		HideEvent.fire(this);
	}

	protected void hideSpecific() {
		// hide me and my svg element
		this.setVisible(false);
		svgElement.hide();
	}

	public final void show() {
		showSpecific();
		// notify handlers
		ShowEvent.fire(this);

	}

	protected void showSpecific() {
		this.setVisible(true);
		svgElement.show();
	}

	public int getZIndex() {
		return Integer.parseInt(this.getElement().getStyle().getZIndex());
	}

	public void setZIndex(int zIndex) {
		getElement().getStyle().setZIndex(zIndex);
	}

	public HandlerRegistration addHideHandler(HideHandler handler) {
		return addHandler(handler, HideEvent.getType());
	}

	public HandlerRegistration addShowHandler(ShowHandler handler) {
		return addHandler(handler, ShowEvent.getType());
	}

	public HandlerRegistration addMouseOutHandler(MouseOutHandler handler) {
		return addHandler(handler, MouseOutEvent.getType());
	}

	public HandlerRegistration addMouseOverHandler(MouseOverHandler handler) {
		return addHandler(handler, MouseOverEvent.getType());
	}

	public HandlerRegistration addContextMenuHandler(ContextMenuHandler handler) {
		return addHandler(handler, ContextMenuEvent.getType());
	}

	public void fireOnDrag() {
		iteratingDragListeners = true;
		for (DiagramElementDragListener listener : dragListeners) {
			listener.onDrag(this);
		}
		iteratingDragListeners = false;
		for (DiagramElementDragListener listener : dragListenersToRemove) {
			dragListeners.remove(listener);
		}
		dragListenersToRemove.clear();

	}

	public void fireOnDragStop() {
		iteratingDragListeners = true;
		for (DiagramElementDragListener listener : dragListeners) {
			listener.onStop(this);
		}
		iteratingDragListeners = false;
		for (DiagramElementDragListener listener : dragListenersToRemove) {
			dragListeners.remove(listener);
		}
		dragListenersToRemove.clear();
	}

	public void fireOnDragStart() {
		iteratingDragListeners = true;
		for (DiagramElementDragListener listener : dragListeners) {
			listener.onStart(this);
		}
		iteratingDragListeners = false;
		for (DiagramElementDragListener listener : dragListenersToRemove) {
			dragListeners.remove(listener);
		}
		dragListenersToRemove.clear();

	}

	public HandlerRegistration addDoubleClickHandler(DoubleClickHandler arg0) {
		return addHandler(arg0, DoubleClickEvent.getType());
	}

	public HandlerRegistration addClickHandler(ClickHandler arg0) {
		return addHandler(arg0, ClickEvent.getType());
	}
	
	// OMG this is such a huge hack ! but since all components does not like deferred commands, we must do this !
	public boolean preferDeffered(){
		return true;
	}
	
	public boolean canBeResized(){
		return isResizable;
	}
	
}
