/**
 * BPMN Editor Domain - Domain Objects for BPMN standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard;

import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.collaboration.CollaborationBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.collaboration.LaneBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.collaboration.LaneSetBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.collaboration.PoolBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.common.FlowNodeBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.common.ItemDefinitionBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.common.MessageBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.common.MessageFlowBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.common.ParticipantBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.common.PartnerRoleBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.infrastructure.DefinitionsBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.infrastructure.ImportBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.infrastructure.schema.ElementBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.infrastructure.schema.SchemaBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.infrastructure.schema.SchemaType;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.ProcessBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.activity.TaskBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.event.EndEventBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.event.StartEventBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.gateway.GatewayBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.gateway.SequenceFlowBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.service.InterfaceBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.service.OperationBean;

public abstract class ModelVisitor {

	private DefinitionsBean defs; 
	
	public ModelVisitor(DefinitionsBean defs) {
		this.defs = defs;
	}
	
	public void visitDefinitionsByLists()throws Exception{
		this.visitDefinitions(defs);
		//IMPORTS
		for(ImportBean ib : defs.getImports()){
			this.visitImport(ib);
			
			//SCHEMA
			SchemaBean schema = ib.getSchema();
			if(schema!=null){
				this.visitSchema(schema);
				
				if(schema.getTypes()!=null){
					for(SchemaType st : schema.getTypes()){
						this.visitSchemaType(st);
					}
					
					this.finalizeTypes(schema);
				}
				
				if(schema.getElements()!=null){
					for(ElementBean eb : schema.getElements()){
						this.visitElementBean(eb);
					}
				}
			}
		}
		
		//MESSAGES
		for(MessageBean msgBean : defs.getMessages()){
			this.visitMessage(msgBean);
			
			ItemDefinitionBean idb = msgBean.getItemDefinition();
			if(idb!=null){
				this.visitItemDefinition(idb);
				
				if(idb.getElement()!=null){
					this.visitItemDefinitionElement(idb.getElement(),idb);
				}
			}
		}
		
		//PARTENR ROlES
		for(PartnerRoleBean pr : defs.getPartnerRoles()){
			this.visitPartnerRole(pr);
		}
		
		//INTERFACES
		for(InterfaceBean itf : defs.getInterfaces()){
			this.visitInterface(itf);
			
			//OPERATIONS
			if(itf.getOperations()!=null){
				for(OperationBean op : itf.getOperations()){
					this.visitOperation(op);
					
					MessageBean msg = op.getMessageIn();
					this.visitOperationMessageIn(msg,op);
					
					msg = op.getMessageOut();
					if(msg!=null){
						this.visitOperationMessageOut(msg,op);
					}
					
					if(op.getErrorMessages()!=null){
						for(MessageBean mb : op.getErrorMessages()){
							this.visitOperationErrorMessage(mb,op);
						}
					}
				}
			}
		}
		
		//PROCESSES
		for(ProcessBean proc : defs.getProcesses()){
			this.visitProcess(proc);
			
			if(proc.getLaneSets()!=null){
				for(LaneSetBean laneSet : proc.getLaneSets()){
					this.visitLaneSet(laneSet);
					
					parseLaneSet(laneSet);
					
				}
			}
		}
		
		//COLLABORATIONS
		for(CollaborationBean collab : defs.getCollaborations()){
			this.visitCollaboration(collab);
			
			//POOLS
			if(collab.getPools()!=null){
				for(PoolBean pool : collab.getPools()){
					this.visitPool(pool);
					
					ParticipantBean pb = pool.getParticipant();
					if(pb!=null){
						this.visitParticipant(pb);
						
						if(pb.getPartnerRole()!=null){
							this.visitParticipantRole(pb.getPartnerRole());
						}
					}
					
					if(pool.getProcess()!=null){
						this.visitPoolProcess(pool.getProcess());
					}
					
					if(pool.getInterface()!=null){
						this.visitPoolInterface(pool.getInterface());
					}
				}
			}
			
			//MESSAGE FLOWS
			if(collab.getMessageFlows()!=null){
				for(MessageFlowBean mfb : collab.getMessageFlows()){
					this.visitMessageFlow(mfb);
					
					MessageBean mb = mfb.getMessage();
					if(mb!=null){
						this.visitMessageFlowMessage(mb);
					}
					
					FlowNodeBean node = mfb.getSource();
					if(node!=null){
						this.visitMessageFlowSource(node);
					}
					
					node = mfb.getTarget(); 
					if(node!=null){
						this.visitMessageFlowTarget(node);
					}
				}
			}
			
		}
	}
	
	public void visitDefinitionsByPools()throws Exception{
		this.visitDefinitions(defs);
		//IMPORTS
		for(ImportBean ib : defs.getImports()){
			this.visitImport(ib);
			
			//SCHEMA
			SchemaBean schema = ib.getSchema();
			if(schema!=null){
				this.visitSchema(schema);
				
				if(schema.getTypes()!=null){
					for(SchemaType st : schema.getTypes()){
						this.visitSchemaType(st);
					}
					
					this.finalizeTypes(schema);
				}
				
				if(schema.getElements()!=null){
					for(ElementBean eb : schema.getElements()){
						this.visitElementBean(eb);
					}
				}
			}
		}
		
		//MESSAGES
		for(MessageBean msgBean : defs.getMessages()){
			this.visitMessage(msgBean);
			
			ItemDefinitionBean idb = msgBean.getItemDefinition();
			if(idb!=null){
				this.visitItemDefinition(idb);
				
				if(idb.getElement()!=null){
					this.visitItemDefinitionElement(idb.getElement(),idb);
				}
			}
		}
		
		//PARTENR ROlES
		for(PartnerRoleBean pr : defs.getPartnerRoles()){
			this.visitPartnerRole(pr);
		}
		
		//INTERFACES
		for(InterfaceBean itf : defs.getInterfaces()){
			this.visitInterface(itf);
			
			//OPERATIONS
			if(itf.getOperations()!=null){
				for(OperationBean op : itf.getOperations()){
					this.visitOperation(op);
					
					MessageBean msg = op.getMessageIn();
					this.visitOperationMessageIn(msg,op);
					
					msg = op.getMessageOut();
					if(msg!=null){
						this.visitOperationMessageOut(msg,op);
					}
					
					if(op.getErrorMessages()!=null){
						for(MessageBean mb : op.getErrorMessages()){
							this.visitOperationErrorMessage(mb,op);
						}
					}
				}
			}
		}
		
		//COLLABORATIONS
		for(CollaborationBean collab : defs.getCollaborations()){
			this.visitCollaboration(collab);
			
			//POOLS
			if(collab.getPools()!=null){
				for(PoolBean pool : collab.getPools()){
					this.visitPool(pool);
					
					ParticipantBean pb = pool.getParticipant();
					if(pb!=null){
						this.visitParticipant(pb);
						if(pb.getPartnerRole()!=null){
							this.visitParticipantRole(pb.getPartnerRole());
						}
					}
					
					if(pool.getProcess()!=null){
						ProcessBean proc = pool.getProcess();
						this.visitPoolProcess(proc);
						
						if(proc.getLaneSets()!=null){
							for(LaneSetBean laneSet : proc.getLaneSets()){
								this.visitLaneSet(laneSet);
								
								parseLaneSet(laneSet);
								
							}
						}
					}
					
					if(pool.getInterface()!=null){
						this.visitPoolInterface(pool.getInterface());
					}
				}
			}
			
			//MESSAGE FLOWS
			if(collab.getMessageFlows()!=null){
				for(MessageFlowBean mfb : collab.getMessageFlows()){
					this.visitMessageFlow(mfb);
					
					MessageBean mb = mfb.getMessage();
					if(mb!=null){
						this.visitMessageFlowMessage(mb);
					}
					
					FlowNodeBean node = mfb.getSource();
					if(node!=null){
						this.visitMessageFlowSource(node);
					}
					
					node = mfb.getTarget(); 
					if(node!=null){
						this.visitMessageFlowTarget(node);
					}
				}
			}
		}
		
		//PROCESSES
		for(ProcessBean proc : defs.getProcesses()){
			this.visitProcess(proc);
		}
	}

	public void parseLaneSet(LaneSetBean laneSet) {
		//LANES
		if(laneSet.getLanes()!=null){
			for(LaneBean lane: laneSet.getLanes()){
				this.visitLane(lane);
				
				LaneSetBean lsb = lane.getChildLaneSet();
				if(lsb!=null){
					this.visitChildLaneSet(lsb);
				}
				
				if(lane.getStartEvents()!=null){
					for(StartEventBean seb : lane.getStartEvents()){
						this.visitStartEvent(seb);
					}
				}
				
				if(lane.getTasks()!=null){
					for(TaskBean tb : lane.getTasks()){
						this.visitTask(tb);
					}
				}
				
				if(lane.getEndEvents()!=null){
					for(EndEventBean eeb : lane.getEndEvents()){
						this.visitEndEvent(eeb);
					}
				}
				if(lane.getGateways()!=null){
					for(GatewayBean gb : lane.getGateways()){
						this.visitGateway(gb);
					}
				}
				if(lane.getSequenceFlows()!=null){
					for(SequenceFlowBean sfb : lane.getSequenceFlows()){
						this.visitSequenceFlow(sfb);
					}
				}
			}
		}
	}

	public void visitChildLaneSet(LaneSetBean lsb) {}
	public void finalizeTypes(SchemaBean schema) {}
	public void visitDefinitions(DefinitionsBean defs) throws Exception {}
	public void visitPoolInterface(InterfaceBean interface1) {}
	public void visitPoolProcess(ProcessBean process) {}
	public void visitParticipantRole(PartnerRoleBean partnerRole) {}
	public void visitParticipant(ParticipantBean participant) {}
	public void visitPool(PoolBean pool) {}
	public void visitMessageFlowSource(FlowNodeBean node) throws Exception{}
	public void visitMessageFlowTarget(FlowNodeBean node) throws Exception{}
	public void visitMessageFlowMessage(MessageBean mb) throws Exception{}
	public void visitMessageFlow(MessageFlowBean mfb) {}
	public void visitCollaboration(CollaborationBean collab) {}
	public void visitSequenceFlow(SequenceFlowBean sfb) {}
	public void visitGateway(GatewayBean gb) {}
	public void visitEndEvent(EndEventBean eeb) {}
	public void visitTask(TaskBean tb) {}
	public void visitStartEvent(StartEventBean seb) {}
	public void visitLane(LaneBean lane) {}
	public void visitLaneSet(LaneSetBean laneSet) {}
	public void visitProcess(ProcessBean proc) {}
	public void visitOperationMessageIn(MessageBean msg,OperationBean opBean) {}
	public void visitOperationMessageOut(MessageBean msg,OperationBean opBean) {}
	public void visitOperationErrorMessage(MessageBean msg,OperationBean opBean) {}
	public void visitOperation(OperationBean op) {}
	public void visitInterface(InterfaceBean itf) {}
	public void visitPartnerRole(PartnerRoleBean pr) {}
	public void visitItemDefinitionElement(ElementBean element, ItemDefinitionBean itemDefinition) {}
	public void visitItemDefinition(ItemDefinitionBean itemDefinition) {}
	public void visitMessage(MessageBean msgBean) {}
	public void visitElementBean(ElementBean eb) {}
	public void visitSchemaType(SchemaType st) {}
	public void visitImport(ImportBean ib) throws Exception {}
	public void visitSchema(SchemaBean sb) throws Exception {}
}
