/**
 * BPMN Editor Domain - Domain Objects for BPMN standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.bpmneditor.business.service.bpmn2.to.serverToClient;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;

import org.apache.commons.lang.NotImplementedException;
import org.ow2.easywsdl.schema.DefaultSchemaImpl;
import org.ow2.easywsdl.schema.api.ComplexType;
import org.ow2.easywsdl.schema.api.Element;
import org.ow2.easywsdl.schema.api.Enumeration;
import org.ow2.easywsdl.schema.api.Restriction;
import org.ow2.easywsdl.schema.api.Schema;
import org.ow2.easywsdl.schema.api.SimpleType;
import org.ow2.easywsdl.schema.api.Type;

import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.Constants.BasicTypes;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.Constants.ItemKind;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.Constants.ProcessTypes;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.collaboration.CollaborationBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.collaboration.LaneBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.collaboration.LaneSetBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.collaboration.PoolBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.common.FlowElementBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.common.ItemDefinitionBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.common.MessageBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.common.MessageFlowBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.common.ParticipantBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.common.PartnerRoleBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.foundation.BPMNElementBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.infrastructure.DefinitionsBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.infrastructure.ImportBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.infrastructure.schema.BasicType;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.infrastructure.schema.ElementBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.infrastructure.schema.RestrictionBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.infrastructure.schema.SchemaBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.infrastructure.schema.SchemaType;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.infrastructure.schema.SequenceBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.ProcessBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.activity.ReceiveTaskBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.activity.SendTaskBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.activity.ServiceTaskBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.activity.TaskBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.data.AssignementBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.data.DataInputAssociationBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.data.DataOutputAssociationBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.data.ExpressionBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.event.EndEventBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.event.StartEventBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.event.definition.EventDefinitionBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.event.definition.MessageEventDefinitionBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.gateway.ExclusiveGatewayBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.gateway.ParallelGatewayBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.process.gateway.SequenceFlowBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.service.InterfaceBean;
import com.ebmwebsourcing.bpmneditor.business.domain.bpmn2.to.standard.service.OperationBean;
import com.ebmwebsourcing.bpmneditor.business.service.bpmn2.to.serverToClient.xmlParsing.GraphicalInformation;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.BPMNElement;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.BPMNException;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.collaboration.Collaboration;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.collaboration.Lane;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.collaboration.LaneSet;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.collaboration.Pool;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.common.ItemDefinition;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.common.Message;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.common.MessageFlow;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.common.Participant;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.common.PartnerRole;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.infrastructure.Definitions;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.infrastructure.Import;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.process.Process;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.process.activity.ReceiveTask;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.process.activity.SendTask;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.process.activity.ServiceTask;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.process.activity.Task;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.process.data.Assignement;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.process.data.DataInputAssociation;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.process.data.DataOutputAssociation;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.process.data.Expression;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.process.event.EndEvent;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.process.event.StartEvent;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.process.event.definition.EventDefinition;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.process.event.definition.MessageEventDefinition;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.process.gateway.ExclusiveGateway;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.process.gateway.Gateway;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.process.gateway.ParallelGateway;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.process.gateway.SequenceFlow;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.service.Interface;
import com.ebmwebsourcing.easybpmn.model.bpmn.api.standard.service.Operation;
import com.ebmwebsourcing.easybpmn.model.bpmn.jaxb.TItemKind;
import com.ebmwebsourcing.easybpmn.model.bpmn.jaxb.TProcessType;


public class ServerToClient {
	
	private static DefinitionsBean defBean;
	private static GraphicalInformation info;
	
	public static DefinitionsBean adapt(Definitions def, String xmlPath) throws BPMNException{
		DefinitionsBean result = new DefinitionsBean(def.getId());
		defBean = result;
		
		info = new GraphicalInformation(def.getId(),xmlPath);
		
		result.setTargetNamespace(def.getTargetNamespace());
		if(def.getDocumentBaseURI()!=null){
			result.setDocumentBaseURI(def.getDocumentBaseURI().toString());
		}
		result.setExpressionLanguage(def.getExpressionLanguage());
		setDocumentation(result, def);
		result.setTypeLanguage(def.getTypeLanguage());
		
		adaptImports(result, def.getImports());
		
		adaptMessages(result,def.getMessages());
		adaptPartnerRoles(result,def.getPartnerRoles());
		adaptInterfaces(result,def.getInterfaces());
		adaptProcesses(result,def.getProcesss());
		adaptCollaborations(result,def.getCollaborations());
		
		return result;
	}

	private static List<CollaborationBean> adaptCollaborations(DefinitionsBean def, List<Collaboration> collaborations) throws BPMNException {
		List<CollaborationBean> result = null;
		if(collaborations!=null){
			result = new ArrayList<CollaborationBean>();
			for(Collaboration collab : collaborations){
				CollaborationBean cBean = new CollaborationBean(collab.getId());
				def.addCollaboration(cBean);
				cBean.setName(collab.getId());
				setDocumentation(cBean, collab);
				
				if(collab.getPools()!=null){
					for(Pool p : collab.getPools()){
						adaptPool(p,cBean,def);
					}
				}
				
				if(collab.getMessageFlows()!=null){
					for(MessageFlow mf : collab.getMessageFlows()){
						adaptMessageFlow(mf,cBean);
					}
				}
			}
		}
		return result;
	}

	private static void adaptMessageFlow(MessageFlow mf, CollaborationBean cBean) {
		MessageFlowBean mfBean = new MessageFlowBean(mf.getId());
		mfBean.setName(mf.getName());
		setDocumentation(mfBean, mf);
		if(mfBean.getMessage()!=null){
			mfBean.setMessage(findMessage(cBean,mf.getMessage().getId()));
		}
		if(mf.getSource()!=null){
			mfBean.setSource(cBean.getFlowNode(mf.getSource().getId()));
		}
		if(mf.getTarget()!=null){
			mfBean.setTarget(cBean.getFlowNode(mf.getTarget().getId()));
		}
		
		cBean.addMessageFlow(mfBean);
		
		info.setGraphicalInfo(mfBean);
	}

	private static MessageBean findMessage(CollaborationBean cBean, String msgId) {
		for(PoolBean p : cBean.getPools()){
			if(p.getInterface().getOperations()!=null){
				for(OperationBean op : p.getInterface().getOperations()){
					if(op.getMessageIn().getId().equals(msgId)){
						return op.getMessageIn();
					}
					if(op.getMessageOut().getId().equals(msgId)){
						return op.getMessageOut();
					}
				}
			}
		}
		return null;
	}

	private static void adaptPool(Pool p, CollaborationBean cBean, DefinitionsBean def) throws BPMNException {
		PoolBean pBean = new PoolBean(p.getId());
		cBean.addPool(pBean);
		pBean.setName(p.getName());
		setDocumentation(pBean, p);
		Process process = p.getProcess();
		if(process!=null){
			pBean.setProcess(def.getProcessBeanById(process.getId()));
		}
		Interface itf = p.getInterface();
		if(itf!=null){
			pBean.setInterface(def.getInterfaceBeanById(itf.getId()));
		}
		Participant participant = p.getParticipant(); 
		if(participant!=null){
			pBean.setParticipant(adaptParticipant(participant,def));
		}
		
		info.setGraphicalInfo(pBean);
	}

	private static ParticipantBean adaptParticipant(Participant participant, DefinitionsBean def) {
		ParticipantBean result = new ParticipantBean(participant.getId());
		result.setPartnerRole(def.getPartnerRoleBeanById(participant.getPartnerRole().getId()));
		setDocumentation(result, participant);
		return result;
	}

	private static List<ProcessBean> adaptProcesses(DefinitionsBean def, List<Process> processes) {
		List<ProcessBean> result  = null;
		if(processes!=null){
			result  = new ArrayList<ProcessBean>();
			for(Process p : processes){
				ProcessBean pBean = new ProcessBean(p.getId());
				setDocumentation(pBean, p);
				pBean.setName(p.getName());
				pBean.setType(retrieveProcessType(p.getProcessType()));
				
				if(p.getLaneSets()!=null){
					Map<String, FlowElementBean> elements = new HashMap<String, FlowElementBean>();
					for(LaneSet ls  : p.getLaneSets()){
						pBean.addLaneSet(adaptLaneSet(ls,elements));
					}
				}
				
				def.addProcess(pBean);
				result.add(pBean);
			}
		}
		return result;
	}

	private static LaneSetBean adaptLaneSet(LaneSet ls, Map<String, FlowElementBean> elements) {
		LaneSetBean lsBean = null;
		if(ls!=null){
			lsBean = new LaneSetBean(ls.getId());
			setDocumentation(lsBean, ls);
			if(ls.getLanes()!=null){
				for(Lane l : ls.getLanes()){
					lsBean.addLane(adaptLane(l,elements));
				}
			}
		}
		return lsBean;
	}

	private static LaneBean adaptLane(Lane l, Map<String, FlowElementBean> elements) {
		LaneBean lBean = null;
		if(l!=null){
			lBean = new LaneBean(l.getId());
			lBean.setName(l.getName());
			setDocumentation(lBean, l);
			lBean.setChildLaneSet(adaptLaneSet(l.getChildLaneSet(),elements));
			fillLane(lBean,l,elements);
			
			info.setGraphicalInfo(lBean);
		}
		return lBean;
	}

	private static void fillLane(LaneBean lBean, Lane l, Map<String, FlowElementBean> elements) {
		//START EVENTS
		List<StartEvent> startEvents = l.getStartEvents();
		if(startEvents!=null){
			for(StartEvent se : startEvents){
				StartEventBean seBean = new StartEventBean(se.getId());
				seBean.setName(se.getName());
				setDocumentation(seBean, se);
				if(se.getTriggers()!=null){
					for(EventDefinition ed : se.getTriggers()){
						EventDefinitionBean edb  = buildEventDefintion(ed);
						setDocumentation(edb, ed);
						seBean.addTrigger(edb);
					}
				}
				lBean.addStartEvent(seBean);
				elements.put(seBean.getId(), seBean);
				
				info.setGraphicalInfo(seBean);
			}
		}
		//TASKS
		if(l.getTasks()!=null){
			for(Task t : l.getTasks()){
				if(t instanceof ReceiveTask){
					adaptReceiveTask(lBean,(ReceiveTask) t,elements);
				}
				else if(t instanceof SendTask){
					adaptSendtask(lBean,(SendTask) t,elements);
				}
				else if(t instanceof ServiceTask){
					adaptServiceTask(lBean,(ServiceTask) t,elements);
				}
				else if(t instanceof Task){
					adaptSimpleTask(lBean,t,elements);
				}
				else{
					throw new NotImplementedException(t.getClass()+" is not supported as a Task.");
				}
			}
		}
		//GATEWAYS
		List<ExclusiveGateway> egList = new ArrayList<ExclusiveGateway>();
		if(l.getGateways()!=null){
			for(Gateway g : l.getGateways()){
				if(g instanceof ParallelGateway){
					adaptParallelGateway(lBean,(ParallelGateway) g,elements);
				}
				else if(g instanceof ExclusiveGateway){
					egList.add((ExclusiveGateway) g);
				}
				else{
					throw new NotImplementedException(g.getClass()+" is not supported yet.");
				}
			}
		}
		//END EVENTS
		if(l.getEndEvents()!=null){
			for(EndEvent ee : l.getEndEvents()){
				EndEventBean eeb = new EndEventBean(ee.getId());
				eeb.setName(ee.getName());
				setDocumentation(eeb, ee);
				if(ee.getResults()!=null){
					for(EventDefinition ed : ee.getResults()){
						EventDefinitionBean edb = buildEventDefintion(ed);
						setDocumentation(edb, ed);
						eeb.addResult(edb);
					}
				}
				lBean.addEndEvent(eeb);
				elements.put(eeb.getId(), eeb);
				
				info.setGraphicalInfo(eeb);
			}
		}
		
		
		for(ExclusiveGateway eg : egList){
			adaptExclusiveGateway(lBean,eg,elements);
		}
		
		
		//SEQUENCE FLOWS
		if(l.getSequenceFlows()!=null){
			for(SequenceFlow sf : l.getSequenceFlows()){
				SequenceFlowBean sfb = new SequenceFlowBean(sf.getId());
				sfb.setName(sf.getName());
				if(sf.getExpression()!=null){
					sfb.setExpression(adaptExpression(sf.getExpression()));
				}
				if(elements.get(sf.getSourceNode().getId()).getClass().equals(TaskBean.class)){
					updateTask((TaskBean)elements.get(sf.getSourceNode().getId()),(Task)sf.getSourceNode(),elements);
				}
				if(elements.get(sf.getTargetNode().getId()).getClass().equals(TaskBean.class)){
					updateTask((TaskBean)elements.get(sf.getTargetNode().getId()), (Task)sf.getTargetNode(), elements);
				}
				sfb.setSourceNode(elements.get(sf.getSourceNode().getId())); 
				sfb.setTargetNode(elements.get(sf.getTargetNode().getId()));
				lBean.addSequenceFlow(sfb);
				
				info.setGraphicalInfo(sfb);
			}
		}
	}
	


	private static void updateTask(TaskBean taskBean, Task t, Map<String,FlowElementBean> elements) {
		if(taskBean instanceof ReceiveTaskBean){
			ReceiveTaskBean rtb = (ReceiveTaskBean) taskBean;
			ReceiveTask rt = (ReceiveTask) t;
			rtb.setInstanciate(rt.isInstanciate());
			if(rt.getOperation()!=null){
				rtb.setOperation(adaptOperation(rt.getOperation()));
			}
			if(rt.getMessage()!=null){
				rtb.setMessage(adaptMessage(rt.getMessage()));
			}
			elements.put(rtb.getId(), rtb);
		}
		else if(taskBean instanceof SendTaskBean){
			SendTaskBean stb = (SendTaskBean) taskBean;
			SendTask st = (SendTask) t;
			if(st.getOperation()!=null){
				stb.setOperation(adaptOperation(st.getOperation()));
			}
			if(st.getMessage()!=null){
				stb.setMessage(adaptMessage(st.getMessage()));
			}
			elements.put(stb.getId(), stb);
		}
		else if(taskBean instanceof ServiceTaskBean){
			ServiceTaskBean stb = (ServiceTaskBean) taskBean;
			ServiceTask st = (ServiceTask) t;
			stb.setServiceName(st.getServiceName().getLocalPart());
			if(st.getOperation()!=null){
				stb.setOperation(adaptOperation(st.getOperation()));
			}
			elements.put(stb.getId(), stb);
		}
	}

	private static void adaptExclusiveGateway(LaneBean lBean, ExclusiveGateway eg, Map<String, FlowElementBean> elements) {
		ExclusiveGatewayBean egb = new ExclusiveGatewayBean(eg.getId());
		egb.setName(eg.getName());
		setDocumentation(egb, eg);
		if(eg.getDefault()!=null){
			egb.setDefaultElement(elements.get(eg.getDefault().getId()));
		}
		lBean.addGateway(egb);
		elements.put(egb.getId(), egb);
		
		info.setGraphicalInfo(egb);
	}

	private static void adaptParallelGateway(LaneBean laneBean, ParallelGateway pg, Map<String, FlowElementBean> elements) {
		ParallelGatewayBean pgb = new ParallelGatewayBean(pg.getId());
		pgb.setName(pg.getName());
		setDocumentation(pgb, pg);
		laneBean.addGateway(pgb);
		elements.put(pgb.getId(), pgb);
		
		info.setGraphicalInfo(pgb);
	}
	
	private static void adaptSimpleTask(LaneBean lBean, Task t, Map<String, FlowElementBean> elements) {
		TaskBean tb = new TaskBean(t.getId());
		tb.setName(t.getName());
		setDocumentation(tb, t);
		addDataAssociations(tb, t);
		lBean.addTask(tb);
		elements.put(tb.getId(), tb);
		
		info.setGraphicalInfo(tb);
	}

	private static void adaptServiceTask(LaneBean lBean, ServiceTask st, Map<String, FlowElementBean> elements) {
		ServiceTaskBean stb = new ServiceTaskBean(st.getId());
		stb.setName(st.getName());
		setDocumentation(stb, st);
		stb.setServiceName(st.getServiceName().getLocalPart());
		if(st.getOperation()!=null){
			stb.setOperation(adaptOperation(st.getOperation()));
		}
		addDataAssociations(stb, st);
		
		lBean.addTask(stb);
		elements.put(stb.getId(), stb);
		
		info.setGraphicalInfo(stb);
	}

	private static void adaptSendtask(LaneBean lBean, SendTask st, Map<String, FlowElementBean> elements) {
		SendTaskBean stb = new SendTaskBean(st.getId());
		stb.setName(st.getName());
		setDocumentation(stb, st);
		if(st.getOperation()!=null){
			stb.setOperation(adaptOperation(st.getOperation()));
		}
		if(st.getMessage()!=null){
			stb.setMessage(adaptMessage(st.getMessage()));
		}
		addDataAssociations(stb,st);
		
		lBean.addTask(stb);
		elements.put(stb.getId(), stb);
		
		info.setGraphicalInfo(stb);
	}

	private static void adaptReceiveTask(LaneBean lBean, ReceiveTask rt, Map<String, FlowElementBean> elements) {
		ReceiveTaskBean rtb = new ReceiveTaskBean(rt.getId()); 
		rtb.setName(rt.getName());
		rtb.setInstanciate(rt.isInstanciate());
		setDocumentation(rtb, rt);
		if(rt.getOperation()!=null){
			rtb.setOperation(adaptOperation(rt.getOperation()));
		}
		if(rt.getMessage()!=null){
			rtb.setMessage(adaptMessage(rt.getMessage()));
		}
		addDataAssociations(rtb,rt);
		
		lBean.addTask(rtb);
		elements.put(rtb.getId(), rtb);
		
		info.setGraphicalInfo(rtb);
	}

	private static void addDataAssociations(TaskBean tb, Task t) {
		if(t.getDataInputAssociations()!=null){
			for(DataInputAssociation dia : t.getDataInputAssociations()){
				DataInputAssociationBean diab = new DataInputAssociationBean(dia.getId());
				setDocumentation(diab, dia);
				if(dia.getAssignements()!=null){
					for(Assignement a : dia.getAssignements()){
						diab.addAssignement(adaptAssignement(a));
					}
				}
				tb.addDataInputAssociation(diab);
			}
		}
		if(t.getDataOutputAssociations()!=null){
			for(DataOutputAssociation doa : t.getDataOutputAssociations()){
				DataOutputAssociationBean doab = new DataOutputAssociationBean(doa.getId());
				setDocumentation(doab, doa);
				if(doa.getAssignements()!=null){
					for(Assignement a : doa.getAssignements()){
						doab.addAssignement(adaptAssignement(a));
					}
				}
				tb.addDataOutputAssociation(doab);
			}
		}
	}

	private static AssignementBean adaptAssignement(Assignement a) {
		AssignementBean ab = null;
		if(a!=null){
			ab = new AssignementBean(a.getId());
			ab.setLanguage(a.getLanguage());
			setDocumentation(ab, a);
			ab.setFrom(adaptExpression(a.getFrom()));
			ab.setTo(adaptExpression(a.getTo()));
		}
		return ab;
	}

	private static ExpressionBean adaptExpression(Expression exp) {
		ExpressionBean expb = new ExpressionBean(exp.getId());
		expb.setContent(exp.getContent());
		setDocumentation(expb, exp);
		return expb;
	}

	private static List<InterfaceBean> adaptInterfaces(DefinitionsBean def, List<Interface> interfaces) {
		List<InterfaceBean> result = null;
		if(interfaces!=null){
			result = new ArrayList<InterfaceBean>();
			for(Interface itf : interfaces){
				InterfaceBean itfBean = new InterfaceBean(itf.getId());
				setDocumentation(itfBean, itf);
				itfBean.setName(itf.getName());
				
				if(itf.getOperations()!=null){
					for(Operation op : itf.getOperations()){
						itfBean.addOperation(adaptOperation(op));
					}
				}
				
				def.addInterface(itfBean);
				result.add(itfBean);
			}
		}
		return result;
	}

	private static OperationBean adaptOperation(Operation op) {
		OperationBean opBean = new OperationBean(op.getId());
		setDocumentation(opBean, op);
		opBean.setName(op.getName());
		opBean.setMessageIn(adaptMessage(op.getMessageIn()));
		if(op.getMessageOut()!=null){
			opBean.setMessageOut(adaptMessage(op.getMessageOut()));
		}
		if(op.getErrorMessages()!=null){
			for(Message errorMessage : op.getErrorMessages()){
				opBean.addErrorMessage(adaptMessage(errorMessage));
			}
		}
		return opBean;
	}

	private static List<PartnerRoleBean> adaptPartnerRoles(DefinitionsBean def, List<PartnerRole> partnerRoles) {
		List<PartnerRoleBean> result = null;
		if(partnerRoles!=null){
			result = new ArrayList<PartnerRoleBean>();
			for(PartnerRole pr : partnerRoles){
				PartnerRoleBean prb = new PartnerRoleBean(pr.getId());
				setDocumentation(prb, pr);
				prb.setName(pr.getName());
				
				def.addPartnerRole(prb);
				result.add(prb);
			}
		}
		return result;
	}

	private static List<MessageBean> adaptMessages(DefinitionsBean def, List<Message> messages) {
		List<MessageBean> result = null;
		if(messages!=null){
			result = new ArrayList<MessageBean>();
			for(Message msg : messages){
				MessageBean msgBean = adaptMessage(msg);
				def.addMessage(msgBean);
				result.add(msgBean);
			}
		}
		return result;
	}

	private static MessageBean adaptMessage(Message msg){
		MessageBean msgBean = new MessageBean(msg.getId());
		msgBean.setName(msg.getName());
		setDocumentation(msgBean, msg);
		msgBean.setItemDefinition(adaptItemDefinition(msg.getItemDefinition()));
		return msgBean;
	}

	private static ItemDefinitionBean adaptItemDefinition(ItemDefinition itemDefinition) {
		ItemDefinitionBean result = null;
		if(itemDefinition!=null){
			result = new ItemDefinitionBean(itemDefinition.getId());
			setDocumentation(result, itemDefinition);
			result.setCollection(itemDefinition.isCollection());
			result.setItemKind(adaptItemKind(itemDefinition.getItemKind()));
			
			if(itemDefinition.getElement()!=null){
//				String namespace = itemDefinition.getElement().getQName().getNamespaceURI();
//				for(ImportBean iBean : defBean.getImports()){
//					if(iBean.getSchema().getTargetNamespace().equals(namespace)){
//						result.setElement(iBean.getSchema().getElement(itemDefinition.getElement().getQName().getLocalPart()));
//					}
//				}
				
				result.setElement(new ElementBean(itemDefinition.getElement().getQName().getLocalPart(), itemDefinition.getElement().getQName().getNamespaceURI()));
			}
		}
		return result;
	}

	private static ItemKind adaptItemKind(TItemKind itemKind) {
		ItemKind result = null;
		if(itemKind!=null){
			result = retrieveItemKind(itemKind);
		}
		return result;
	}

	private static List<ImportBean> adaptImports(DefinitionsBean defBean, List<Import> imports) {
		List<ImportBean> result = null;
		if(imports!=null){
			result = new ArrayList<ImportBean>();
			for(Import impt : imports){
				ImportBean importBean = new ImportBean();
				importBean.setLocation(impt.getLocation().toString());
				importBean.setNamespace(impt.getNamespace().toString());
				setDocumentation(importBean, impt);
				importBean.setImportType(impt.getImportType().toString());
				
//				if(impt.getSchema()!=null){
//					SchemaBean schema = buildSchemaBean(impt.getSchema());
//					importBean.setSchema(schema);
//				}
				
				defBean.addImport(importBean);
				result.add(importBean);
			}
		}
		return result;
	}

//	private static SchemaBean buildSchemaBean(Schema schema) {
//		SchemaBean schemaBean = new SchemaBean();
//		schemaBean.setDocumentURI(schema.getDocumentURI().toString());
//		schemaBean.setTargetNamespace(schema.getTargetNamespace());
//		buildTypes(schemaBean,schema);
//		if(schema.getElements()!=null){
//			for(Element elem : schema.getElements()){
//				ElementBean eBean = new ElementBean(elem.getQName().getLocalPart(), null, schemaBean);
//				SchemaType t = schemaBean.getType(elem.getType().getQName().getLocalPart());
//				if(t==null){
//					eBean.setType(retrieveBasicType(elem.getType()));
//				}
//				else{
//					eBean.setType(t);
//				}
//				schemaBean.addElement(eBean);
//			}
//		}
//		return schemaBean;
//	}
//
//	private static void buildTypes(SchemaBean schemaBean, Schema schema) {
//		if(schema.getTypes()!=null){
//			for(Type t : schema.getTypes()){
//				if(t instanceof SimpleType){
//					SimpleType st = (SimpleType) t;
//					if(st.getRestriction()!=null){
//						Restriction r = st.getRestriction();
//						if(r.getEnumerations()!=null){
//							RestrictionBean rBean = new RestrictionBean();
//							rBean.setName(st.getQName().getLocalPart());
//							schemaBean.addType(rBean);
//							for(Enumeration e : r.getEnumerations()){
//								rBean.addValue(e.getValue());
//							}
//						}
//					}
//				}
//				else if(t instanceof ComplexType){
//					ComplexType ct = (ComplexType) t;
//					if(ct.getSequence()!=null){
//						SequenceBean sb = new SequenceBean();
//						sb.setName(ct.getQName().getLocalPart());
//						schemaBean.addType(sb);
//						for(Element e : ct.getSequence().getElements()){
//							ElementBean eBean = new ElementBean(e.getQName().getLocalPart(), null, schemaBean);
//							SchemaType type = schemaBean.getType(e.getType().getQName().getLocalPart());
//							if(type==null){
//								eBean.setType(retrieveBasicType(e.getType()));
//							}
//							else{
//								eBean.setType(type);
//							}
//							sb.addElement(eBean);
//						}
//					}
//				}
//			}
//		}
//	}
	
	
	private static void setDocumentation(BPMNElementBean bean, BPMNElement element){
		//TODO decommenter quand la fonction getDocumentation aura été implémentée
//		if(element.getDocumentation()!=null){
//			bean.setDocumentation(element.getDocumentation().getContent());
//		}
	}
	
	private static EventDefinitionBean buildEventDefintion(EventDefinition ed){
		EventDefinitionBean result = null;
		if(ed instanceof MessageEventDefinition){
			result = new MessageEventDefinitionBean(ed.getId());
			QName messageName = ((MessageEventDefinition)ed).getMessageName();
			if(messageName!=null){
				((MessageEventDefinitionBean)result).setMessageQName(messageName.getLocalPart());
			}
		}
		else{
			throw new NotImplementedException("Only message event definition are accepted for the moment "+ed.getClass());
		}
		return result;
	}
	
	private static ProcessTypes retrieveProcessType(TProcessType type) {
		ProcessTypes result = null;
		if(type!=null){
			switch (type) {
			case EXECUTABLE:
				result = ProcessTypes.EXECUTABLE;
				break;
			case NON_EXECUTABLE:
				result = ProcessTypes.NON_EXECUTABLE;
				break;
			case NONE:
				result = ProcessTypes.NONE;
				break;
			case PUBLIC:
				result = ProcessTypes.PUBLIC;
				break;
			}
		}
		return result;
	}
	
	private static ItemKind retrieveItemKind(TItemKind itemKind) {
		ItemKind result = null;
		switch (itemKind) {
		case INFORMATION:
			result = ItemKind.INFORMATION;
			break;
		case PHYSICAL:
			result = ItemKind.PHYSICAL;
			break;
		}
		return result;
	}

	private static SchemaType retrieveBasicType(Type type) {
		if(type.equals(DefaultSchemaImpl.getInstance().getTypeBoolean())){
			return new BasicType(BasicTypes.BOOLEAN);
		}
		else if(type.equals(DefaultSchemaImpl.getInstance().getTypeDateTime())){
			return new BasicType(BasicTypes.DATE);
		}
		else if(type.equals(DefaultSchemaImpl.getInstance().getTypeDouble())){
			return new BasicType(BasicTypes.DOUBLE);
		}
		else if(type.equals(DefaultSchemaImpl.getInstance().getTypeFloat())){
			return new BasicType(BasicTypes.FLOAT);
		}
		else if(type.equals(DefaultSchemaImpl.getInstance().getTypeInt())){
			return new BasicType(BasicTypes.INT);
		}
		else if(type.equals(DefaultSchemaImpl.getInstance().getTypeString())){
			return new BasicType(BasicTypes.STRING);
		}
		else{
			return null;
		}
	}
}
