/**
 * Web Designer Framework - A simple framework for creating a web based designer - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.webdesigner.presentation.gwt.client.designer;

import com.ebmwebsourcing.webdesigner.business.domain.syntaxloader.SyntaxModel;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.component.DrawingPanelComponent;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.diagram.syntax.DiagramSyntax;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.event.DesignerListener;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.layout.DrawingPanel;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.DeferredCommand;


/**
 * Controller that exposes the main functionalities for the designer <br>
 * 
 * @author nfleury
 *
 */
public class DesignerController {

	private Designer designer;
	
	private DesignerModel model;
	
	public DesignerController(Designer designer) {
		this.designer = designer;
		this.model	  = new DesignerModel(this);
	}
	
	/**
	 * Loads a new diagram in a new tab
	 * @param syntax
	 * @param model
	 */
	public void loadDiagram(final DiagramSyntax syntax,final SyntaxModel model){
		
		this.mask("Loading diagram, please wait");
		
		DrawingPanelComponent drawingPanelComponent = getDesigner().getRegistry().getDrawingPanelComponentBySyntax(syntax);
		
		assert(drawingPanelComponent!=null);
		
		drawingPanelComponent.setModel(model);
		
		final DrawingPanel drawingPanel = (DrawingPanel) drawingPanelComponent.getComponent(); 
		
		
		getDesigner().getRegistry().getLayout().getDrawingPanelTab().addDrawingPanel(drawingPanel);
		getDesigner().getRegistry().getLayout().getDrawingPanelTab().doLayout();

		DeferredCommand.addCommand(new Command(){
			public void execute() {

				//trigger the onLoadDiagram event
				for(DesignerListener listener:getDesigner().getListeners()){
					listener.onLoadDiagram(syntax, model, drawingPanel);
				}
				
				unmask();
				
			}
		});
		
	}
	
	/**
	 * Reload a diagram with a specific id with the specified model 
	 * @param id of the syntaxmodel diagram that should already be loaded
	 * @param model the new model to display
	 * @param activate if True the tab will be activated
	 * @TODO:Find a better way ... its tricky but works quiet well ...
	 */
	public void refreshDiagram(String id,final SyntaxModel model,boolean activate) {
	
		if (activate){
			this.activateDrawingPanelById(id);
		}
		
		//first get the concerned drawingPanel
		final DrawingPanel drawingPanel = getModel().getDrawingPanelBySyntaxModelId(id);
		
		assert(drawingPanel!=null);

		DeferredCommand.addCommand(new Command(){
			public void execute() {

				//remove it
				getDesigner().getRegistry().getLayout().getDrawingPanelTab().remove(drawingPanel, true);
				//and reload all the stuff
				loadDiagram(drawingPanel.getSyntax(), model);

			}
		});		

	}
	
	
	public void activateDrawingPanelById(String id){
		
		getDesigner().getRegistry().getLayout().getDrawingPanelTab().activateDrawingPanelBySyntaxModelId(id);
		
	}
	
	public void showInfo(String msg){
		//TODO:
	}
	
	public void showError(String msg){
		//TODO:
	}
	
	public void mask(){
		designer.getRegistry().getLayout().getEl().mask();
	}
	
	public void mask(String msg){
		designer.getRegistry().getLayout().getEl().mask(msg,true);
	}
	
	public void unmask(){
		designer.getRegistry().getLayout().getEl().unmask();
	}
	
	/**
	 * Masks the DrawingPanel/Tab that currently has the focus
	 */
	public void maskActiveDrawingPanel(){
		designer.getRegistry().getLayout().getDrawingPanelTab().getCurrentDrawingPanel().getEl().mask();
	}
	
	/**
	 * Masks the DrawingPanel/Tab that currently has the focus
	 * @param msg Message to display
	 */
	public void maskActiveDrawingPanel(String msg){
		designer.getRegistry().getLayout().getDrawingPanelTab().getCurrentDrawingPanel().getEl().mask(msg,true);
	}
	
	/**
	 * Unmask the DrawingPanel/Tab that currently has the focus
	 */
	public void unmaskActiveDrawingPanel(){
		designer.getRegistry().getLayout().getDrawingPanelTab().getCurrentDrawingPanel().getEl().unmask();
	}
	
	public DesignerModel getModel() {
		return model;
	}
	
	public Designer getDesigner() {
		return designer;
	}
	
	

	
}
