/**
 * Web Designer Framework - A simple framework for creating a web based designer - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.webdesigner.presentation.gwt.client.layout;

import java.util.ArrayList;
import java.util.HashMap;

import com.ebmwebsourcing.gwt.raphael.client.diagram.DiagramPanel;
import com.ebmwebsourcing.webdesigner.business.domain.syntaxloader.SyntaxModel;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.diagram.syntax.DiagramElementInstance;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.diagram.syntax.DiagramElementType;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.diagram.syntax.DiagramSyntax;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.event.DiagramElementTypeInstantiationHandler;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.event.DrawingPanelListener;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.event.DrawingPanelListenerAdapter;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.log.AddElementLogEntry;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.log.ImpossibleOperationException;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.log.Log;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.log.PropertyChangedLogEntry;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.log.RemoveElementLogEntry;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.metamodel.DiagramModel;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.metamodel.properties.DiagramProperty;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.metamodel.properties.PropertyChangedHandler;
import com.google.gwt.user.client.ui.Widget;
import com.gwtext.client.widgets.Panel;
import com.gwtext.client.widgets.event.PanelListenerAdapter;

/**
 * 
 * @author nfleury
 * 
 */
public abstract class DrawingPanel extends Panel {
	

	
	private DiagramSyntax syntax;
	
	private DiagramModel diagramModel;
	
	private SyntaxModel syntaxModel;
	
	private DDPaletteHandler ddPaletteHander;
	
	private DrawingPanelTab drawingPanelTab;
	
	private ArrayList<DiagramElementInstance> selectedInstances;
	
	protected ArrayList<DrawingPanelListener> listeners = new ArrayList<DrawingPanelListener>();
	
	private HashMap<Widget, DiagramElementInstance> elementIntances = new HashMap<Widget, DiagramElementInstance>();
	
	private ArrayList<PasteHandler> pasteHandlers = new ArrayList<PasteHandler>();
	
	private Log log;

	private DrawingPanelListenerAdapter defaultDrawingPanelListener;
	
	private ArrayList<DiagramElementInstance> copiedElements = new ArrayList<DiagramElementInstance>();
	
	public DrawingPanel(DiagramSyntax syntax,SyntaxModel syntaxModel,DDPaletteHandler ddPaletteHandler) {
		
		this.syntax			 = syntax;
		
		this.ddPaletteHander = ddPaletteHandler;
		
		this.ddPaletteHander.setDrawingPanel(this);

		
		this.log 			= new Log(this);
		
		this.setBorder(false);
		this.setClosable(true);
		this.syntaxModel = syntaxModel;
		
		
		this.addDefaultListeners();
				
		this.bindListeners();

	}
	

	protected abstract void bindListeners();	
	

	public ArrayList<DiagramElementInstance> getElementsInstancesByType(DiagramElementType type){
		
		ArrayList<DiagramElementInstance> result = new ArrayList<DiagramElementInstance>();
		
		
		for(DiagramElementInstance instance : this.elementIntances.values()){
			
			if (instance.getElementType().equals(type)){
				result.add(instance);
			}
			
		}
		
		
		return result;
	}
	
	public void setSyntaxModel(SyntaxModel syntaxModel) {
		this.syntaxModel = syntaxModel;
	}
	
	public SyntaxModel getSyntaxModel() {
		return syntaxModel;
	}
	
	public Log getLog() {
		return log;
	}
	
	public void setDrawingPanelTab(DrawingPanelTab drawingPanelTab) {
		this.drawingPanelTab = drawingPanelTab;
	}
	
	public DrawingPanelTab getDrawingPanelTab() {
		return drawingPanelTab;
	}
	
	
	public abstract DiagramElementInstance getElementInstanceById(String id);
	
	protected void addElementInstance(Widget element,DiagramElementInstance elementInstance){
		this.elementIntances.put(element, elementInstance);
	}
	
	public HashMap<Widget, DiagramElementInstance> getElementIntances() {
		return elementIntances;
	}
	
	public void addListener(DrawingPanelListener listener){
		this.listeners.add(listener);
	}
	
	public DiagramSyntax getSyntax() {
		return syntax;
	}
	
	public DDPaletteHandler getDDPaletteHander() {
		return ddPaletteHander;
	}
	
	public ArrayList<PasteHandler> getPasteHandlers() {
		return pasteHandlers;
	}
	
	public void addPasteHandler(PasteHandler h){
		this.pasteHandlers.add(h);
	}
	
	public void undo() throws ImpossibleOperationException{
		this.log.undo();
	}
	
	
	public void redo() throws ImpossibleOperationException{
		this.log.redo();
	}
	
	
	public void copy(){

		copiedElements.clear();
		ArrayList<DiagramElementInstance> copiedElements = new ArrayList<DiagramElementInstance>();
		copiedElements.addAll(getSelectedInstances());
		this.copiedElements = copiedElements;

	}
	
	public void paste(){

		for(DiagramElementInstance ei:copiedElements){
			
			DiagramElementInstance eiCloned = ei.getClone(); 
			
			this.addElement(eiCloned);
			
			for(PasteHandler p:pasteHandlers){
				p.onPaste(ei, eiCloned);
			}
			
		}
		
	}
	

	
	
	
	/**
	 * Remove an element instance from current drawing panel
	 */
	public void removeElement(DiagramElementInstance elementInstance){
		
		for(DrawingPanelListener listener:listeners){
			listener.onDiagramElementRemoved(elementInstance);
		}
		
		this.elementIntances.remove(elementInstance.getViewInstance());
		this.removeElementViewFromPanel(elementInstance);
		
	}
	
	/**
	 * Removes the concrete view of an instance from the panel
	 * @param elementInstance
	 */
	protected abstract void removeElementViewFromPanel(DiagramElementInstance elementInstance);
	
	
	
	
	public void addElement(DiagramElementInstance instance){
		
		this.addElementViewToPanel(instance.getViewInstance());
		
		this.addElementInstance(instance.getViewInstance(), instance);
		
		for(DrawingPanelListener listener:listeners){
			listener.onDiagramElementAdded(instance);
		}
		
		//trigger the instantiation on type
		for(DiagramElementTypeInstantiationHandler handler:instance.getElementType().getInstantiationHandlers()){
			
			handler.onInstantiation(instance);
		}
		
	}
	
	public void addElement(DiagramElementInstance instance,int x,int y){
		
		this.addElementViewToPanel(instance.getViewInstance(),x,y);
		
		this.addElementInstance(instance.getViewInstance(), instance);
		
		for(DrawingPanelListener listener:listeners){
			listener.onDiagramElementAdded(instance,x,y);
		}
		

		//trigger the instantiation on type
		for(DiagramElementTypeInstantiationHandler handler:instance.getElementType().getInstantiationHandlers()){
			
			handler.onInstantiation(instance);
		}
	
	}
	
	/**
	 * Add an element type to drawing panel at the specified positions. <br>
	 * The view instance is automatically generated by using the syntax factory for current type syntax <br>
	 * @param type
	 * @param x
	 * @param y
	 */
	public DiagramElementInstance addElement(final DiagramElementType type,int x,int y){
		
		
		DiagramElementInstance elementInstance = this.genericAddElement(type);
		
		this.addElementViewToPanel(elementInstance.getViewInstance(), x, y);
		
		//trigger the onDiagramElementAdded event
		for(DrawingPanelListener listener:listeners){
			listener.onDiagramElementAdded(elementInstance, x, y);
		}
		
		return elementInstance;
	}
	
	/**
	 * Add an element type to drawing panel at the specified positions. <br>
	 * The view instance will not be auto generated as it should be specified<br>
	 * @param type
	 * @param viewInstance
	 * @param x
	 * @param y
	 * @return
	 */
	public DiagramElementInstance addElement(final DiagramElementType type,Widget viewInstance,int x,int y){
		
		
		DiagramElementInstance elementInstance = this.genericAddElement(viewInstance,type);
		
		this.addElementViewToPanel(elementInstance.getViewInstance(), x, y);
		
		//trigger the onDiagramElementAdded event
		for(DrawingPanelListener listener:listeners){
			listener.onDiagramElementAdded(elementInstance, x, y);
		}
		
		return elementInstance;
	}
	
	
	

	public DiagramElementInstance addElement(DiagramElementType type){
		
		DiagramElementInstance elementInstance = this.genericAddElement(type);
		
		this.addElementViewToPanel(elementInstance.getViewInstance());
		
		//trigger the onDiagramElementAdded event
		for(DrawingPanelListener listener:listeners){
			listener.onDiagramElementAdded(elementInstance);
		}
		
		return elementInstance;
	}
	
	public DiagramElementInstance addElement(DiagramElementType type,Widget viewInstance){
		
		DiagramElementInstance elementInstance = this.genericAddElement(viewInstance,type);
		
		this.addElementViewToPanel(viewInstance);
		
		//trigger the onDiagramElementAdded event
		for(DrawingPanelListener listener:listeners){
			listener.onDiagramElementAdded(elementInstance);
		}
		
		return elementInstance;
	}
	
	
	private DiagramElementInstance genericAddElement(DiagramElementType type){
		
		Widget elementInstanceView 			   = type.generateViewInstance();
		
		assert(elementInstanceView!=null);
		
		DiagramElementInstance elementInstance = new DiagramElementInstance(elementInstanceView,this,type);


		this.addElementInstance(elementInstanceView, elementInstance);		

		this.addPaletteListenersToInstance(elementInstance);
		

		//trigger the instantiation on type
		for(DiagramElementTypeInstantiationHandler handler:type.getInstantiationHandlers()){
			
			handler.onInstantiation(elementInstance);
		}
		
		return elementInstance;
	}
	
	
	public abstract DiagramPanel getDrawingZone();
	
	private DiagramElementInstance genericAddElement(Widget elementInstanceView,DiagramElementType type){
		

		DiagramElementInstance elementInstance = new DiagramElementInstance(elementInstanceView,this,type);


		this.addElementInstance(elementInstanceView, elementInstance);		

		this.addPaletteListenersToInstance(elementInstance);
		

		//trigger the instantiation on type
		for(DiagramElementTypeInstantiationHandler handler:type.getInstantiationHandlers()){
			
			handler.onInstantiation(elementInstance);
		}
		
		return elementInstance;
		
	}
	
	
	/**
	 * Should be used to add the concrete element view to current drawing panel
	 * @param elementInstanceView
	 * @param x
	 * @param y
	 */
	protected abstract void addElementViewToPanel(Widget elementInstanceView,int x,int y);

	/**
	 * Should be used to add the concrete element view to current drawing panel
	 * @param elementInstanceView
	 */
	protected abstract void addElementViewToPanel(Widget elementInstanceView);

	/**
	 * Method triggered each time that an element is added to <br>
	 * the drawing panel. It should be used to handle a drag and drop <br>
	 * from the palette to the drawing zone or on another element in the drawingzone <br>
	 * @param elementInstance
	 */
	protected abstract void addPaletteListenersToInstance(DiagramElementInstance elementInstance);

	
	
	public void setDiagramModel(DiagramModel diagramModel) {
		this.diagramModel = diagramModel;
	}
		
	public DiagramModel getDiagramModel() {
		return diagramModel;
	}
	
	@Override
	protected void onLoad() {
		super.onLoad();

	}
	

	public ArrayList<DrawingPanelListener> getListeners() {
		return listeners;
	}
	

	
	private void addDefaultListeners(){
		
		//elements are selected
		this.defaultDrawingPanelListener = new DrawingPanelListenerAdapter(){
			
			@Override
			public void onDiagramElementsSelected(ArrayList<DiagramElementInstance> elementInstance) {
			
				selectedInstances = elementInstance;
				
			}
			
			@SuppressWarnings("unchecked")
			@Override
			public void onDiagramElementAdded(final DiagramElementInstance elementInstance, int x, int y) {
				
				AddElementLogEntry log = new AddElementLogEntry(elementInstance,x,y);
				
				DrawingPanel.this.log.addEntry(log);
				
				elementInstance.getElementModel().addPropertyChangedHandler(new PropertyChangedHandler(){
					
					public void onChange(DiagramProperty p, Object oldValue,Object newValue) {
						
						PropertyChangedLogEntry log = new PropertyChangedLogEntry(elementInstance, p, oldValue, newValue);
						
						DrawingPanel.this.log.addEntry(log);
						
					}
					
					
				});
				
				
			}
			
			
			@Override
			public void onDiagramElementRemoved(DiagramElementInstance elementInstance) {
				
				RemoveElementLogEntry log = new RemoveElementLogEntry(elementInstance);
				
				DrawingPanel.this.log.addEntry(log);
			}
			
			
			
		};
		
		this.addListener(this.defaultDrawingPanelListener);
		
		this.addListener(new PanelListenerAdapter(){
			public void onClose(Panel panel) {
				drawingPanelTab.removeDrawingPanel(DrawingPanel.this);
		    }
		});
	}
	
	
	public DrawingPanelListenerAdapter getDefaultDrawingPanelListener() {
		return defaultDrawingPanelListener;
	}
	
	public void removeSelectedInstances(){
		
		for(DiagramElementInstance instance:getSelectedInstances()){
			
			this.removeElement(instance);
		
		}
	
	}
	
	public ArrayList<DiagramElementInstance> getSelectedInstances() {
		return selectedInstances;
	}


	public abstract void refresh();
	
}
