/**
 * Web Designer Framework - A simple framework for creating a web based designer - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.webdesigner.presentation.gwt.client.log;

import java.util.ArrayList;

import com.ebmwebsourcing.webdesigner.presentation.gwt.client.event.LogListener;
import com.ebmwebsourcing.webdesigner.presentation.gwt.client.layout.DrawingPanel;


/**
 * FIFO based log.
 * 
 * TODO: Work on memory leaks !!
 * @author gcrosmarie, nfleury
 *
 */
public class Log {

	private TimeSortedList entries;
	private int pointer;
	private DrawingPanel drawingPanel;
	private ArrayList<LogListener> listeners	 = new ArrayList<LogListener>();
	
	private ArrayList<AbstractLogEntry> undoList = new ArrayList<AbstractLogEntry>();
	private ArrayList<AbstractLogEntry> redoList = new ArrayList<AbstractLogEntry>();
	
	private int gotUndo;
	private int gotRedo;
	
	public Log(DrawingPanel panel){
		entries = new TimeSortedList();
		
		gotUndo = 0;
		gotRedo = 0;

		this.drawingPanel = panel;
	}
	
	public void addEntry(AbstractLogEntry entry){
		
		if(gotUndo>0){
			gotUndo --;
		}
		else if(gotRedo>0){
			gotRedo --;
		}
		else{
			entries.add(entry);
			
			//when adding an entry it goes straight to undo list
			undoList.add(entry);
			
			entry.setLog(this);
			
			for(LogListener log:listeners){
				log.onEntryAdded(entry);
			}
		}
	}
	
	public void undo() throws ImpossibleOperationException{

			
			//grab the last entry and put it in redo list
			AbstractLogEntry lastLogEntry = undoList.get(undoList.size()-1);
			redoList.add(lastLogEntry);

			gotUndo ++;
			
			//make the undo stuff
			lastLogEntry.undo();
			
			
			for(LogListener listener:listeners){
				listener.onUndo(undoList.get(undoList.size()-1));
			}

			undoList.remove(lastLogEntry);

	}
	
	public void redo() throws ImpossibleOperationException{

			
			//just redo the last inserted entry and remove the stuff
			AbstractLogEntry lastLog = redoList.get(redoList.size()-1); 
			undoList.add(lastLog);
			
			gotRedo ++;
			
			lastLog.redo();

			for(LogListener listener:listeners){
				listener.onRedo(redoList.get(redoList.size()-1));
			}
			
			redoList.remove(redoList.size()-1);

	}
	
	public void undo(int times) throws ImpossibleOperationException{
		if(pointer-times+1<0){
			throw new ImpossibleOperationException("Cannot do "+times+" undo at this point.");
		}
		else{
			for(int i=0; i<times; i++){
				entries.get(pointer-i).undo();
			}
			pointer = pointer-times+1;
		}
	}
	
	public void redo(int times) throws ImpossibleOperationException{
		if(pointer+times>entries.size()-1){
			throw new ImpossibleOperationException("Cannot do "+times+" redo at this point.");
		}
		else{
			for(int i=0; i<times; i++){
				pointer++;
				entries.get(pointer).redo();
			}
		}
	}
	
	
	public String[] getDisplayStrings(int numberOfLogs){
		if(numberOfLogs>0){
			String[] result = new String[Math.min(numberOfLogs, entries.size())];
			int i = 0;
			while(i<numberOfLogs && i<entries.size()){
				result[i] = entries.get(i).getDescription();
				i++;
			}
			return result;
		}
		else{
			throw new IllegalArgumentException("The asked number of logs must be strictly greater then 0. ("+numberOfLogs+")");
		}
	}
	
	public String[] getDisplayStrings(){
		String[] result = new String[entries.size()];
		for(int i=0; i<result.length; i++){
			result[i] = entries.get(i).getDescription();
		}
		return result;
	}
	
	public void clear(){
		entries.clear();
		pointer = -1;
	}
	
	public TimeSortedList getEntries() {
		return entries;
	}

	public DrawingPanel getDrawingPanel() {
		return drawingPanel;
	}

	public void setDrawingPanel(DrawingPanel drawingPanel) {
		this.drawingPanel = drawingPanel;
	}
	
	
	public boolean hasMoreUndo(){
		return undoList.size()>0;
	}
	
	public boolean hasMoreRedo(){
		if (redoList.size()>=1){
			return true;
		}
		return false;
	}
	
	
	public void addListener(LogListener listener){
		this.listeners.add(listener);
	}
	

	
	
	public ArrayList<LogListener> getListeners() {
		return listeners;
	}
	
	
}
