/**
 * geasy-diagram-editor - A project for editing diagrams based on OMG Diagram Definition standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.geasytools.diagrameditor;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramElementView;
import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramView;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.diagramcommon.layout.IPoint;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.diagramcommon.layout.Point;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IDiagram;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IDiagramElement;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IEdge;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IShape;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.IConnector;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.IConnectorEnd;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.IConnectorPoint;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.events.IAddWayPointEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.events.IConnectionEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.events.IDisconnectionEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.events.IRefreshEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.events.IRemoveWayPointEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.core.IContainer;
import com.ebmwebsourcing.geasytools.geasyui.api.core.events.IAddUIElementEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.core.events.IBoundsUpdateHandler;
import com.ebmwebsourcing.geasytools.geasyui.api.core.events.IContainerHandler;
import com.ebmwebsourcing.geasytools.geasyui.api.core.events.IPositionChangeEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.core.events.ISizeChangeEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.uipanel.events.INewUIElementEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.uipanel.events.IRemoveUIElementEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.uipanel.events.IResizeRequestEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.uipanel.events.ISelectedElementsEvent;
import com.ebmwebsourcing.geasytools.geasyui.api.uipanel.events.IUIPanelHandler;
import com.ebmwebsourcing.geasytools.geasyui.api.uipanel.events.IUnselectedElementsEvent;
import com.ebmwebsourcing.geasytools.geasyui.impl.connectable.events.ConnectorHandler;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.UndoRedoSession;
import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Window;



/**
 * Builds a Diagram Object based on user interactions only.
 * Adds default handlers to DiagramUI to build a default Diagram Interchange.
 * 
 * @author nfleury
 *
 */
public class DiagramInterchangeModelBuilder {
	
	private IDiagramView diagramUI;
	private DiagramController diagramController;
	
	private IDiagram diModel;
	
	private UndoRedoSession undoRedoSession;
	
	private IUIPanelHandler panelHandler;
	
	private HashMap<IEdge, List<IPoint>> edgeWayPoints;
	
	public DiagramInterchangeModelBuilder(IDiagramView diagram, DiagramController diagramController) {
		
		this.edgeWayPoints = new HashMap<IEdge, List<IPoint>>();
		
		this.diagramUI = diagram;
		this.diagramController = diagramController;
		
		this.undoRedoSession = new UndoRedoSession();
		
		//initialize diagrams models
		this.initialize();
//		
		addListeners();
	
	}
	
	
	public IDiagramView getDiagram() {
		return diagramUI;
	}
	
	public IDiagram getDiagramInterchangeModel(){		
		return diModel;
	}
	
	public void initialize(){
		
		this.diModel = diagramUI.getDiagram();
		
		this.undoRedoSession.registerModel(diModel);
		
	}
	
	
	public void addListeners(){
		
		this.panelHandler = new IUIPanelHandler() {
			
			@Override
			public void onUnSelectUIElement(IUnselectedElementsEvent unselectedElementsEvent) {
				
				
				
			}
			
			@Override
			public void onSelectUIElement(ISelectedElementsEvent selectedElementsEvent) {
				
				
			}
			
			
			@Override
			public void onNewUIElement(INewUIElementEvent addElementEvent) {
				

				//1- When adding an element we 
				//a- add it to diagrams model plane
				//b- register the editors model to actual undoredosession
	
				if (addElementEvent.getNewElement() instanceof IDiagramElementView){
					
					
					final IDiagramElementView diView = (IDiagramElementView) addElementEvent.getNewElement();
					
					
					
					//register editor model to undoredosession
					IDiagramElement diElement = diView.getDiagramElement();
//					undoRedoSession.registerModel(diElement);

					//1 - add diagram element to plane;
					diModel.addDiagramElement(diElement);
					
					//initialize bounds if its a shape
					if (diView.getDiagramElement() instanceof IShape){

						final IShape shape  = (IShape) diView.getDiagramElement();
						
							shape.getBounds().setX(diView.getRelativeX()+diView.getUIPanel().getScrollLeft());
							shape.getBounds().setY(diView.getRelativeY()+diView.getUIPanel().getScrollTop());
							shape.getBounds().setWidth(diView.getWidth());
							shape.getBounds().setHeight(diView.getHeight());
						
					}
					
					
					//2- If actual diagram element is a container add 
					//receive di into di container
					if (diView instanceof IContainer){
						
						IContainer container = (IContainer) diView;
						
						container.addContainerHandler(new IContainerHandler() {
							
							@Override
							public void onRemoveUIElement(
									com.ebmwebsourcing.geasytools.geasyui.api.core.events.IRemoveUIElementEvent event) {
								

								if (event.getUIElement() instanceof IDiagramElementView){
									
									IDiagramElementView removedElement = (IDiagramElementView) event.getUIElement() ;
									
									diView.getDiagramElement().removeDiagramElement(removedElement.getDiagramElement());
									
									//bind syntax model
									diView.getSyntaxModelBinder().bindSyntaxModel(diView.getDiagramElement().getModelElement(), diView.getEditorModel(), diView.getDiagramElement());
									
								}
								
							}
							
							@Override
							public void onAddUIElement(IAddUIElementEvent event) {
								
								if (event.getUIElement() instanceof IDiagramElementView){
									
									IDiagramElementView addedElement = (IDiagramElementView) event.getUIElement() ;
									
									diView.getDiagramElement().addDiagramElement(addedElement.getDiagramElement());
								
									//bind syntax model
									diView.getSyntaxModelBinder().bindSyntaxModel(diView.getDiagramElement().getModelElement(), diView.getEditorModel(), diView.getDiagramElement());
									
								}
								
								
								
							}
						});
						
					}

					//3- Add handler on connectors so that an edge
					// is created when they are connected
					if (diView instanceof IConnector){
					
						IConnector connector = (IConnector) diView;
						
						connector.addConnectorHandler(new ConnectorHandler() {
							
							@Override
							public void onRemoveWayPoint(IRemoveWayPointEvent event) {
								
								//update edge wayPoints
								IEdge edge = (IEdge) diView.getDiagramElement();
								IConnector connector = (IConnector) diView;
								
								//remove previous waypoints from current edge
								edge.clearWayPoints();
								
								Point endWayPoint = null;
								
								//Add new ones
								Iterator<IConnectorPoint> it = connector.getAllConnectorPoints().iterator();
								while (it.hasNext()){
									
									//end points will always we on 2 position
									//grab it and put in in the end
									IConnectorPoint p = it.next();
									
									Point wayPoint = new Point(p.getRelativeX()+p.getUIPanel().getScrollLeft()+Window.getScrollLeft(), p.getRelativeY()+p.getUIPanel().getScrollTop()+Window.getScrollTop()); 

									if (p instanceof IConnectorEnd==false){										
										edge.addWayPoint(wayPoint);										

									}else{
										endWayPoint = wayPoint;
									}
									
								}
								

								
								edge.addWayPoint(endWayPoint);

							}
							
							@Override
							public void onConnection(IConnectionEvent event) {
									
								IEdge edge = (IEdge) diView.getDiagramElement();
								
								IConnector connector = (IConnector) diView;

								
								//Start 
								if (event.getConnectionSource()!=null){
										
									IDiagramElement sourceModel = ((IDiagramElementView)event.getConnectionSource()).getDiagramElement();
									edge.setSource(sourceModel);
									
									//add wayPoint to edge
									Point wayPoint = new Point(connector.getConnectorStartPoint().getRelativeX(), connector.getConnectorStartPoint().getRelativeY());									
									edge.addWayPoint(wayPoint);
									
									
								}else{
									edge.setSource(null);
								}
								
								//End
								if (event.getConnectionTarget()!=null){
									
									IDiagramElement targetModel = ((IDiagramElementView)event.getConnectionTarget()).getDiagramElement();
									edge.setTarget(targetModel);
									
									//add wayPoint to edge
									Point wayPoint = new Point(connector.getConnectorEndPoint().getRelativeX(), connector.getConnectorEndPoint().getRelativeY());									
									edge.addWayPoint(wayPoint);
									

									
								}else{
									edge.setTarget(null);
								}

								
								//bind => as we cannot access actual syntax model it has to be done in 
								//the syntax binder
								diView.getSyntaxModelBinder().bindSyntaxModel(diView.getDiagramElement().getModelElement(),
										diView.getEditorModel(),
										diView.getDiagramElement());
								
							}
							
							@Override
							public void onAddWayPoint(IAddWayPointEvent event) {
								
								IEdge edge = (IEdge) diView.getDiagramElement();
								
								Point wayPoint = new Point(event.getWayPointPosition().getX(), event.getWayPointPosition().getY()); 
								
								edge.addWayPoint(wayPoint);
								
							}

							@Override
							public void onDisconnection(
									IDisconnectionEvent event) {
								// TODO Auto-generated method stub
								
							}

							@Override
							public void onRefresh(IRefreshEvent event) {
								
								//update edge wayPoints
								IEdge edge = (IEdge) diView.getDiagramElement();
								IConnector connector = (IConnector) diView;
								
								//remove previous waypoints from current edge
								edge.clearWayPoints();
								
								Point endWayPoint = null;
								
								//Add new ones
								Iterator<IConnectorPoint> it = connector.getAllConnectorPoints().iterator();
								while (it.hasNext()){
									
									//end points will always we on 2 position
									//grab it and put in in the end
									IConnectorPoint p = it.next();
									
									Point wayPoint = new Point(p.getRelativeX()+p.getUIPanel().getScrollLeft()+Window.getScrollLeft(), p.getRelativeY()+p.getUIPanel().getScrollTop()+Window.getScrollTop()); 

									if (p instanceof IConnectorEnd==false){										
										edge.addWayPoint(wayPoint);										

									}else{
										endWayPoint = wayPoint;
									}
									
								}
								
								edge.addWayPoint(endWayPoint);

							}
						});
						
					}
					
					

					//4- Add bound update handlers if actual DI is a Shape
					//so that we update corresponding di bounds
					if (diView.getDiagramElement() instanceof IShape){

						final IShape shape  = (IShape) diView.getDiagramElement();
						
						diView.addBoundUpdateHandler(new IBoundsUpdateHandler() {
							
							@Override
							public void onSizeChange(ISizeChangeEvent event) {

								shape.getBounds().setWidth(event.getNewWidth());
								shape.getBounds().setHeight(event.getNewHeight());
								
							}
							
							@Override
							public void onPositionChange(IPositionChangeEvent event) {
								//System.out.println(diView.getName()+" position x:"+event.getNewRelativeX()+" y:"+event.getNewRelativeY());
								shape.getBounds().setX(event.getNewRelativeX());
								shape.getBounds().setY(event.getNewRelativeY());
								
							}
						});

							
					}
				
				
				
				}
				

				//build diagram syntax model
				//FIXME: maybe not the right place to put that ...
				diagramUI.getSyntaxModelBuilder().getSyntaxModel();
				
				
			}

			@Override
			public void onResizeRequest(IResizeRequestEvent event) {
				// TODO Auto-generated method stub
				
			}
			
			@Override
			public void onRemoveUIElement(IRemoveUIElementEvent removeElementEvent) {
					
					IDiagramElementView elView = (IDiagramElementView) removeElementEvent.getRemovedElement();
				
					diModel.getRootElement().removeDiagramElement(elView.getDiagramElement());
					
			}
		};
 
		diagramUI.addUIPanelHandler(panelHandler);		
	}


	
	
	
}
