package com.ebmwebsourcing.petalsview.service.flowref;

import java.util.List;

import javax.annotation.Resource;

import org.springframework.stereotype.Service;

import com.ebmwebsourcing.petalsview.persistence.dao.flowref.FlowStepErrorRefDAO;
import com.ebmwebsourcing.petalsview.persistence.dao.flowref.FlowStepRefDAO;
import com.ebmwebsourcing.petalsview.persistence.model.flowref.FlowRef;
import com.ebmwebsourcing.petalsview.persistence.model.flowref.FlowStepErrorRef;
import com.ebmwebsourcing.petalsview.persistence.model.flowref.FlowStepRef;
import com.ebmwebsourcing.petalsview.persistence.model.flowref.StepParameter;

@Service("flowStepRefManager")
public class FlowStepRefManagerImpl implements FlowStepRefManager {

	@Resource(name = "flowStepRefDAO")
	private FlowStepRefDAO flowStepRefDAO;

	@Resource(name = "flowStepErrorRefDAO")
	private FlowStepErrorRefDAO flowStepErrorRefDAO;

	@Resource(name = "flowRefManager")
	private FlowRefManager flowRefManager;
	
	//Methods to load, insert or remove Flow Step referential
	public List<FlowStepRef> loadSteps(final int type) {
		final List<FlowStepRef> result = flowStepRefDAO.loadByFlowType(type);
		return result;
	}

	public FlowStepRef loadStepRef(final int type, final String interfaceName,
			final String serviceName) throws FlowRefException{
		if(interfaceName == null){
			throw new FlowRefException("You must specify a flow step referential interface name ");
		}
		if(serviceName == null){
			throw new FlowRefException("You must specify a flow step referential service name ");
		}
		
		final FlowStepRef result = flowStepRefDAO
				.loadByFlowTypeIntNameServName(type, interfaceName, serviceName);
		return result;
	}

	public FlowStepRef loadStepRef(final String id) throws FlowRefException{
		
		if(id == null){
			throw new FlowRefException("You must specify a flow step referential id ");
		}
		
		final FlowStepRef result = (FlowStepRef) flowStepRefDAO.get(id);
		return result;
	}

	public void removeStepRefs(final String[] ids) throws FlowRefException{
		if(ids == null){
			throw new FlowRefException("You must specify a flow step referential id ");
		}
		for (final String id : ids) {
			flowStepRefDAO.remove(flowStepRefDAO.get(id));
		}
	}

	public String saveOrUpdateStepRef(final FlowStepRef flowStepRef) throws FlowRefException{

		if(flowStepRef == null){
			throw new FlowRefException("You must specify a flow step referential");
		}
		FlowStepRef flowStepRefTmp = new FlowStepRef();
		if (flowStepRef.getId() != null) {
			flowStepRefTmp = flowStepRefDAO.get(flowStepRef.getId());
		} else {
			flowStepRefTmp.setFlowStartStep(false);
			flowStepRefTmp.setFlowEndStep(false);
		}

		verification(flowStepRef, flowStepRefTmp);

		verificationdate(flowStepRef, flowStepRefTmp);

		flowStepRefTmp.setInterfaceName(flowStepRef.getInterfaceName());
		flowStepRefTmp.setName(flowStepRef.getName());
		flowStepRefTmp.setServiceName(flowStepRef.getServiceName());
		flowStepRefTmp.setFlowref(flowStepRef.getFlowref());
		flowStepRefTmp.setFlowStartStep(flowStepRef.isFlowStartStep());
		flowStepRefTmp.setFlowEndStep(flowStepRef.isFlowEndStep());
		flowStepRefTmp.setSuccessMessage(flowStepRef.getSuccessMessage());
		flowStepRefTmp.setDefaultErrorMessage(flowStepRef
				.getDefaultErrorMessage());

		flowStepRefDAO.save(flowStepRefTmp);

		final String result = flowStepRefTmp.getId();
		return result;
	}
	
	
	//Methods to load, insert or remove Flow Step Error Referential
	
	public FlowStepErrorRef loadStepErrorRef(final String id) throws FlowRefException{
		
		if(id == null){
			throw new FlowRefException("You must specify a flow step error referential id ");
		}
		final FlowStepErrorRef result = (FlowStepErrorRef) flowStepErrorRefDAO
				.get(id);
		return result;
	}

	public List<FlowStepErrorRef> loadStepRefErrors(final String stepRefId) throws FlowRefException{
		
		if(stepRefId == null){
			throw new FlowRefException("You must specify a flow step referential id ");
		}
		
		final List<FlowStepErrorRef> result = flowStepErrorRefDAO
				.loadByStepRefId(stepRefId);
		return result;
	}

	public String saveOrUpdateErrorRef(final FlowStepErrorRef stepErrorRef) throws FlowRefException{

		
		FlowStepErrorRef stepErrorRefTmp = flowStepErrorRefDAO.loadByErrorCode(stepErrorRef.getErrorCode());
		
		verificationErrorRef(stepErrorRef,stepErrorRefTmp);
		if(stepErrorRefTmp == null){
			 stepErrorRefTmp = new FlowStepErrorRef ();
		}
		
		stepErrorRefTmp.setErrorCode(stepErrorRef.getErrorCode());
		stepErrorRefTmp.setMessage(stepErrorRef.getMessage());
		stepErrorRefTmp.setFlowstepref(stepErrorRef.getFlowstepref());
		
		flowStepErrorRefDAO.save(stepErrorRefTmp);
		final String result = stepErrorRefTmp.getId();
		return result;
	}
	
	public void removeErrorRefs(final String[] ids) throws FlowRefException{
		if(ids == null){
			throw new FlowRefException("you must specify valid flow step error referential ");
		}
		for (final String id : ids) {
			flowStepErrorRefDAO.remove(flowStepErrorRefDAO.get(id));
		}
	}
	
	//Methods to insert or remove Flow Step Parameter Referential
	
	public void addParameterName(final String flowRefId,
			final String parameterName, final boolean global) throws FlowRefException{
		if(parameterName == null){
			throw new FlowRefException("You must specify a flow step parameter name referential");
		}
		
		this.addParameterName(flowRefId, -1, parameterName, global);
	}

	public void addParameterName(final String flowStepRefId, final int index,
			final String parameterName, final boolean global) throws FlowRefException {
		
		verificationParameter(flowStepRefId, parameterName);
		
		final FlowStepRef flowstepRef = (FlowStepRef) flowStepRefDAO
				.get(flowStepRefId);
		if (index > -1) {
			final StepParameter parameter = new StepParameter(parameterName,
					global);
			flowstepRef.addParameterName(index, parameter);
		} else {
			final StepParameter parameter = new StepParameter(parameterName,
					global);
			flowstepRef.addParameterName(parameter);
		}
		flowStepRefDAO.save(flowstepRef);
	}

	public void updateParameterName(final String flowStepRefId,
			final int index, final String parameterName, final boolean global) throws FlowRefException
	{
		verificationParameter(flowStepRefId, parameterName); 
		final FlowStepRef flowStepRef = (FlowStepRef) flowStepRefDAO
				.get(flowStepRefId);
		final StepParameter parameter = new StepParameter(parameterName, global);
		flowStepRef.updateParameterName(index, parameter);
		flowStepRefDAO.save(flowStepRef);
	}
	
	public void removeParameterName(final String flowStepId, final int index) throws FlowRefException {
		
		if(flowStepId == null){
			throw new FlowRefException("You must specify a flow step referential ");
		}
		
		final FlowStepRef result = (FlowStepRef) flowStepRefDAO.get(flowStepId);
		result.removeParameterName(index);
		flowStepRefDAO.save(result);
	}

	public void removeParameterNames(final String flowStepId,
			final int[] indexes) throws FlowRefException{
		
		if(flowStepId == null){
			throw new FlowRefException("You must specify a flow step referential ");
		}
		
		final FlowStepRef result = (FlowStepRef) flowStepRefDAO.get(flowStepId);
		this.removeParameterName(result, indexes);
		flowStepRefDAO.save(result);
	}

	private void removeParameterName(final FlowStepRef flowStepRef,
			final int[] indexes) {
		
		
		if (indexes.length > 0) {
			flowStepRef.removeParameterName(indexes[0]);

			final int[] decrementedIndexes = new int[indexes.length - 1];
			for (int i = 1; i < indexes.length; i++) {
				final int treatedInt = indexes[i];
				if (treatedInt > indexes[0]) {
					decrementedIndexes[i - 1] = treatedInt - 1;
				} else {
					decrementedIndexes[i - 1] = treatedInt;
				}
			}
			this.removeParameterName(flowStepRef, decrementedIndexes);
		}
	}

	/*
	 * Verification flow step referential before save or update in database
	 * 
	 */

	private void verificationdate(final FlowStepRef flowStepRef,
			FlowStepRef flowStepRefTmp) {
		// Avoid multiple start steps

		if (flowStepRef.isFlowStartStep() && !flowStepRefTmp.isFlowStartStep()) {
			final FlowStepRef existingStartStep = flowStepRefDAO
					.loadStartStepByFlowType(flowStepRef.getFlowref().getType());
			if (existingStartStep != null) {
				if (existingStartStep.getId() != flowStepRef.getId()) {
					throw new FlowRefManagerRuntimeException(
							"Already existing start flow step for this type of flow");
				}
			}
		}

		// Avoid multiple end steps
		if (flowStepRef.isFlowEndStep() && !flowStepRefTmp.isFlowEndStep()) {
			final FlowStepRef existingEndStep = flowStepRefDAO
					.loadEndStepByFlowType(flowStepRef.getFlowref().getType());
			if (existingEndStep != null) {
				if (existingEndStep.getId() != flowStepRef.getId()) {
					throw new FlowRefManagerRuntimeException(
							"Already existing end flow step for this type of flow");
				}
			}
		}
	}

	private void verification(final FlowStepRef flowStepRef,
			FlowStepRef flowStepRefTmp) throws FlowRefException{
		
		if(flowStepRef.getName() == null){
			throw new FlowRefManagerRuntimeException("You must specify a name ");
		}
		if(flowStepRef.getFlowref() == null){
			throw new FlowRefManagerRuntimeException("You must specify a flow referential ");
		}
		FlowRef flowRef = flowRefManager.loadFlowRef(flowStepRef.getFlowref().getId());
		if(flowRef == null){
			throw new FlowRefException("you must specify a valid flow referential");
		}
		
		int type = flowStepRef.getFlowref().getType();
		String interfaceName = flowStepRef.getInterfaceName();
		String serviceName = flowStepRef.getServiceName();

		if (flowStepRefTmp.getId() != null) {

			if (!flowStepRef.getServiceName().equals(
					flowStepRefTmp.getServiceName())
					|| !flowStepRef.getInterfaceName().equals(
							flowStepRefTmp.getInterfaceName())) {

				FlowStepRef flowStepTmp = flowStepRefDAO
						.loadByFlowTypeIntNameServName(type, interfaceName,
								serviceName);

				if (flowStepTmp != null) {
					if (!flowStepTmp.getId().equals(flowStepRef.getId())) {
						throw new FlowRefManagerRuntimeException(
								"a flow step with that service and interface already exist in database.");
					}
				}
			}
			if(!flowStepRef.getName().equals(flowStepRefTmp.getName())){
				FlowStepRef tmp = flowStepRefDAO.loadStartStepByName(flowStepRef.getName());
				if(tmp != null){
					throw new FlowRefManagerRuntimeException(
					"a flow step with that name already exist in database.");
				}
			}

		} else {
			FlowStepRef flowStepTmp = flowStepRefDAO
					.loadByFlowTypeIntNameServName(type, interfaceName,
							serviceName);

			if (flowStepTmp != null) {
				if (!flowStepTmp.getId().equals(flowStepRef.getId())) {
					throw new FlowRefManagerRuntimeException(
							"a flow step with that service and interface already exist in database.");
				}
			}
			
			flowStepTmp = new FlowStepRef ();
			flowStepTmp = flowStepRefDAO.loadStartStepByName(flowStepRef.getName());
			if(flowStepTmp != null){
				throw new FlowRefManagerRuntimeException(
				"a flow step with that name already exist in database.");
			}
			
		}
	}

	private void verificationErrorRef (FlowStepErrorRef stepErrorRef,FlowStepErrorRef stepErrorRefTmp) throws FlowRefException{
		
		if(stepErrorRefTmp !=null){
			if(stepErrorRef.getId() != null){
				if(!stepErrorRef.getId().equals(stepErrorRefTmp.getId()) && stepErrorRef.getErrorCode()==stepErrorRefTmp.getErrorCode() ){
					throw new FlowRefException("An error referential already exist with that error code in database " + stepErrorRef.getErrorCode());
				}
			}else {
				throw new FlowRefException("An error referential already exist with that error code in database " + stepErrorRef.getErrorCode());
			}
		}
		
		if(stepErrorRef.getFlowstepref() == null || stepErrorRef.getFlowstepref().getId() == null){
			throw new FlowRefException("You must specify a flow step referential ");
		}
		if(stepErrorRef.getMessage() == null){
			throw new FlowRefException("You must specify a flow step error message referential ");
		}
	}
	
	private void verificationParameter(String flowStepRefId, String parameterName) throws FlowRefException{
		if(flowStepRefId == null){
			throw new FlowRefException("You must specify a flow step referential ");
		}
		if(parameterName == null){
			throw new FlowRefException("You must specify a flow step parameter name referential ");
		}
	}

}
