/**
 * Petals View - Functional Supervision.
 * Copyright (c) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * FlowStepRef.java
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.petalsview.persistence.model.flowref;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.JoinColumn;
import javax.persistence.JoinTable;
import javax.persistence.ManyToOne;
import javax.persistence.MapKey;
import javax.persistence.OneToMany;
import javax.persistence.Table;
import javax.persistence.UniqueConstraint;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.builder.ToStringStyle;
import org.hibernate.annotations.CollectionOfElements;
import org.hibernate.annotations.ForeignKey;
import org.hibernate.annotations.IndexColumn;

import com.ebmwebsourcing.webcommons.persistence.bo.StringIdBaseObject;

/**
 * This class represents a functional flow step definition.
 * 
 * @author Pierre SMEYERS, ofabre
 */
@Entity
@Table(name = "FLOWSTEPREF", uniqueConstraints = {
        @UniqueConstraint(columnNames = { "flowref_id", "name" }),
        @UniqueConstraint(columnNames = { "flowref_id", "interfaceName", "serviceName" }) })
public class FlowStepRef extends StringIdBaseObject {
    /**
	 * 
	 */
    private static final long serialVersionUID = -4099365746136084661L;

    // ========================================================================
    // === Attributes
    // ========================================================================

    private String interfaceName;

    private String serviceName;

    private String name;

    private boolean flowStartStep;

    private boolean flowEndStep;

    private FlowRef flowref;

    private Map<Integer, FlowStepErrorRef> errorMessages;

    private String successMessage;

    private String defaultErrorMessage;

    private List<StepParameter> parameterNames;

    // ========================================================================
    // === Constructors
    // ========================================================================
    public FlowStepRef() {
        this.parameterNames = new ArrayList<StepParameter>();
    }

    // ========================================================================
    // === Attributes Getters & Setters
    // ========================================================================

    public boolean isFlowStartStep() {
        return this.flowStartStep;
    }

    public void setFlowStartStep(final boolean flowStartStep) {
        this.flowStartStep = flowStartStep;
    }

    public boolean isFlowEndStep() {
        return this.flowEndStep;
    }

    public void setFlowEndStep(final boolean flowEndStep) {
        this.flowEndStep = flowEndStep;
    }

    public String getName() {
        return this.name;
    }

    public String getInterfaceName() {
        return this.interfaceName;
    }

    public void setInterfaceName(final String interfaceName) {
        this.interfaceName = interfaceName;
    }

    public String getServiceName() {
        return this.serviceName;
    }

    public void setServiceName(final String serviceName) {
        this.serviceName = serviceName;
    }

    public void setName(final String name) {
        this.name = name;
    }

    @ManyToOne(fetch = FetchType.LAZY)
    @JoinColumn(name = "flowref_id", nullable = false, updatable = false)
    @ForeignKey(name = "FK_FLOWREF_id_FLOWSTEPREF")
    public FlowRef getFlowref() {
        return this.flowref;
    }

    public void setFlowref(final FlowRef flowref) {
        this.flowref = flowref;
    }

    // FIXME Cette classe Serializable définit un champ d'instance non primitif
    // qui n'est ni transient, ni Serializable ou java.lang.Object, et qui ne
    // semble pas implémenter l'interface Externalizable ou les méthodes
    // readObject() et writeObject(). Les objets de cette classe ne seront pas
    // désérialisés correctement si un
    // objet non Serializable est stocké dans ce champ
    @OneToMany(mappedBy = "flowstepref", fetch = FetchType.EAGER, cascade = CascadeType.ALL)
    @MapKey(name = "errorCode")
    public Map<Integer, FlowStepErrorRef> getErrorMessages() {
        return this.errorMessages;
    }

    public void setErrorMessages(final Map<Integer, FlowStepErrorRef> errorMessages) {
        this.errorMessages = errorMessages;
    }

    public String getSuccessMessage() {
        return this.successMessage;
    }

    public void setSuccessMessage(final String successMessage) {
        this.successMessage = successMessage;
    }

    public String getDefaultErrorMessage() {
        return this.defaultErrorMessage;
    }

    public void setDefaultErrorMessage(final String defaultErrorMessage) {
        this.defaultErrorMessage = defaultErrorMessage;
    }

    @CollectionOfElements(fetch = FetchType.LAZY)
    @JoinTable(name = "FLOWREF_PARAMNAMES")
    @IndexColumn(name = "idx")
    @ForeignKey(name = "FK_PARAMNAMES_id_FLOWREF")
    public List<StepParameter> getParameterNames() {
        return this.parameterNames;
    }

    public void setParameterNames(final List<StepParameter> parameterNames) {
        this.parameterNames = parameterNames;
    }

    public void addParameterName(final StepParameter parameterName) {
        this.parameterNames.add(parameterName);
    }

    public void addParameterName(final int index, final StepParameter parameterName) {
        this.parameterNames.add(index, parameterName);
    }

    public void updateParameterName(final int index, final StepParameter parameterName) {
        this.parameterNames.set(index, parameterName);
    }

    public void removeParameterName(final int index) {
        this.parameterNames.remove(index);
    }

    @Override
    public boolean equals(final Object other) {
        if (!(other instanceof FlowStepRef))
            return false;
        FlowStepRef castOther = (FlowStepRef) other;
        return new EqualsBuilder().append(interfaceName, castOther.interfaceName).append(
                serviceName, castOther.serviceName).append(name, castOther.name).append(
                flowStartStep, castOther.flowStartStep).append(flowEndStep, castOther.flowEndStep)
                .append(flowref, castOther.flowref).append(errorMessages, castOther.errorMessages)
                .append(successMessage, castOther.successMessage).append(defaultErrorMessage,
                        castOther.defaultErrorMessage).isEquals();
    }

    @Override
    public int hashCode() {
        return new HashCodeBuilder().append(interfaceName).append(serviceName).append(name).append(
                flowStartStep).append(flowEndStep).append(flowref).append(errorMessages).append(
                successMessage).append(defaultErrorMessage).toHashCode();
    }
    
    @Override
    public String toString() {
        return new ToStringBuilder(this, ToStringStyle.MULTI_LINE_STYLE).appendSuper(
                super.toString()).append("name", this.name).append("interfaceName", this.interfaceName)
                .append("serviceName", this.serviceName).append("flowref", this.flowref)
                .append("errorMessages", this.errorMessages).append("successMessage", this.successMessage)
                .append("defaultErrorMessage", this.defaultErrorMessage)
                .append("parameterNames", this.parameterNames)
               .toString();
    }

}
