/**
 * bpmn-diagram - SVG/VML web based editor for BPMN Standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.bpmndiagram.events;

import java.util.HashSet;

import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramElementView;
import com.ebmwebsourcing.geasytools.diagrameditor.api.validation.IDiagramElementViewConformityRule;
import com.ebmwebsourcing.geasytools.diagrameditor.api.validation.IRuleLevel;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.validation.AbstractDiagramElementViewConformityRule;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.validation.RuleLevel;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.IConnectable;
import com.ebmwebsourcing.geasytools.geasyui.api.connectable.IConnector;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.connectors.SequenceFlow;

public class DescriptiveProcessEventValidationRules {
	
	private HashSet<IDiagramElementViewConformityRule> rules;
	
	public DescriptiveProcessEventValidationRules(Event event) {
		
		this.rules = new HashSet<IDiagramElementViewConformityRule>();
	
		if (event instanceof StartEvent){
			
			this.rules.add(new StartEventMustHave1OutgoingSeqFlowAtLeast(event));	
			this.rules.add(new StartEventCannotHaveIncomingSequenceFlow(event));
			
		}else if (event instanceof IntermediateEvent){
			
			this.rules.add(new IntermediateEventMustHaveAtLeast1IncommingSeqFlowAnd1OutgoingSeqFlow(event));
			
		}else if (event instanceof EndEvent){
			
			this.rules.add(new EndEventMustHave1IncommingSeqFlowAtLeast(event));
			this.rules.add(new EndEventCannotHaveOutgoingSequenceFlows(event));
		}
		
	
	}
	
	public HashSet<IDiagramElementViewConformityRule> getRules() {
		return rules;
	}
	
	
	/////////////////////////
	//ERRORS
	////////////
	
	//////////A Start Event must have at least 1 outgoing Sequence Flow
	private class StartEventMustHave1OutgoingSeqFlowAtLeast extends AbstractDiagramElementViewConformityRule {

		public StartEventMustHave1OutgoingSeqFlowAtLeast(IDiagramElementView diagramElementView) {
			super(diagramElementView);
			
		}

		@Override
		public boolean isConform(IDiagramElementView diagramElementView) {
			
			IConnectable connectable = (IConnectable) diagramElementView;
			
			if (connectable.getOutgoingConnectors().size()==0){
				return false;
			}
			
			return true;
		}

		@Override
		public boolean canResolveNonConformity() {
			// TODO Auto-generated method stub
			return false;
		}

		@Override
		public String getResolveConformityDescription() {
			return "Connect actual element to a another flow node with a Sequence Flow";
		}

		@Override
		public String getRuleDescription() {
			return "A Start Event has to be connected to another flow node by a Sequence Flow";
		}

		@Override
		public IRuleLevel getRuleLevel() {
			return RuleLevel.ERROR;
		}

		@Override
		public String getRuleName() {
			return "Outgoing Sequence Flow Required";
		}

		@Override
		public void resolveNonConformity() {
			// TODO Auto-generated method stub
			
		}
		

	}
	

	//////////An End Event must have at least 1 incoming Sequence Flow
	private class EndEventMustHave1IncommingSeqFlowAtLeast extends AbstractDiagramElementViewConformityRule {

		public EndEventMustHave1IncommingSeqFlowAtLeast(IDiagramElementView diagramElementView) {
			super(diagramElementView);
			
		}

		@Override
		public boolean isConform(IDiagramElementView diagramElementView) {
			
			IConnectable connectable = (IConnectable) diagramElementView;
			
			if (connectable.getIncommingConnectors().size()==0){
				return false;
			}
			
			return true;
		}

		@Override
		public boolean canResolveNonConformity() {
			// TODO Auto-generated method stub
			return false;
		}

		@Override
		public String getResolveConformityDescription() {
			return "Connect another flow node to actual element with a Sequence Flow";
		}

		@Override
		public String getRuleDescription() {
			return "An End Event has to be connected to another flow node by a Sequence Flow";
		}

		@Override
		public IRuleLevel getRuleLevel() {
			return RuleLevel.ERROR;
		}

		@Override
		public String getRuleName() {
			return "Incomming Sequence Flow Required";
		}

		@Override
		public void resolveNonConformity() {
			// TODO Auto-generated method stub
			
		}
		

	}
	
	
	//////////An intermediate Event must have 1 incoming sequence flow and 1 outgoing sequence Flow
	private class IntermediateEventMustHaveAtLeast1IncommingSeqFlowAnd1OutgoingSeqFlow extends AbstractDiagramElementViewConformityRule {

		public IntermediateEventMustHaveAtLeast1IncommingSeqFlowAnd1OutgoingSeqFlow(IDiagramElementView diagramElementView) {
			super(diagramElementView);
			
		}

		@Override
		public boolean isConform(IDiagramElementView diagramElementView) {
			
			
			IConnectable connectable = (IConnectable) diagramElementView;
			
			if (connectable.getIncommingConnectors().size()==0 || connectable.getOutgoingConnectors().size()==0){
				return false;
			}
			
			return true;
		}

		@Override
		public boolean canResolveNonConformity() {
			// TODO Auto-generated method stub
			return false;
		}

		@Override
		public String getResolveConformityDescription() {
			return "Add an incomming and and outgoing Sequence Flow";
		}

		@Override
		public String getRuleDescription() {
			return "An Intermediate Event must have an incomming and an outgoing Sequence Flow";
		}

		@Override
		public IRuleLevel getRuleLevel() {
			return RuleLevel.ERROR;
		}

		@Override
		public String getRuleName() {
			return "Incomming and outgoing Sequence Flow required";
		}

		@Override
		public void resolveNonConformity() {
		
		}
		
		
		
		
	}
	

	/////////A START EVENT CANNOT HAVE INCOMING SEQUENCE FLOWS
	private class StartEventCannotHaveIncomingSequenceFlow extends AbstractDiagramElementViewConformityRule {

		public StartEventCannotHaveIncomingSequenceFlow(
				IDiagramElementView diagramElementView) {
			super(diagramElementView);

		}

		@Override
		public boolean isConform(IDiagramElementView diagramElementView) {
			
			IConnectable connectable = (IConnectable) diagramElementView;
			
			if (connectable.getIncommingConnectors().size()>0) {
			    for(IConnector incoming : connectable.getIncommingConnectors()) {
			        if(incoming instanceof SequenceFlow) {
			            return false;
			        }
			    }
			}
			
			return true;
		}

		@Override
		public boolean canResolveNonConformity() {

			return false;
		}

		@Override
		public String getResolveConformityDescription() {
			return "Remove all incoming Sequence Flows";
		}

		@Override
		public String getRuleDescription() {
			return "An Start Event cannot have incoming Sequence Flows";
		}

		@Override
		public IRuleLevel getRuleLevel() {
			return RuleLevel.ERROR;
		}

		@Override
		public String getRuleName() {
			return "Start Event cannot have incoming Sequence Flow";
		}

		@Override
		public void resolveNonConformity() {
			// TODO Auto-generated method stub
			
		}
		
		
		
		
	}
	
	//////////A END EVENT CANNOT HAVE OUTGOING SEQUENCE FLOWS
	private class EndEventCannotHaveOutgoingSequenceFlows extends AbstractDiagramElementViewConformityRule {

		public EndEventCannotHaveOutgoingSequenceFlows(
				IDiagramElementView diagramElementView) {
			super(diagramElementView);

		}

		@Override
		public boolean isConform(IDiagramElementView diagramElementView) {
			
			IConnectable connectable = (IConnectable) diagramElementView;
			
			if (connectable.getOutgoingConnectors().size()>0){
				return false;
			}
			
			return true;
		}

		@Override
		public boolean canResolveNonConformity() {
			return false;
		}

		@Override
		public String getResolveConformityDescription() {
			return "Remove all outgoing Sequence Flows";
		}

		@Override
		public String getRuleDescription() {
			return "An End Event cannot have outgoing Sequence Flows";
		}

		@Override
		public IRuleLevel getRuleLevel() {
			return RuleLevel.ERROR;
		}

		@Override
		public String getRuleName() {
			return "End Event cannot have outgoing Sequence Flow";
		}

		@Override
		public void resolveNonConformity() {

		}
		
		
		
		
		
	}
	
	
	
}
