/**
 * bpmn-diagram - SVG/VML web based editor for BPMN Standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.bpmndiagram;

import java.util.HashSet;

import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramElementView;
import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramView;
import com.ebmwebsourcing.geasytools.diagrameditor.api.validation.IDiagramViewConformityRule;
import com.ebmwebsourcing.geasytools.diagrameditor.api.validation.IRuleLevel;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.validation.AbstractDiagramViewConformityRule;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.validation.RuleLevel;
import com.ebmwebsourcing.geasytools.geasyui.api.core.IUIElement;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.collaboration.CollapsedPool;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.collaboration.Pool;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.common.connector.ConnectorElement;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IParticipantBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.event.EndEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.event.StartEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.IBPMNShape;

public class DescriptiveProcessRules {
	
	
	private HashSet<IDiagramViewConformityRule> rules;
	
	public DescriptiveProcessRules(ProcessPanel processPanel) {
		
		this.rules = new HashSet<IDiagramViewConformityRule>();
		
		this.rules.add(new ProcessCannotBeEmpty(processPanel));
		this.rules.add(new ProcessWithEndEventMustHaveAStartEvent());
		this.rules.add(new ProcessWithStartEventMustHaveAEndEvent());
		this.rules.add(new ProcessShouldEndWithEndEvent());
		this.rules.add(new ProcessShouldStartWithStartEvent());
		this.rules.add(new HavingExpandedProcessExcludeHavingFlowNodesOnProcessPanel(processPanel));
		
		
		
	}
	
	public HashSet<IDiagramViewConformityRule> getRules() {
		return rules;
	}
	
	////////////
	//ERRORS
	///////////////////
	
	private class HavingExpandedProcessExcludeHavingFlowNodesOnProcessPanel extends AbstractDiagramViewConformityRule{

		public HavingExpandedProcessExcludeHavingFlowNodesOnProcessPanel(
				IDiagramView diagramView) {
			super(diagramView);
			// TODO Auto-generated constructor stub
		}

		private boolean hasExpandedPool(IDiagramView diagramView){
			
			for(IUIElement el:diagramView.getUIElements().values()){
				
				IDiagramElementView view = (IDiagramElementView) el;
				
				if (view.getDiagramElement() instanceof IBPMNShape){
					
					IBPMNShape shape = (IBPMNShape) view.getDiagramElement();
					
					if (shape.getModelElement() instanceof IParticipantBean && shape.isExpanded()){
						return true;
					}
					
				}
				
			}
			
			return false;
		}
		
		private boolean hasFlowNodesOnProcessPanel(IDiagramView diagramView){			
			
			for(IUIElement el:diagramView.getChildrenUIElements().values()){
				
				IDiagramElementView view = (IDiagramElementView) el;
				//if it contains something else than a Pool or CollapsedPool and Connector => true
				if ((view instanceof Pool ==false) && (view instanceof CollapsedPool==false) && (view instanceof ConnectorElement==false) && (view.getContainer() instanceof ProcessPanel)){
					return true;
				}
				
			}
			
			return false;
		}
		
		
		@Override
		public boolean isConform(IDiagramView diagramView) {
			
			boolean hasExpandedPool 			= hasExpandedPool(diagramView); 
			
			boolean hasFlowNodesOnProcessPanel 	= hasFlowNodesOnProcessPanel(diagramView); 
			
			if (hasExpandedPool&&hasFlowNodesOnProcessPanel){
				return false;
			}
			
			
			return true;
		}

		@Override
		public boolean canResolveNonConformity() {
			// TODO Auto-generated method stub
			return false;
		}

		@Override
		public String getResolveConformityDescription() {
			return "Put all flow nodes either in a Pool or directly on the Process Panel";
		}

		@Override
		public String getRuleDescription() {
			return "You can design a process in a Pool or directly on the Process Panel, but you cannot do both";
		}

		@Override
		public IRuleLevel getRuleLevel() {
			return RuleLevel.ERROR;
		}

		@Override
		public String getRuleName() {
			return "Having an expanded Pool excludes having flow nodes on Process Panel";
		}

		@Override
		public void resolveNonConformity() {
			// TODO Auto-generated method stub
			
		}
		
		
		
	}
	
	
	
	//////////PROCESS CANNOT BE EMPTY
	public class ProcessCannotBeEmpty extends AbstractDiagramViewConformityRule{

		public ProcessCannotBeEmpty(IDiagramView diagramView) {
			super(diagramView);
		}

		@Override
		public boolean isConform(IDiagramView diagramView) {
			
			if (diagramView.getUIElements().size()==0) return false;
			
			return true;
		}

		@Override
		public boolean canResolveNonConformity() {
			// TODO Auto-generated method stub
			return false;
		}

		@Override
		public String getResolveConformityDescription() {
			return "Add some elements to design a Process";
		}

		@Override
		public String getRuleDescription() {
			return "Actual Process doesn't contain any elements";
		}

		@Override
		public IRuleLevel getRuleLevel() {
			return RuleLevel.ERROR;
		}

		@Override
		public String getRuleName() {
			return "Process cannot be empty";
		}

		@Override
		public void resolveNonConformity() {
			// TODO Auto-generated method stub
			
		}


		
		
	}
	
	
	/////////PROCESS WITH A START EVENT MUST HAVE A END EVENT

	public class ProcessWithStartEventMustHaveAEndEvent implements IDiagramViewConformityRule{
		
		private IDiagramView diagramView;
		
		public IDiagramView getNonConformDiagramView() {
			return diagramView;
		}

		public boolean isConform(IDiagramView diagramView) {
			
			this.diagramView = diagramView;
			
			boolean hasStartEvent = false;
			
			if (diagramView.getUIElements().size()==0) return true; 
			
			for(IUIElement el:diagramView.getUIElements().values()){
				
				IDiagramElementView elView = (IDiagramElementView) el;
				
				if (elView.getDiagramElement().getModelElement() instanceof StartEventBean){
					
					hasStartEvent = true;
					
					break;
					
				}
				
				
			}
			
			//if a start event in present => check the presence of a end event
			if (hasStartEvent){
				

				
				for(IUIElement el:diagramView.getUIElements().values()){
					
					IDiagramElementView elView = (IDiagramElementView) el;
					
					if (elView.getDiagramElement().getModelElement() instanceof EndEventBean){
						
						return true;
						
					}
					
					
				}

			}else {
				
				return true;
			}
			
			
			
			return false;
		}

		public boolean canResolveNonConformity() {
			return false;
		}

		public String getResolveConformityDescription() {
			return null;
		}

		public String getRuleDescription() {
			return "A process having a start event must at least have an end event";
		}

		public IRuleLevel getRuleLevel() {
			return RuleLevel.ERROR;
		}

		public String getRuleName() {
			return "Process must have a end event";
		}

		public void resolveNonConformity() {
			
		}
		
	}	

	/////////PROCESS WITH AN END EVENT MUST HAVE A START EVENT

	public class ProcessWithEndEventMustHaveAStartEvent implements IDiagramViewConformityRule{
		
		private IDiagramView diagramView;
		
		public IDiagramView getNonConformDiagramView() {
			return diagramView;
		}

		public boolean isConform(IDiagramView diagramView) {
			
			this.diagramView = diagramView;
			
			boolean hasEndEvent = false;

			if (diagramView.getUIElements().size()==0) return true;
			
			for(IUIElement el:diagramView.getUIElements().values()){
				
				IDiagramElementView elView = (IDiagramElementView) el;
				
				if (elView.getDiagramElement().getModelElement() instanceof EndEventBean){
					
					hasEndEvent = true;
					
					break;
					
				}
				
				
			}
			
			//if a end event in present => check the presence of a start event
			if (hasEndEvent){
			
				
				for(IUIElement el:diagramView.getUIElements().values()){
					
					IDiagramElementView elView = (IDiagramElementView) el;
					
					if (elView.getDiagramElement().getModelElement() instanceof StartEventBean){
						
						return true;
						
					}
					
					
				}

			}else {
				return true;
			}
			
			
			
			return false;
		}

		public boolean canResolveNonConformity() {
			return false;
		}

		public String getResolveConformityDescription() {
			return null;
		}

		public String getRuleDescription() {
			return "A process having an end event must at least have a start event";
		}

		public IRuleLevel getRuleLevel() {
			return RuleLevel.ERROR;
		}

		public String getRuleName() {
			return "Process must have a start event";
		}

		public void resolveNonConformity() {
			
		}
		
	}	
	
	
	
	
	
	
	////////////
	//WARNINGS
	///////////////////
	
	/////////PROCESS SHOULD WITH START EVENT
	
	public class ProcessShouldStartWithStartEvent implements IDiagramViewConformityRule{
		
		private IDiagramView diagramView;
		
		public IDiagramView getNonConformDiagramView() {
			return diagramView;
		}

		public boolean isConform(IDiagramView diagramView) {
			
			this.diagramView = diagramView;
			

			for(IUIElement el:diagramView.getUIElements().values()){
				
				IDiagramElementView elView = (IDiagramElementView) el;
				
				if (elView.getDiagramElement().getModelElement() instanceof StartEventBean){
					
					return true;
					
				}
				
				
			}
			
			
			return false;
		}

		public boolean canResolveNonConformity() {
			return false;
		}

		public String getResolveConformityDescription() {
			return null;
		}

		public String getRuleDescription() {
			return "It is highly recommended to start the process with a start event";
		}

		public IRuleLevel getRuleLevel() {
			return RuleLevel.WARNING;
		}

		public String getRuleName() {
			return "Process must have a start event";
		}

		public void resolveNonConformity() {
			
		}
		
	}
	
	/////////PROCESS SHOULD END WITH END EVENT
	
	public class ProcessShouldEndWithEndEvent implements IDiagramViewConformityRule{
		
		private IDiagramView diagramView;
		
		public IDiagramView getNonConformDiagramView() {
			return diagramView;
		}

		public boolean isConform(IDiagramView diagramView) {
			
			this.diagramView = diagramView;
			

			for(IUIElement el:diagramView.getUIElements().values()){
				
				IDiagramElementView elView = (IDiagramElementView) el;
				
				if (elView.getDiagramElement().getModelElement() instanceof EndEventBean){
					
					return true;
					
				}
				
				
			}
			
			
			return false;
		}

		public boolean canResolveNonConformity() {
			return false;
		}

		public String getResolveConformityDescription() {
			return null;
		}

		public String getRuleDescription() {
			return "It is highly recommended to end the process with an end event";
		}

		public IRuleLevel getRuleLevel() {
			return RuleLevel.WARNING;
		}

		public String getRuleName() {
			return "Process should have an end event";
		}

		public void resolveNonConformity() {
			
		}
		
	}
	
	
}
