/**
 * bpmn-diagram - SVG/VML web based editor for BPMN Standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.bpmndiagram.builders;


import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramElementView;
import com.ebmwebsourcing.geasytools.diagrameditor.api.graphic.IDiagramView;
import com.ebmwebsourcing.geasytools.diagrameditor.api.modeleditor.IEditorModel;
import com.ebmwebsourcing.geasytools.diagrameditor.api.syntax.IDiagramSyntaxModelBuilder;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IDiagramElement;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IMainModelElement;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IModelElement;
import com.ebmwebsourcing.geasytools.geasyui.api.core.IUIElement;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.ProcessPanel;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.editormodels.PoolEditorModel;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.editormodels.ProcessEditorModel;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.Constants.ProcessTypes;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ICollaborationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ILaneSetBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IMessageFlowBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IParticipantBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.infrastructure.IDefinitionsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.IProcessBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.ICallActivityBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.ITaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataObjectBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IEndEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IIntermediateCatchEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IIntermediateThrowEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IStartEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.gateway.IGatewayBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.gateway.ISequenceFlowBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.service.IInterfaceBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.mock.IdGenerator;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.DefinitionsHelper;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.collaboration.CollaborationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.collaboration.LaneBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.collaboration.LaneSetBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.ParticipantBean;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.IBPMNPlane;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.IBPMNShape;

/**
 * 
 * 
 * @author nfleury
 *
 */
public class BPMNProcessSyntaxModelBuilder implements IDiagramSyntaxModelBuilder{


	private IDiagramView diagramView;
	
	private ArrayList<IInterfaceBean> processInterfaces;
	private ArrayList<IInterfaceBean> expandedPoolInterfaces;
	
	public BPMNProcessSyntaxModelBuilder(IDiagramView diagramView) {
		this.diagramView = diagramView;
	}

	@Override
	public IModelElement getSyntaxModel() {
	
		IDefinitionsBean definitionsBean 				= (IDefinitionsBean) diagramView.getMainModel();
		
		IProcessBean processBean 						= 	(IProcessBean) diagramView.getDiagram().getRootElement().getModelElement();
		
		ProcessPanel descriptiveProcessPanel = (ProcessPanel) diagramView;
		
		ProcessEditorModel processEditorModel 			= (ProcessEditorModel) descriptiveProcessPanel.getEditorModel();
		
		boolean isPublicProcess 						= isPublicProcess();
		
		processBean.setName(processEditorModel.getName());
		processBean.setDocumentation(processEditorModel.getDocumentation());
		
		if (isPublicProcess==false){
			
			processBean.setType(ProcessTypes.PRIVATE);
			
		}else{
			
			processBean.setType(ProcessTypes.PUBLIC);
		
		}
		
		//Add process to actual definition
		if (definitionsBean.getProcesses().contains(processBean)==false) definitionsBean.addProcess(processBean);
		
		
		
		//clear all elements from processBean as the previous ones may have been deleted
		processBean.getCallActivities().clear();
		processBean.getDataObjects().clear();
		processBean.getEndEvents().clear();
		processBean.getGateways().clear();
		processBean.getIntermediateCatchEvents().clear();
		processBean.getIntermediateThrowEvents().clear();
		processBean.getStartEvents().clear();
		processBean.getTasks().clear();
		processBean.getLaneSets().clear();
		processBean.getSequenceFlows().clear();
		

		//Add all flow nodes into process 
		for(IDiagramElement el:diagramView.getDiagram().getRootElement().getOwnedElements()){


		    if (el.getModelElement() instanceof ICallActivityBean){

		        processBean.addCallActivity((ICallActivityBean) el.getModelElement());

		    }else if (el.getModelElement() instanceof IDataObjectBean){

		        processBean.addDataObject((IDataObjectBean) el.getModelElement());

		    }else if (el.getModelElement() instanceof IEndEventBean){

		        processBean.addEndEvent((IEndEventBean) el.getModelElement());

		    }else if (el.getModelElement() instanceof IGatewayBean){

		        processBean.addGateway((IGatewayBean) el.getModelElement());

		    }else if (el.getModelElement() instanceof IIntermediateCatchEventBean){

		        processBean.addIntermediateCatchEvent((IIntermediateCatchEventBean) el.getModelElement());

		    }else if (el.getModelElement() instanceof IIntermediateThrowEventBean){

		        processBean.addIntermediateThrowEvent((IIntermediateThrowEventBean) el.getModelElement());

		    }else if (el.getModelElement() instanceof IStartEventBean){

		        processBean.addStartEvent((IStartEventBean) el.getModelElement());

		    }else if (el.getModelElement() instanceof ITaskBean){

		        processBean.addTask((ITaskBean) el.getModelElement());

		    }else if (el.getModelElement() instanceof ISequenceFlowBean){

		        processBean.addSequenceFlow((ISequenceFlowBean)el.getModelElement());

		    }else if (el.getModelElement() instanceof IParticipantBean){

		        //add every lane sets from actual participant

		        processBean.getLaneSets().addAll(getLaneSets(el));


		    }

		}


		//Create collaboration
		//Event if actual process is not public we have a definitional collaboration => actual process
		//may represent a participant in the future


		//=> if a collaboration with actual process doesn't exist yet , create it	
		ICollaborationBean definitionalCollaboration = processBean.getDefinitionalCollaboration(); 

		if (definitionalCollaboration==null){

		    definitionalCollaboration = new CollaborationBean(IdGenerator.createUniqueId());
		    definitionsBean.addCollaboration(definitionalCollaboration);
		    processBean.setDefinitionalCollaboration(definitionalCollaboration);
		}

		//finally add all participants of actual process after removing the existing ones
		definitionalCollaboration.getParticipants().clear();
		definitionalCollaboration.getMessageFlows().clear();
		
		for(IMessageFlowBean mf : getMessageFlows()) {
		    if(!definitionalCollaboration.getMessageFlows().contains(mf)) {
		        definitionalCollaboration.addMessageFlow(mf);
		    }
		}

		for(IParticipantBean participant:getParticipants()){

		    if (definitionalCollaboration.getParticipants().contains(participant)==false){
		        definitionalCollaboration.addParticipant(participant);
		    }

		}

	
	   //If no expanded pool exists (meaning all process was designed on BPMN Plane) we should'nt have
	   //any Participant yet that represents our actual process  => we need to create one
	   
 	   //we dont have any expanded pool
		IDiagramElementView expandedPool = getExpandedPool(); 
		
       if (expandedPool==null){

    	   //if an expanded pool already existed previously
    	   //grab all its interfaces and put it in actual participant
		   if (expandedPoolInterfaces!=null){
			   
			   processEditorModel.setInterfaces(expandedPoolInterfaces);
			   
		   }
    	   
    	   //create a participant
    	   ParticipantBean participant = new ParticipantBean(IdGenerator.createUniqueId());
    	   participant.setInterfaces(processEditorModel.getInterfaces());
    	   participant.setProcess(processBean);
    	   
    	   //add participant to collaboration
    	   definitionalCollaboration.addParticipant(participant);
    	   

		   this.expandedPoolInterfaces = null;
	   
       }else{
           //we do have an expanded pool 
           IParticipantBean p = (IParticipantBean) expandedPool.getDiagramElement().getModelElement();
           p.setProcess(processBean);
           
    	   //if no expanded process existed previously 
    	   //grab the "process/participant" interfaces and put it in actual participant 
    	   if (processInterfaces!=null){
    		   
    		  PoolEditorModel expandedPoolEditorModel = (PoolEditorModel) expandedPool.getEditorModel();
    		  expandedPoolEditorModel.setInterfaces(processInterfaces);
    		  
    	   }
    	   
		   processInterfaces = null;
	   }
		
		
		
	return processBean;
	}
	
	
	
	private List<ILaneSetBean> getLaneSets(IDiagramElement laneContainer){
		
		List<ILaneSetBean> result = new ArrayList<ILaneSetBean>();
		
		//if actual laneContainer is a Pool => create a laneSet and
		//add every lanes in it
		if (laneContainer.getModelElement() instanceof IParticipantBean){
			
			LaneSetBean laneSetBean = new LaneSetBean(IdGenerator.createUniqueId());
			
			for(IDiagramElement lane:laneContainer.getOwnedElements()){
				
				if (lane.getModelElement() instanceof LaneBean){
					
					LaneBean laneBean = (LaneBean) lane.getModelElement();
					
					laneSetBean.addLane(laneBean);
					
					result.addAll(getLaneSets(lane));
				
				}
				
			}
			
			result.add(laneSetBean);
			
		}else{
			
			for(IDiagramElement lane:laneContainer.getOwnedElements()){
				
				if (lane.getModelElement() instanceof LaneBean){
					
					LaneBean laneBean = (LaneBean) lane.getModelElement();
				
					result.add(laneBean.getChildLaneSet());
				
				}
			}			
			
		}
		

		
		
		return result;
	}
	
	
	
	/**
	 * If actual process contains 1 collapsed pool(participant) then its a public process
	 */
	private boolean isPublicProcess(){
		
		IBPMNPlane bpmnPlane = (IBPMNPlane) diagramView.getDiagram().getRootElement();
		
		for(IDiagramElement el:bpmnPlane.getOwnedElements()){
			
			if (el instanceof IBPMNShape){
				IBPMNShape shape = (IBPMNShape) el;
				
				if (shape.getModelElement() instanceof IParticipantBean && shape.isExpanded()==false){
					return true;
				}
			}
		}
		
		return false;
	}
	
	private Set<IParticipantBean> getParticipants(){
		
		HashSet<IParticipantBean> result = new HashSet<IParticipantBean>();
		
		IBPMNPlane bpmnPlane = (IBPMNPlane) diagramView.getDiagram().getRootElement();
		
		for(IDiagramElement el:bpmnPlane.getOwnedElements()){
			
			if (el instanceof IBPMNShape){
				IBPMNShape shape = (IBPMNShape) el;
				
				if (shape.getModelElement() instanceof IParticipantBean){
					result.add((IParticipantBean) shape.getModelElement());
				}
			}
		}
		
		return result;
	}
	
	private Set<IMessageFlowBean> getMessageFlows() {
        HashSet<IMessageFlowBean> result = new HashSet<IMessageFlowBean>();
        for(IUIElement el:diagramView.getUIElements().values()){
            IDiagramElementView elView = (IDiagramElementView) el;
            if ((elView.getDiagramElement().getModelElement() instanceof IMessageFlowBean)){
                result.add((IMessageFlowBean) elView.getDiagramElement().getModelElement());
            }
        }
        return result;
	}
	
	/**
	 * Retrieves the expanded pool if one exists
	 * @return
	 */
	private IDiagramElementView getExpandedPool(){
		
		
		for(IUIElement el:diagramView.getUIElements().values()){
			
			IDiagramElementView elView = (IDiagramElementView) el;
			
			if ((elView.getDiagramElement().getModelElement() instanceof IParticipantBean)){
				
				if (elView.getDiagramElement() instanceof IBPMNShape){
					
					IBPMNShape shape = (IBPMNShape) elView.getDiagramElement();
					
					if (shape.isExpanded()) return elView;
					
				}
				
			}
			
			
		}
		
		
		return null;
	}
	
	@Override
	public IDiagramView getDiagramView() {
		return diagramView;
	}

	@Override
	public void initializeEditorModel(IEditorModel editorModel,
			IModelElement modelElement, IMainModelElement mainModelElement) {
		
		ProcessEditorModel processEditorModel = (ProcessEditorModel) editorModel;
		IProcessBean processBean = (IProcessBean) modelElement;
		CollaborationBean collaboration = (CollaborationBean) processBean.getDefinitionalCollaboration();
		
		//retrieve the participant of actual process
		IParticipantBean participant = collaboration.getParticipantByProcess(processBean);
		
		processEditorModel.setName(processBean.getName());
		processEditorModel.setDocumentation(processBean.getDocumentation());
		if (participant!=null) processEditorModel.setInterfaces((ArrayList<IInterfaceBean>) participant.getInterfaces());
	
	}
	

}
