/**
 * bpmn-diagram - SVG/VML web based editor for BPMN Standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.ebmwebsourcing.petalsbpm.bpmndiagram.model.builder;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IDiagramElement;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IEdge;
import com.ebmwebsourcing.geasytools.diagrameditor.impl.syntax.SyntaxModelBuilder;
import com.ebmwebsourcing.geasytools.modeleditor.modelmanager.client.modelbinder.IWatchedModelProxy;
import com.ebmwebsourcing.petalsbpm.bpmndiagram.editor.model.swimlane.PoolEditorModel;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.IInteractionNodeBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ILaneBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IFlowElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IFlowNodeBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IParticipantBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.artifact.IArtifactBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.foundation.IBaseElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.IActivityBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IAssignmentBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataInputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataOutputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IExpressionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IItemAwareElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.WithDataInputAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.WithDataOutputAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.ICatchEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IThrowEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.collaboration.CollaborationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.collaboration.LaneBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.FlowElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.MessageBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.MessageFlowBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.ParticipantBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.artifact.ArtifactBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.artifact.AssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.data.DataAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.data.DataInputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.data.DataOutputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.data.ExpressionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.gateway.SequenceFlowBean;


public class CollaborationBuilder extends SyntaxModelBuilder<CollaborationBean>{

	private IParticipantBean defaultParticipant;
	private ILaneBean defaultLane;
	private PoolEditorModel defaultParticipantEditorModel;

	public CollaborationBuilder(IDiagramElement diagramElement,PoolEditorModel defaultParticipantEditorModel) {
		super(diagramElement);
		this.defaultParticipantEditorModel = defaultParticipantEditorModel;
		initBuilders();
	}

	private void initBuilders() {
		this.register(new ParticipantBuilder());
	}

	private void initModel(){
		IWatchedModelProxy<CollaborationBean> proxy = (IWatchedModelProxy<CollaborationBean>) defaultParticipantEditorModel;

		CollaborationBean collaboration = proxy.getBindedModel();
		this.defaultParticipant = DefaultParticipants.getDefaultParticipant(collaboration);
		this.defaultLane		= new LaneBean();

		this.defaultParticipant.getProcess().getLanes().clear();
		this.defaultParticipant.getProcess().addLane(defaultLane);
	}

	@Override
	public CollaborationBean getSyntaxModel() {
		initModel();

		CollaborationBean collaboration = (CollaborationBean) getMainDiagramElement().getModelElement();

		//remove the participants and process that migth have been added during a previous step of modeling
		if(collaboration.getParticipants()!=null) collaboration.getParticipants().clear();
		if(collaboration.getParentDefinitions()!=null) {
			if(collaboration.getParentDefinitions().getProcesses()!=null) collaboration.getParentDefinitions().getProcesses().clear();
		}

		//add participants 
		Set<ParticipantBean> participants = this.getAll(ParticipantBean.class,true);

		boolean foundPool = false;
		for(ParticipantBean p:participants){
			if(p.getProcess()!=null) {
				foundPool = foundPool || !p.getProcess().getFlowNodes().isEmpty();
			}
		}


		//if we don't have any participants (Pool) => flow elements may have been dropped 
		//directly on BPMN Plane
		if (participants.size()==0 || !foundPool){

			collaboration.addParticipant(defaultParticipant);

			//check for flow elements directly on plane
			HashSet<FlowElementBean> flowElements = (HashSet<FlowElementBean>) this.getAll(FlowElementBean.class, false);


			defaultParticipant.getProcess().clearFlowElements();
			defaultParticipant.getProcess().clearIOSpec();


			for(FlowElementBean fe:flowElements){
				defaultLane.addFlowElement(fe);
			}

			//grab all sequence flows
			HashMap<IDiagramElement,SequenceFlowBean> sequenceFlows = (HashMap<IDiagramElement, SequenceFlowBean>) this.getEdges(SequenceFlowBean.class);
			for(IDiagramElement di:sequenceFlows.keySet()){

				SequenceFlowBean s = sequenceFlows.get(di);
				s.setSourceNode((IFlowElementBean) ((IEdge)di).getSource().getModelElement());
				s.setTargetNode((IFlowElementBean) ((IEdge)di).getTarget().getModelElement());

				defaultParticipant.getProcess().addSequenceFlow(s);
			}

		}
		
		for(ParticipantBean p:participants){
			collaboration.addParticipant(p);
		}

		//get all dataAssociations
		HashMap<IDiagramElement,DataAssociationBean> dataAssociations = (HashMap<IDiagramElement, DataAssociationBean>) this.getEdges(DataAssociationBean.class);
		for(IDiagramElement di:dataAssociations.keySet()){
			IEdge edge = (IEdge) di; 

			//DATA / ACTIVITY
			if (edge.getTarget().getModelElement() instanceof IItemAwareElementBean
					&& edge.getSource().getModelElement() instanceof WithDataOutputAssociationBean){

				//target is DATA
				IItemAwareElementBean data = (IItemAwareElementBean) edge.getTarget().getModelElement();
				IFlowNodeBean source = (IFlowNodeBean) edge.getSource().getModelElement();

				IDataOutputBean dob = getDataOutput(source, "DataOuput_"+source.getId()+"_"+data.getId());

				//set itemDefinition on dataOutput
				IDataAssociationBean dab = (IDataAssociationBean)di.getModelElement(); 

				dob.setItemSubject(dab.getSourceItem());

				addDataOutput(source,data,dob);
				((WithDataOutputAssociationBean)source).addDataOutputAssociation(dab);

				List<IItemAwareElementBean> sources = new ArrayList<IItemAwareElementBean>();
				sources.add(dob);
				dab.setTarget(data);
				dab.setSources(sources);

				addDefaultAsssignmentExpressions(dab);
			}else if (edge.getSource().getModelElement() instanceof IItemAwareElementBean
					&& edge.getTarget().getModelElement() instanceof IFlowNodeBean){

				//source is DATA
				IItemAwareElementBean data = (IItemAwareElementBean) edge.getSource().getModelElement();
				IFlowNodeBean target = (IFlowNodeBean) edge.getTarget().getModelElement();

				IDataInputBean dib = getDataInput(target, "DataInput_"+target.getId()+"_"+data.getId());

				//set itemDefinition on dataInput
				IDataAssociationBean dab = (IDataAssociationBean)di.getModelElement();
				dib.setItemSubject(dab.getTargetItem());

				addDataInput(target,data,dib);
				((WithDataInputAssociationBean)target).addDataInputAssociation(dab);

				List<IItemAwareElementBean> sources = new ArrayList<IItemAwareElementBean>();
				sources.add(data);
				dab.setSources(sources);
				dab.setTarget(dib);

				addDefaultAsssignmentExpressions(dab);
			}

		}

		//add message flows
		Map<IDiagramElement,MessageFlowBean> messageFlows = this.getEdges(MessageFlowBean.class);
		for(IDiagramElement di : messageFlows.keySet()){
			MessageFlowBean mf = messageFlows.get(di);
			mf.setSource((IInteractionNodeBean) ((IEdge)di).getSource().getModelElement());
			mf.setTarget((IInteractionNodeBean) ((IEdge)di).getTarget().getModelElement());
			collaboration.addMessageFlow(mf);
		}

		////////////MESSAGE
		Set<MessageBean> messages = this.getAll(MessageBean.class, false);
		for(MessageBean m:messages){
			collaboration.addMessage(m);
		}

		//////////////////ARTIFACTS
		//Get all associations
		Map<IDiagramElement,AssociationBean> associations = this.getEdges(AssociationBean.class);
		//assign association source / target
		for(IDiagramElement el:associations.keySet()){
			IEdge edge = (IEdge) el;
			AssociationBean assoc = associations.get(el);
			assoc.setSource((IBaseElementBean) edge.getSource().getModelElement());
			assoc.setTarget((IBaseElementBean) edge.getTarget().getModelElement());
		}

		collaboration.getArtifacts().clear();
		//add associations to collaboration
		for(AssociationBean a:associations.values()){
			collaboration.addArtifact(a);
		}

		//GET ALL ARTIFACTS => add them to collaboration
		Set<ArtifactBean> artifacts = this.getAll(ArtifactBean.class, false);

		for(IArtifactBean a:artifacts){
			collaboration.addArtifact(a);
		}

		return collaboration;
	}

	private IDataOutputBean getDataOutput(IFlowNodeBean fn, String defaultId) {
		if(fn instanceof IActivityBean) {
			IActivityBean acti = (IActivityBean) fn;
			if(!acti.getIoSpecification().getDataOutputs().isEmpty()) {
				return acti.getIoSpecification().getDataOutputs().get(0);
			}
		}
		else if(fn instanceof ICatchEventBean) {
			ICatchEventBean event = (ICatchEventBean) fn;
			if(!event.getDataOutputs().isEmpty()) {
				return event.getDataOutputs().get(0);
			}
		}
		return new DataOutputBean(defaultId);
	}

	private IDataInputBean getDataInput(IFlowNodeBean fn, String defaultId) {
		if(fn instanceof IActivityBean) {
			IActivityBean acti = (IActivityBean) fn;
			if(!acti.getIoSpecification().getDataInputs().isEmpty()) {
				return acti.getIoSpecification().getDataInputs().get(0);
			}
		}
		else if(fn instanceof IThrowEventBean) {
			IThrowEventBean event = (IThrowEventBean) fn;
			if(!event.getDataInputs().isEmpty()) {
				return event.getDataInputs().get(0);
			}
		}
		return new DataInputBean(defaultId);
	}

	private void addDataOutput(IFlowNodeBean fn, IItemAwareElementBean target, IDataOutputBean dob) {
		if(fn instanceof IActivityBean) {
			((IActivityBean) fn).addDataOutput(dob, target);
		}
		else if(fn instanceof ICatchEventBean) {
			((ICatchEventBean) fn).addDataOutput(dob, target);
		}
	}

	private void addDataInput(IFlowNodeBean fn, IItemAwareElementBean source, IDataInputBean dib) {
		if(fn instanceof IActivityBean) {
			((IActivityBean) fn).addDataInput(source, dib);
		}
		else if(fn instanceof IThrowEventBean) {
			((IThrowEventBean) fn).addDataInput(source, dib);
		}
	}

	private void addDefaultAsssignmentExpressions(IDataAssociationBean dab) {
		if(dab.getAssignements().isEmpty()) return;
		IAssignmentBean a = dab.getAssignements().get(0);
		if(a.getFrom()==null || a.getFrom().getContent()==null || a.getFrom().getContent().isEmpty()) {
			IExpressionBean fromExp = new ExpressionBean();
			String fromExpContent = "$"+dab.getSources().get(0).getId();
			fromExp.setContent(fromExpContent);
			a.setFrom(fromExp);
		}
		if(a.getTo()==null || a.getTo().getContent()==null || a.getTo().getContent().isEmpty()) {
			IExpressionBean toExp = new ExpressionBean();
			String toExpContent = "$"+dab.getTarget().getId();
			toExp.setContent(toExpContent);
			a.setTo(toExp);
		}
	}


	@Override
	public Class<? extends CollaborationBean> getSyntaxModelType() {
		return CollaborationBean.class;
	}

}
