/**
 * sa-deployer - A Web based designer for BPMN 2.0 standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
        Petals Webconsole - Copyright (c) 2010 EBM Websourcing,
        http://www.ebmwebsourcing.com/ This library is free software; you can
        redistribute it and/or modify it under the terms of the GNU Lesser
        General Public License as published by the Free Software Foundation;
        either version 2.1 of the License, or (at your option) any later
        version. This library is distributed in the hope that it will be
        useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
        Lesser General Public License for more details. You should have
        received a copy of the GNU Lesser General Public License along with
        this library; if not, write to the Free Software Foundation, Inc., 59
        Temple Place, Suite 330, Boston, MA 02111-1307 USA Initial
        developer(s): EBM WebSourcing
 */

package com.ebmwebsourcing.sa.deployer;

import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.management.MBeanAttributeInfo;
import javax.management.ObjectName;

import org.ow2.petals.jmx.AdminServiceClient;
import org.ow2.petals.jmx.ComponentClient;
import org.ow2.petals.jmx.ConfigurationInstallerComponentClient;
import org.ow2.petals.jmx.DeploymentServiceClient;
import org.ow2.petals.jmx.EndpointRegistryClient;
import org.ow2.petals.jmx.InstallationServiceClient;
import org.ow2.petals.jmx.InstallerComponentClient;
import org.ow2.petals.jmx.JMXClient;
import org.ow2.petals.jmx.LoggerServiceClient;
import org.ow2.petals.jmx.PetalsAdminServiceClient;
import org.ow2.petals.jmx.RuntimeConfigurationComponentClient;
import org.ow2.petals.jmx.exception.AdminDoesNotExistException;
import org.ow2.petals.jmx.exception.AdminServiceErrorException;
import org.ow2.petals.jmx.exception.AttributeErrorException;
import org.ow2.petals.jmx.exception.ComponentDoesNotExistException;
import org.ow2.petals.jmx.exception.ComponentErrorException;
import org.ow2.petals.jmx.exception.ConnectionErrorException;
import org.ow2.petals.jmx.exception.DeploymentServiceDoesNotExistException;
import org.ow2.petals.jmx.exception.DeploymentServiceErrorException;
import org.ow2.petals.jmx.exception.EndpointRegistryServiceDoesNotExistException;
import org.ow2.petals.jmx.exception.EndpointRegistryServiceErrorException;
import org.ow2.petals.jmx.exception.InstallationServiceDoesNotExistException;
import org.ow2.petals.jmx.exception.InstallationServiceErrorException;
import org.ow2.petals.jmx.exception.InstallerComponentDoesNotExistException;
import org.ow2.petals.jmx.exception.LoggerDoesNotExistException;
import org.ow2.petals.jmx.exception.LoggerServiceErrorException;
import org.ow2.petals.jmx.exception.PerformActionErrorException;
import org.ow2.petals.jmx.exception.PetalsAdminDoesNotExistException;
import org.ow2.petals.jmx.exception.PetalsAdminServiceErrorException;
import org.ow2.petals.jmx.exception.RuntimeConfigurationDoesNotExistException;
import org.ow2.petals.jmx.exception.RuntimeConfigurationErrorException;

/**
 * @author aruffie this class provide a management service implementation, thank
 *         to JMX technology
 */
public class ManagementServiceJMXImpl  {

    private final static Logger LOGGER = Logger.getLogger(ManagementServiceJMXImpl.class.toString());

    private JMXClient jmxClient;

    private String loadedInstaller;

    private String hostname;

    private Integer port;

    private String login;

    private String password;

    ManagementServiceJMXImpl() {

    }

    /**
     * Allow to set jmx client for this monitoring service implementation. This
     * implementation being based on JMX one client is required
     * 
     * @param jmxClient
     */
    public void setJmxClient(final JMXClient jmxClient) {
        this.jmxClient = jmxClient;
    }

    /**
     * Allow to start the specified component all service assemblies
     * 
     * @param componentName
     *            the name of the specified component, which must be start.
     * @throws PetalsServiceFunctionalException
     * @throws <PetalsServiceException>
     */
    public void startComponent(final String componentName) throws PetalsServiceTechnicalException,
            PetalsServiceFunctionalException {

        try {

            // check component existence
            final ComponentClient cc = this.getComponentClient(componentName);

            if (ComponentClient.State.STARTED.equals(cc.getState())) {
                LOGGER.warning("Component: " + componentName + " already started");
                throw new PetalsServiceFunctionalException("Component: " + componentName
                        + " already started");
            }

            // Start the component if this state is shutdown or stopped
            if ((ComponentClient.State.SHUTDOWN.equals(cc.getState()))
                    || (ComponentClient.State.STOPPED.equals(cc.getState()))) {
                cc.start();
            } else {
                LOGGER.warning("Can't perform start action, because the component: " + componentName
                        + " is in the following state: " + cc.getState());
                throw new PetalsServiceFunctionalException(
                        "Can't perform start action, because the component: " + componentName
                                + " is in the following state: " + cc.getState());
            }

        } catch (final ComponentErrorException e) {
            LOGGER.severe("Error occurred during " + componentName + " component starting");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    /**
     * Allow to stop the specified component all service assemblies
     * 
     * @param componentName
     *            the name of the specified component, which must be stop.
     * @throws PetalsServiceFunctionalException
     * @throws <PetalsServiceException>
     */
    public void stopComponent(final String componentName) throws PetalsServiceTechnicalException,
            PetalsServiceFunctionalException {
        try {

            // check component existence
            final ComponentClient cc = this.getComponentClient(componentName);

            if (ComponentClient.State.STARTED.equals(cc.getState())) {
                cc.stop();
            } else {
                LOGGER.warning("Can't stop the component: " + componentName
                        + " because, is it in the following state: " + cc.getState());
                throw new PetalsServiceFunctionalException("Can't stop the component: "
                        + componentName + " because, is it in the following state: "
                        + cc.getState());
            }

            /*
             * Wrap all exception type in PetalsServiceException, to pick up the
             * same exception type
             */
        } catch (final ComponentErrorException e) {
            LOGGER.severe("Error occurred during " + componentName + " component stoping");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    public void shutdownComponent(final String componentName)
            throws PetalsServiceTechnicalException, PetalsServiceFunctionalException {
        try {

            // check component existence
            final ComponentClient cc = this.getComponentClient(componentName);

            if (ComponentClient.State.STOPPED.equals(cc.getState())) {
                cc.shutdown();
            } else {
                LOGGER.warning("Can't shutdown the component: " + componentName
                        + " because, is it in the following state: " + cc.getState());
                throw new PetalsServiceFunctionalException("Can't shutdown the component: "
                        + componentName + " because, is it in the following state: "
                        + cc.getState());
            }
        } catch (final ComponentErrorException e) {
            LOGGER.severe("Error occurred during " + componentName + " component shutdown");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    /**
     * Allow to recover all components
     * 
     * @return <ObjectName[]>
     * @throws <PetalsServiceException>
     */
    public List<ObjectName> componentsInventory() throws PetalsServiceTechnicalException {

        final AdminServiceClient asc = this.getAdminServiceClient();

        try {
            final ObjectName[] bcs = asc.getBindingComponents();
            LOGGER.finest(bcs.length + " binding components recovered");
            
            final ObjectName[] ses = asc.getEngineComponents();
            LOGGER.finest(bcs.length + " engine components recovered");
            
            List<ObjectName> res = new ArrayList<ObjectName>();
            res.addAll(Arrays.asList(bcs));
            res.addAll(Arrays.asList(ses));
            return res;
        } catch (final AdminServiceErrorException e) {
            LOGGER.severe("Error occurred during binding component inventory");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    public ObjectName[] enginesInventory() throws PetalsServiceTechnicalException {

        final AdminServiceClient asc = this.getAdminServiceClient();

        try {
            final ObjectName[] ses = asc.getEngineComponents();
            LOGGER.finest(ses.length + " service engines recovered");
            return ses;
        } catch (final AdminServiceErrorException e) {
            LOGGER.severe("Error occurred during service engine inventory");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    /**
     * Allow to recover all service assemblies
     * 
     * @return <String[]>
     * @throws <PetalsServiceException>
     */
    public String[] serviceAssembliesInventory() throws PetalsServiceTechnicalException {

        final DeploymentServiceClient dsc = this.getDeploymentServiceClient();

        try {
            final String[] sas = dsc.getDeployedServiceAssemblies();
            LOGGER.finest(sas.length + " service assemblies recovered");
            return sas;
        } catch (final DeploymentServiceErrorException e) {
            LOGGER.severe("Error occurred during service assemblies inventory");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    /**
     * Allow to recover all shared libraries
     * 
     * @return <String[]>
     * @throws <PetalsServiceException>
     */
    public String[] sharedLibrariesInventory() throws PetalsServiceTechnicalException {

        final InstallationServiceClient isc = this.getInstallationServiceClient();

        try {
            final String[] sls = isc.getInstalledSharedLibraries();
            LOGGER.finest(sls.length + " shared libraries recovered");
            return sls;
        } catch (final InstallationServiceErrorException e) {
            LOGGER.severe("Error occurred during shared libraries inventory");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    /**
     * Allow to install the specified component file provided with parameter
     * 
     * @param fileURL
     *            The URL of the component file
     * @param attributes
     *            The attribute infos map linked to this installation
     * @param floadedInstaller
     *            The loaded installer name for this component (allow to
     *            retrieve the installation loader into Petals side)
     * @throws <PetalsServiceException>
     */
    public void installComponent(final URL fileURL,
            final Map<MBeanAttributeInfo, Object> attributes, final String loadedInstaller)
            throws PetalsServiceTechnicalException {

        final InstallationServiceClient isc = this.getInstallationServiceClient();

        try {
            final InstallerComponentClient icc = isc.loadInstaller(loadedInstaller);

            if (icc == null) {
                LOGGER.warning(loadedInstaller
                        + " installer not correctly loaded during component installation");
                throw new PetalsServiceTechnicalException(loadedInstaller
                        + " installer not correctly loaded component installation");
            }
            LOGGER.finest(loadedInstaller + " installer correctly loaded component installation");
            if (!attributes.isEmpty()) {
                final ConfigurationInstallerComponentClient ccic = icc
                        .getConfigurationInstallerClient();
                try {
                    ccic.setAttributes(attributes);
                } catch (final AttributeErrorException e) {
                    LOGGER.severe("Error occurred during the getting of configuration installer client for: "
                            + loadedInstaller + " installer");
                    throw new PetalsServiceTechnicalException(e);
                }
            }
            icc.install();
            LOGGER.finest("Component for " + loadedInstaller
                    + " loaded installer correctly installed");
        } catch (final InstallationServiceErrorException e) {
            LOGGER.severe("Error occurred during component installation for: " + loadedInstaller
                    + " installer");
            throw new PetalsServiceTechnicalException(e);
        } catch (final PerformActionErrorException e) {
            LOGGER.severe("Error occurred during component installation for: " + loadedInstaller
                    + " installer");
            throw new PetalsServiceTechnicalException(e);
        } catch (InstallerComponentDoesNotExistException e) {
           throw new PetalsServiceTechnicalException(e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.petals.tools.webconsole.services.management.ManagementService
     * #installServiceAssembly(java.net.URL)
     */
    public String installServiceAssembly(final URL saURL) throws PetalsServiceTechnicalException {

        final DeploymentServiceClient dsc = this.getDeploymentServiceClient();

        try {
            final String result = dsc.deploy(saURL);
            LOGGER.finest("Result: " + result + " for service assembly deployment of " + saURL);
            return result;
        } catch (final DeploymentServiceErrorException e) {
            LOGGER.severe("Error occurred during service assembly deployment for: " + saURL);
            throw new PetalsServiceTechnicalException(e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.petals.tools.webconsole.services.management.ManagementService
     * #installSharedLibrary(java.net.URL)
     */
    public String installSharedLibrary(final URL slURL) throws PetalsServiceTechnicalException {

        final InstallationServiceClient isc = this.getInstallationServiceClient();

        try {
            final String result = isc.installSharedLibrary(slURL);
            LOGGER.finest("Result: " + result + " for shared library installation of " + slURL);
            return result;
        } catch (final InstallationServiceErrorException e) {
            LOGGER.severe("Error occurred during shared library  installation for: " + slURL);
            throw new PetalsServiceTechnicalException(e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.petals.tools.webconsole.services.management.ManagementService
     * #startServiceAssembly(java.lang.String)
     */
    public void startServiceAssembly(final String saName) throws PetalsServiceTechnicalException,
            PetalsServiceFunctionalException {

        final DeploymentServiceClient dsc = this.getDeploymentServiceClient();

        try {

            if (dsc.isServiceAssemblyDeployed(saName)) {
                // Check its state to know if we can start if
                if (DeploymentServiceClient.State.SHUTDOWN.equals(dsc.getState(saName))
                        || DeploymentServiceClient.State.STOPPED.equals(dsc.getState(saName))) {
                    dsc.start(saName);
                    LOGGER.finest(saName + " service assembly correclty started");
                } else {
                    if (DeploymentServiceClient.State.STARTED.equals(dsc.getState(saName))) {
                        // SA already started
                        LOGGER.warning(saName + " service-assembly already started");
                        throw new PetalsServiceFunctionalException(saName
                                + " service-assembly already started");
                    } else {
                        LOGGER.warning("Can't start: " + saName + " service assembly, "
                                + "because is in the following state: " + dsc.getState(saName));
                        throw new PetalsServiceFunctionalException("Can't start: " + saName
                                + " service assembly, " + "because is in the following state: "
                                + dsc.getState(saName));
                    }
                }
            } else {
                LOGGER.warning(saName + " service-assembly is not deployed");
                throw new PetalsServiceFunctionalException(saName
                        + " service-assembly is not deployed.");
            }
        } catch (final DeploymentServiceErrorException e) {
            LOGGER.severe("Error occured during the start action of " + saName + " service assembly");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.petals.tools.webconsole.services.management.ManagementService
     * #stopServiceAssembly(java.lang.String)
     */
    public void stopServiceAssembly(final String saName) throws PetalsServiceTechnicalException,
            PetalsServiceFunctionalException {

        final DeploymentServiceClient dsc = this.getDeploymentServiceClient();

        try {

            if (dsc.isServiceAssemblyDeployed(saName)) {
                // check its state to know if we can stop it
                if (DeploymentServiceClient.State.STARTED.equals(dsc.getState(saName))) {
                    dsc.stop(saName);
                    LOGGER.finest(saName + " service assembly correclty stopped");
                } else {
                    LOGGER.warning("Can't stop " + saName + " service assembly, "
                            + "because is in the following state: " + dsc.getState(saName));
                    throw new PetalsServiceFunctionalException("Can't stop " + saName
                            + " service assembly, " + "because is in the following state: "
                            + dsc.getState(saName));
                }
            } else {
                LOGGER.warning("Can't perform stop action, because the service-" + "assembly: "
                        + saName + " is not deployed.");
                throw new PetalsServiceFunctionalException(
                        "Can't perform stop action, because the service-" + "assembly: " + saName
                                + " is not deployed.");
            }
        } catch (final DeploymentServiceErrorException e) {
            LOGGER.severe("Error occured during the stop action of " + saName + " service assembly");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.petals.tools.webconsole.services.management.ManagementService
     * #uninstallComponent(java.lang.String)
     */
    public void uninstallComponent(final String componentName)
            throws PetalsServiceTechnicalException {

        final InstallationServiceClient isc = this.getInstallationServiceClient();

        try {
            final InstallerComponentClient icc = isc.loadInstaller(componentName);

            if (icc == null) {
                LOGGER.warning(componentName
                        + " installer not correctly loaded for component uninstallation");
                throw new PetalsServiceTechnicalException(componentName
                        + " installer not correctly loaded for component uninstallation");
            }
            LOGGER.finest(componentName + " installer correctly loaded for component uninstallation");
            if (icc.isInstalled()) {
                icc.uninstall();
                LOGGER.finest(componentName + " component correctly uninstalled");
            }
            isc.unloadInstaller(componentName);
            LOGGER.finest(componentName
                    + " installer correctly unloaded for component uninstallation");
        } catch (final InstallationServiceErrorException e) {
            LOGGER.severe("Error occurred during " + componentName + "component uninstallation");
            throw new PetalsServiceTechnicalException(e);
        } catch (final PerformActionErrorException e) {
            LOGGER.severe("Error occurred during " + componentName + " component uninstallation");
            throw new PetalsServiceTechnicalException(e);
        } catch (InstallerComponentDoesNotExistException e) {
            throw new PetalsServiceTechnicalException(e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.petals.tools.webconsole.services.management.ManagementService
     * #uninstallServiceAssembly(java.lang.String)
     */
    public String uninstallServiceAssembly(final String saName)
            throws PetalsServiceTechnicalException {

        final DeploymentServiceClient dsc = this.getDeploymentServiceClient();

        try {
            final String result = dsc.undeploy(saName);
            LOGGER.finest("Result: " + result + " for service assembly uninstallation of: " + saName);
            return result;
        } catch (final DeploymentServiceErrorException e) {
            LOGGER.severe("Error occurred during " + saName + " service assembly uninstallation");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    public boolean uninstallSharedLibrary(final String slName)
            throws PetalsServiceTechnicalException {

        final InstallationServiceClient isc = this.getInstallationServiceClient();

        try {
            final boolean result = isc.uninstallSharedLibrary(slName);
            LOGGER.finest("Result: " + result + " for shared library uninstallation of: " + slName);
            return result;
        } catch (final InstallationServiceErrorException e) {
            LOGGER.severe("Error occurred during " + slName + " shared library uninstallation");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    public String shutdownServiceAssembly(final String saName)
            throws PetalsServiceTechnicalException, PetalsServiceFunctionalException {

        final DeploymentServiceClient dsc = this.getDeploymentServiceClient();

        try {

            /*
             * Check if the specified service assembly is already deployed, else
             * throw exception to inform the user
             */
            if (dsc.isServiceAssemblyDeployed(saName)) {

                // check its state to know of we can shutdown it
                if (DeploymentServiceClient.State.STOPPED.equals(dsc.getState(saName))) {
                    final String result = dsc.shutdown(saName);
                    LOGGER.finest("Result: " + result + " for service assembly shutdown of: "
                            + saName);
                    return result;
                } else {
                    LOGGER.warning("Can't shutdown the service assembly: " + saName
                            + " because, is in the following state: " + dsc.getState(saName));
                    throw new PetalsServiceFunctionalException(
                            "Can't shutdown the service assembly: " + saName
                                    + " because, is in the following state: "
                                    + dsc.getState(saName));
                }
            } else {
                LOGGER.warning("Service assembly: " + saName + ", isn't deployed");
                throw new PetalsServiceFunctionalException("Service assembly: " + saName
                        + ", isn't deployed");
            }

        } catch (final DeploymentServiceErrorException e) {
            LOGGER.severe("Error occurred during service assembly shutdown of: " + saName);
            throw new PetalsServiceTechnicalException(e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.petals.tools.webconsole.services.management.ManagementService
     * #getTopology(java.lang.String, java.lang.Integer, java.lang.String,
     * java.lang.String)
     */
    public Set<Map<String, String>> getTopology(final String hostname, final Integer port,
            final String login, final String password) throws PetalsServiceTechnicalException {

        PetalsAdminServiceClient pasc = this.getPetalsAdminServiceClient();

        try {
            final Set<Map<String, String>> topology = pasc.retrieveTopology();
            LOGGER.finest("Domain topology where " + hostname + "/" + port
                    + " [hostname/port] node is hosted, correctly retrieved");
            return topology;
        } catch (final PetalsAdminServiceErrorException e) {
            LOGGER.severe("Error occurred during domain topology retrieving for " + hostname + "/"
                    + port + " [hostname/port] node");
            throw new PetalsServiceTechnicalException("Can't get petals topology for " + hostname
                    + "/" + port + " [hostname/port] node", e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.petals.tools.webconsole.services.management.ManagementService
     * #getComponentState(java.lang.String)
     */
    public String getComponentState(final String componentName)
            throws PetalsServiceTechnicalException {

        final ComponentClient cc = this.getComponentClient(componentName);

        try {

            // check component existence
            final String state = cc.getState();
            LOGGER.finest("State: " + state + " for " + componentName + " component recovered");
            return state;
        } catch (final ComponentErrorException e) {
            LOGGER.severe("Error occurred during state retrieving for: " + componentName
                    + " component");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.petals.tools.webconsole.services.management.ManagementService
     * #getAllEndpoints()
     */
    public List<Map<String, Object>> getAllEndpoints() throws PetalsServiceTechnicalException {

        final EndpointRegistryClient erc = this.getEndpointRegistryClient();

        try {
            final List<Map<String, Object>> endpoints = erc.getAllEndpoints();
            LOGGER.finest(endpoints.size() + " endpoints recovered");
            return endpoints;
        } catch (final EndpointRegistryServiceErrorException e) {
            LOGGER.severe("Error occurred during endpoints recovering");
            throw new PetalsServiceTechnicalException(e);
        }

    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.petals.tools.webconsole.services.management.ManagementService
     * #getServiceAssemblyState(java.lang.String)
     */
    public String getServiceAssemblyState(final String saName)
            throws PetalsServiceTechnicalException {

        final DeploymentServiceClient dsc = this.getDeploymentServiceClient();

        try {
            final String state = dsc.getState(saName);
            LOGGER.finest("State: " + state + " for " + saName + " service assembly recovered");
            return state;
        } catch (final DeploymentServiceErrorException e) {
            LOGGER.severe("Error occurred during state retrieving for: " + saName + " component");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.petals.tools.webconsole.services.management.ManagementService
     * #getConfigComponent(java.net.URL)
     */
    public Map<MBeanAttributeInfo, Object> getComponentConfiguration(final URL fileURL)
            throws PetalsServiceTechnicalException {

        final InstallationServiceClient isc = this.getInstallationServiceClient();

        try {
            final InstallerComponentClient icc = isc.loadNewInstaller(fileURL);

            if (icc != null) {
                LOGGER.info("Component installer for: " + fileURL + " correctly loaded");
                this.loadedInstaller = icc.getMBeanName().getKeyProperty("name");
                ConfigurationInstallerComponentClient cicc;
                Map<MBeanAttributeInfo, Object> attributes;
                try {
                    cicc = icc.getConfigurationInstallerClient();
                    attributes = cicc.getAttributes();
                } catch (final AttributeErrorException e) {
                    LOGGER.severe("Error occurred during the getting of client installer configuration for: "
                            + fileURL);
                    throw new PetalsServiceTechnicalException(e);
                } catch (final PerformActionErrorException e) {
                    LOGGER.severe("Error occurred during the getting of client installer configuration for: "
                            + fileURL);
                    throw new PetalsServiceTechnicalException(e);
                } catch (final javax.management.RuntimeOperationsException e) {
                    attributes = new HashMap<MBeanAttributeInfo, Object>();
                }
                LOGGER.finest(attributes.size() + " attributes recovered for: " + fileURL
                        + " component configuration");
                return attributes;
            } else {
                LOGGER.warning("Component installer for: " + fileURL + " not correctly loaded");
                throw new PetalsServiceTechnicalException("Component installer for: " + fileURL
                        + " not correctly loaded");
            }

        } catch (final InstallationServiceErrorException e) {
            LOGGER.severe("Error occurred during component configuration getting");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.petals.tools.webconsole.services.management.ManagementService
     * #getDocument(java.lang.String, java.lang.String)
     */
    public String getDocument(final String serviceName, final String endpointName)
            throws PetalsServiceTechnicalException {

        if (serviceName != null && !serviceName.equals("") && endpointName != null
                && !endpointName.equals("")) {
            final EndpointRegistryClient erc = this.getEndpointRegistryClient();
            try {
                return erc.getDescription(serviceName, endpointName);
            } catch (final EndpointRegistryServiceErrorException e) {
                LOGGER.severe("Error occurred during document getting of " + serviceName
                        + endpointName + " [service/endpoint]");
                throw new PetalsServiceTechnicalException(e);
            }
        }
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.petals.tools.webconsole.services.management.ManagementService
     * #getLoadedInstaller()
     */
    public String getLoadedInstaller() {
        return loadedInstaller;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.petals.tools.webconsole.services.management.ManagementService
     * #recoverRuntimeAttributes(java.lang.String)
     */
    public Map<MBeanAttributeInfo, Object> recoverRuntimeAttributes(final String componentName)
            throws PetalsServiceTechnicalException {

        Map<MBeanAttributeInfo, Object> attributes = null;
        final RuntimeConfigurationComponentClient rccc = this
                .getRuntimeConfigurationComponentClient(componentName);
        if (rccc != null) {
            try {
                attributes = rccc.getConfigurationMBeanAttributes();
            } catch (final RuntimeConfigurationErrorException e) {
                LOGGER.severe("Error occurred during runtime attributes recovering for: "
                        + componentName);
                throw new PetalsServiceTechnicalException(e);
            } catch (final PerformActionErrorException e) {
                LOGGER.severe("Error occurred during runtime attributes recovering for: "
                        + componentName);
                throw new PetalsServiceTechnicalException(e);
            }
        } else {
            LOGGER.warning("Runtime configuration component client not corretly recovered for: "
                    + componentName);
            throw new PetalsServiceTechnicalException(
                    "Runtime configuration component client not corretly recovered for: "
                            + componentName);
        }
        LOGGER.finest(attributes.size() + " runtime attributes recovered for: " + componentName);
        return attributes;
    }

    public void setRuntimeAttributes(final Map<MBeanAttributeInfo, Object> attributes,
            final String componentName) throws PetalsServiceTechnicalException {

        final RuntimeConfigurationComponentClient rccc = this
                .getRuntimeConfigurationComponentClient(componentName);
        if (rccc != null) {
            try {
                rccc.setAttributes(attributes);
                LOGGER.finest(attributes.size() + " runtime attributes set for: " + componentName
                        + " component");
            } catch (final RuntimeConfigurationErrorException e) {
                LOGGER.severe("Error occurred during runtime attributes setting for: "
                        + componentName);
                throw new PetalsServiceTechnicalException(e);
            } catch (final PerformActionErrorException e) {
                LOGGER.severe("Error occurred during runtime attributes setting for: "
                        + componentName);
                throw new PetalsServiceTechnicalException(e);
            }
        } else {
            LOGGER.warning("Runtime configuration component client not corretly recovered for: "
                    + componentName);
            throw new PetalsServiceTechnicalException(
                    "Runtime configuration component client not corretly recovered for: "
                            + componentName);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.petals.tools.webconsole.services.management.ManagementService
     * #forceUnloadInstaller(java.lang.String)
     */
    public boolean forceUnloadInstaller(final String componentName)
            throws PetalsServiceTechnicalException {

        final InstallationServiceClient isc = this.getInstallationServiceClient();
        if (isc != null) {
            try {
                final boolean result = isc.forceUnloadInstaller(componentName);
                LOGGER.finest("Result: " + result + " for the force unload installer action on "
                        + componentName + " component");
                return result;
            } catch (final InstallationServiceErrorException e) {
                LOGGER.severe("Error occurred during the force unload installer action on "
                        + componentName + " component");
                throw new PetalsServiceTechnicalException(e);
            }
        } else {
            LOGGER.warning("Installation service client not correctly recovered for " + componentName
                    + " component");
            throw new PetalsServiceTechnicalException(
                    "Installation service client not correctly recovered for " + componentName
                            + " component");
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.petals.tools.webconsole.services.management.ManagementService
     * #forceUnDeploy(java.lang.String)
     */
    public boolean forceUnDeploy(final String saName) throws PetalsServiceTechnicalException {

        final DeploymentServiceClient dsc = this.getDeploymentServiceClient();
        if (dsc != null) {
            try {
                final boolean result = dsc.forceUndeploy(saName);
                LOGGER.finest("Result: " + result + " for force undeploy action of " + saName
                        + " service assembly");
                return result;
            } catch (final DeploymentServiceErrorException e) {
                LOGGER.severe("Error occurred during the force undeploy action of " + saName
                        + " service assembly");
                throw new PetalsServiceTechnicalException(e);
            }
        } else {
            LOGGER.warning("Deployment service client not correctly recovered for " + saName
                    + " service assembly");
            throw new PetalsServiceTechnicalException(
                    "Deployment service client not correctly recovered for " + saName
                            + " service assemblyt");
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#finalize()
     */
    
    protected void finalize() throws Throwable {
        // Release jmx connection
        this.jmxClient.disconnect();
        LOGGER.finest("JMX Client correctly disconnected for " + this.hostname + "/" + this.port
                + " [host/port] server");
        super.finalize();
    }

    
    public void reInitializeConnection(final String hostname, final Integer port,
            final String login, final String password) throws PetalsServiceTechnicalException {
        this.jmxClient = ConnectionHelper.connect(hostname, port, login, password, true);
        this.hostname = hostname;
        this.port = port;
        this.login = login;
        this.password = password;
        LOGGER.finest("JMX Client correclty reinitialized for " + this.hostname + "/" + this.port
                + " [host/port] server");
    }

    /**
     * @return the hostname
     */
    public String getHostname() {
        return hostname;
    }

    /**
     * @param hostname
     *            the hostname to set
     */
    public void setHostname(String hostname) {
        this.hostname = hostname;
    }

    /**
     * @return the port
     */
    public Integer getPort() {
        return port;
    }

    /**
     * @param port
     *            the port to set
     */
    public void setPort(Integer port) {
        this.port = port;
    }

    /**
     * @return the login
     */
    public String getLogin() {
        return login;
    }

    /**
     * @param login
     *            the login to set
     */
    public void setLogin(String login) {
        this.login = login;
    }

    /**
     * @return the password
     */
    public String getPassword() {
        return password;
    }

    /**
     * @param password
     *            the password to set
     */
    public void setPassword(String password) {
        this.password = password;
    }

    /**
     * Allow to recover client API of a component associated to a JBI component.
     * 
     * @param componentName
     *            The component name
     * @return <ComponentClient>
     * @throws <PetalsServiceException>
     */
    private ComponentClient getComponentClient(final String componentName)
            throws PetalsServiceTechnicalException {
        ComponentClient cc = null;
        try {
            cc = this.jmxClient.getComponentClient(componentName);
        } catch (final ConnectionErrorException e) {
            LOGGER.finest("Error occurred during the first component client getting action on "
                    + componentName + "component");
            this.jmxClient = ConnectionHelper.connect(hostname, port, login, password, true);
            try {
                cc = this.jmxClient.getComponentClient(componentName);
            } catch (final ComponentDoesNotExistException ex) {
                LOGGER.severe("Error occurred during the second component client getting action on "
                        + componentName + "component");
                throw new PetalsServiceTechnicalException(ex);
            } catch (final ComponentErrorException ex) {
                LOGGER.severe("Error occurred during the second component client getting action on "
                        + componentName + "component");
                throw new PetalsServiceTechnicalException(ex);
            } catch (final ConnectionErrorException ex) {
                LOGGER.severe("Error occurred during the second component client getting action on "
                        + componentName + "component");
                throw new PetalsServiceTechnicalException(ex);
            }
        } catch (final ComponentDoesNotExistException e) {
            LOGGER.severe("Error occurred during the first component client getting action on "
                    + componentName + "component");
            throw new PetalsServiceTechnicalException(e);
        } catch (final ComponentErrorException e) {
            LOGGER.severe("Error occurred during the first component client getting action on "
                    + componentName + "component");
            throw new PetalsServiceTechnicalException(e);
        }

        if (cc == null) {
            LOGGER.warning("Component client not correclty recovered for " + componentName
                    + " component");
            throw new PetalsServiceTechnicalException(
                    "Component client not correclty recovered for " + componentName + " component");
        } else {
            return cc;
        }
    }

    /**
     * Allow to recover client API of the admin service.
     * 
     * @return <AdminServiceClient>
     * @throws <PetalsServiceException>
     */
    private AdminServiceClient getAdminServiceClient() throws PetalsServiceTechnicalException {

        AdminServiceClient asc = null;

        try {
            asc = this.jmxClient.getAdminServiceClient();
        } catch (final AdminDoesNotExistException e) {
            LOGGER.severe("Error occurred during the first admin service client getting action");
            throw new PetalsServiceTechnicalException(e);
        } catch (final AdminServiceErrorException e) {
            LOGGER.severe("Error occurred during the first admin service client getting action");
            throw new PetalsServiceTechnicalException(e);
        } catch (final ConnectionErrorException e) {
            LOGGER.finest("Error occurred during the first admin service client getting action");
            this.jmxClient = ConnectionHelper.connect(hostname, port, login, password, true);
            try {
                asc = this.jmxClient.getAdminServiceClient();
            } catch (final ConnectionErrorException ex) {
                LOGGER.severe("Error occurred during the second admin service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            } catch (final AdminDoesNotExistException ex) {
                LOGGER.severe("Error occurred during the second admin service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            } catch (final AdminServiceErrorException ex) {
                LOGGER.severe("Error occurred during the second admin service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            }
        }

        if (asc == null) {
            LOGGER.warning("Admin service client not correclty recovered");
            throw new PetalsServiceTechnicalException(
                    "Admin service client not correclty recovered");
        } else {
            return asc;
        }
    }

    /**
     * Allow to recover client API of the deployment service.
     * 
     * @return <DeploymentServiceClient>
     * @throws <PetalsServiceException>
     */
    private DeploymentServiceClient getDeploymentServiceClient()
            throws PetalsServiceTechnicalException {

        DeploymentServiceClient dsc = null;

        try {
            dsc = this.jmxClient.getDeploymentServiceClient();
        } catch (final DeploymentServiceDoesNotExistException e) {
            LOGGER.severe("Error occurred during the first deployment service client getting action");
            throw new PetalsServiceTechnicalException(e);
        } catch (final DeploymentServiceErrorException e) {
            LOGGER.severe("Error occurred during the first deployment service client getting action");
            throw new PetalsServiceTechnicalException(e);
        } catch (final ConnectionErrorException e) {
            LOGGER.finest("Error occurred during the first deployment service client getting action");
            this.jmxClient = ConnectionHelper.connect(hostname, port, login, password, true);
            try {
                dsc = this.jmxClient.getDeploymentServiceClient();
            } catch (final ConnectionErrorException ex) {
                LOGGER.severe("Error occurred during the second deployment service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            } catch (final DeploymentServiceErrorException ex) {
                LOGGER.severe("Error occurred during the second deployment service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            } catch (final DeploymentServiceDoesNotExistException ex) {
                LOGGER.severe("Error occurred during the second deployment service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            }
        }

        if (dsc == null) {
            LOGGER.warning("Deployment service client not correclty recovered");
            throw new PetalsServiceTechnicalException(
                    "Deployment service client not correclty recovered");
        } else {
            return dsc;
        }
    }

    /**
     * Allow to recover client API of the installation service.
     * 
     * @return <InstallationServiceClient>
     * @throws <PetalsServiceException>
     */
    private InstallationServiceClient getInstallationServiceClient()
            throws PetalsServiceTechnicalException {

        InstallationServiceClient isc = null;

        try {
            isc = this.jmxClient.getInstallationServiceClient();
        } catch (final InstallationServiceDoesNotExistException e) {
            LOGGER.severe("Error occurred during the first installation service client getting action");
            throw new PetalsServiceTechnicalException(e);
        } catch (final InstallationServiceErrorException e) {
            LOGGER.severe("Error occurred during the first installation service client getting action");
            throw new PetalsServiceTechnicalException(e);
        } catch (final ConnectionErrorException e) {
            LOGGER.finest("Error occurred during the first installation service client getting action");
            this.jmxClient = ConnectionHelper.connect(hostname, port, login, password, true);
            try {
                isc = this.jmxClient.getInstallationServiceClient();
            } catch (final ConnectionErrorException ex) {
                LOGGER.severe("Error occurred during the second installation service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            } catch (final InstallationServiceErrorException ex) {
                LOGGER.severe("Error occurred during the second installation service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            } catch (final InstallationServiceDoesNotExistException ex) {
                LOGGER.severe("Error occurred during the second installation service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            }
        }

        if (isc == null) {
            LOGGER.warning("Installation service client not correclty recovered");
            throw new PetalsServiceTechnicalException(
                    "Installation service client not correclty recovered");
        } else {
            return isc;
        }
    }

    /**
     * Allow to recover client API of the Petals admin service.
     * 
     * @return <PetalsAdminServiceClient>
     * @throws <PetalsServiceException>
     */
    private PetalsAdminServiceClient getPetalsAdminServiceClient()
            throws PetalsServiceTechnicalException {

        PetalsAdminServiceClient pasc = null;

        try {
            pasc = this.jmxClient.getPetalsAdminServiceClient();
        } catch (final PetalsAdminDoesNotExistException e) {
            LOGGER.severe("Error occurred during the first Petals admin service client getting action");
            throw new PetalsServiceTechnicalException(e);
        } catch (final PetalsAdminServiceErrorException e) {
            LOGGER.severe("Error occurred during the first Petals admin service client getting action");
            throw new PetalsServiceTechnicalException(e);
        } catch (final ConnectionErrorException e) {
            LOGGER.finest("Error occurred during the first Petals admin service client getting action");
            this.jmxClient = ConnectionHelper.connect(hostname, port, login, password, true);
            try {
                pasc = this.jmxClient.getPetalsAdminServiceClient();
            } catch (final ConnectionErrorException ex) {
                LOGGER.severe("Error occurred during the second Petals admin service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            } catch (final PetalsAdminServiceErrorException ex) {
                LOGGER.severe("Error occurred during the second Petals admin service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            } catch (final PetalsAdminDoesNotExistException ex) {
                LOGGER.severe("Error occurred during the second Petals admin service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            }
        }

        if (pasc == null) {
            LOGGER.warning("Petals admin service client not correclty recovered");
            throw new PetalsServiceTechnicalException(
                    "Petals admin service client not correclty recovered");
        } else {
            return pasc;
        }
    }

    /**
     * Allow to recover client API of the endpoint registry.
     * 
     * @return <EndpointRegistryClient>
     * @throws <PetalsServiceException>
     */
    private EndpointRegistryClient getEndpointRegistryClient()
            throws PetalsServiceTechnicalException {

        EndpointRegistryClient erc = null;

        try {
            erc = this.jmxClient.getEndpointRegistryClient();
        } catch (final EndpointRegistryServiceDoesNotExistException e) {
            LOGGER.severe("Error occurred during the first endpoint registry service client getting action");
            throw new PetalsServiceTechnicalException(e);
        } catch (final EndpointRegistryServiceErrorException e) {
            LOGGER.severe("Error occurred during the first endpoint registry service client getting action");
            throw new PetalsServiceTechnicalException(e);
        } catch (final ConnectionErrorException e) {
            LOGGER.finest("Error occurred during the first endpoint registry service client getting action");
            this.jmxClient = ConnectionHelper.connect(hostname, port, login, password, true);
            try {
                erc = this.jmxClient.getEndpointRegistryClient();
            } catch (final EndpointRegistryServiceDoesNotExistException ex) {
                LOGGER.severe("Error occurred during the second endpoint registry service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            } catch (final EndpointRegistryServiceErrorException ex) {
                LOGGER.severe("Error occurred during the second endpoint registry service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            } catch (final ConnectionErrorException ex) {
                LOGGER.severe("Error occurred during the second endpoint registry service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            }
        }

        if (erc == null) {
            LOGGER.warning("Endpoint registry service client not correclty recovered");
            throw new PetalsServiceTechnicalException(
                    "Endpoint registry service client not correclty recovered");
        } else {
            return erc;
        }
    }

    /**
     * Allow to recover client API of the runtime configuration component.
     * 
     * @return <RuntimeConfigurationComponentClient>
     * @throws <PetalsServiceException>
     */
    private RuntimeConfigurationComponentClient getRuntimeConfigurationComponentClient(
            final String componentName) throws PetalsServiceTechnicalException {

        RuntimeConfigurationComponentClient rccc = null;

        try {
            rccc = this.jmxClient.getRuntimeConfigurationClient(componentName);
        } catch (final RuntimeConfigurationDoesNotExistException e) {
            LOGGER.severe("Error occurred during the first runtime configuration component client getting action for "
                    + componentName + " component");
            throw new PetalsServiceTechnicalException(e);
        } catch (final RuntimeConfigurationErrorException e) {
            LOGGER.severe("Error occurred during the first runtime configuration component client getting action for "
                    + componentName + " component");
            throw new PetalsServiceTechnicalException(e);
        } catch (final ConnectionErrorException e) {
            LOGGER.finest("Error occurred during the first runtime configuration component client getting action for "
                    + componentName + " component");
            this.jmxClient = ConnectionHelper.connect(hostname, port, login, password, true);
            try {
                rccc = this.jmxClient.getRuntimeConfigurationClient(componentName);
            } catch (final RuntimeConfigurationDoesNotExistException ex) {
                LOGGER.severe("Error occurred during the second runtime configuration component client getting action for "
                        + componentName + " component");
                throw new PetalsServiceTechnicalException(ex);
            } catch (final RuntimeConfigurationErrorException ex) {
                LOGGER.severe("Error occurred during the second runtime configuration component client getting action for "
                        + componentName + " component");
                throw new PetalsServiceTechnicalException(ex);
            } catch (final ConnectionErrorException ex) {
                LOGGER.severe("Error occurred during the second runtime configuration component client getting action for "
                        + componentName + " component");
                throw new PetalsServiceTechnicalException(ex);
            }
        }

        if (rccc == null) {
            LOGGER.warning("Runtime configuration component client not correclty recovered for "
                    + componentName + " component");
            throw new PetalsServiceTechnicalException(
                    "Runtime configuration component client not correclty recovered for "
                            + componentName + " component");
        } else {
            return rccc;
        }
    }

    public List<String> uninstallAllComponents() throws PetalsServiceTechnicalException {

        final InstallationServiceClient isc = this.getInstallationServiceClient();

        if (isc == null) {
            LOGGER.warning("Installation service client not correctly recovered");
            throw new PetalsServiceTechnicalException(
                    "Installation service client not correctly recovered");
        }

        final DeploymentServiceClient dsc = this.getDeploymentServiceClient();

        if (dsc == null) {
            LOGGER.warning("Deployment service client not correctly recovered");
            throw new PetalsServiceTechnicalException(
                    "Deployment service client not correctly recovered");
        }

        final List<String> uninstalledComponents = new ArrayList<String>();

        try {
            final AdminServiceClient asc = this.getAdminServiceClient();

            if (asc == null) {
                LOGGER.warning("Admin service client not correctly recovered");
                throw new PetalsServiceTechnicalException(
                        "Admin service client not correctly recovered");
            }

            final ObjectName[] bcs = asc.getBindingComponents();

            final ObjectName[] ses = asc.getEngineComponents();

            final List<ObjectName> components = new ArrayList<ObjectName>();

            if (bcs != null) {
                LOGGER.finest(bcs.length + " binding components recovered");
                components.addAll(Arrays.asList(bcs));
            }
            if (ses != null) {
                LOGGER.finest(ses.length + " service engines recovered");
                components.addAll(Arrays.asList(ses));
            }

            final String[] undeployedSas = dsc.undeployAllServiceAssemblies(true);

            if (LOGGER.isLoggable(Level.FINEST)) {
                final StringBuilder builder = new StringBuilder("[");
                for (final String undeployedSa : undeployedSas) {
                    builder.append(undeployedSa).append(",");
                }
                builder.append("] service assemblies undeployed");
                LOGGER.finest(builder.toString());
            }

            if (components != null) {
                for (final ObjectName component : components) {
                    final String componentName = component.getKeyProperty("name");

                    // check component existence
                    final ComponentClient cc = this.getComponentClient(componentName);
                    LOGGER.finest("Component client correctly recovered for " + componentName
                            + " component");
                    try {
                        if (ComponentClient.State.STARTED.equals(cc.getState())) {
                            cc.stop();
                            LOGGER.finest(componentName + " component correctly stopped");
                            cc.shutdown();
                            LOGGER.finest(componentName + " component correctly shutdown");
                        } else if (ComponentClient.State.STOPPED.equals(cc.getState())) {
                            cc.shutdown();
                            LOGGER.finest(componentName + " component correctly shutdown");
                        }
                    } catch (final ComponentErrorException e) {
                        LOGGER.severe("Error occurred during uninstall all components action, unable to uninstall component: "
                                + componentName + e);
                    }
                }
            }

            final String[] ucs = isc.uninstallAllComponents();

            StringBuilder builder = null;

            if (LOGGER.isLoggable(Level.FINEST)) {
                builder = new StringBuilder("[");
            }

            for (final String uc : ucs) {
                uninstalledComponents.add(uc);
                if (LOGGER.isLoggable(Level.FINEST)) {
                    builder.append(uc).append(",");
                }
            }

            if (LOGGER.isLoggable(Level.FINEST)) {
                builder.append("] component uninstalled");
                LOGGER.finest(builder.toString());
            }

        } catch (final InstallationServiceErrorException e) {
            LOGGER.severe("Error occurred during uninstall all component action");
            throw new PetalsServiceTechnicalException(e);
        } catch (final AdminServiceErrorException e) {
            LOGGER.severe("Error occurred during uninstall all component action");
            throw new PetalsServiceTechnicalException(e);
        } catch (final DeploymentServiceErrorException e) {
            LOGGER.severe("Error occurred during uninstall all component action");
            throw new PetalsServiceTechnicalException(e);
        }

        return uninstalledComponents;
    }

    
    public List<String> unloadAllInstallers() throws PetalsServiceTechnicalException {
        final InstallationServiceClient isc = this.getInstallationServiceClient();

        if (isc == null) {
            LOGGER.warning("Installation service client not correctly recovered");
            throw new PetalsServiceTechnicalException(
                    "Installation service client not correctly recovered");
        }

        String[] results = null;
        final List<String> unloadedInstallers = new ArrayList<String>();
        try {
            results = isc.unloadAllInstallers(true);
        } catch (final InstallationServiceErrorException e) {
            LOGGER.severe("Error occurred during unload all installers");
            throw new PetalsServiceTechnicalException(e);
        }

        StringBuilder builder = null;

        if (LOGGER.isLoggable(Level.FINEST)) {
            builder = new StringBuilder("[");
        }

        if (results != null) {
            for (final String result : results) {
                unloadedInstallers.add(result);

                if (LOGGER.isLoggable(Level.FINEST)) {
                    builder.append(result).append(",");
                }
            }
        }

        if (LOGGER.isLoggable(Level.FINEST)) {
            builder.append("] installers unloaded");
            LOGGER.finest(builder.toString());
        }

        return unloadedInstallers;
    }

    
    public List<String> uninstallAllSharedLibraries() throws PetalsServiceTechnicalException {
        final InstallationServiceClient isc = this.getInstallationServiceClient();

        if (isc == null) {
            LOGGER.warning("Installation service client not correctly recovered");
            throw new PetalsServiceTechnicalException(
                    "Installation service client not correctly recovered");
        }

        String[] results = null;
        final List<String> uninstalledSharedLibraries = new ArrayList<String>();
        try {
            results = isc.uninstallAllSharedLibrary();
        } catch (final InstallationServiceErrorException e) {
            LOGGER.severe("Error occurred during uninstall all libraries");
            throw new PetalsServiceTechnicalException(e);
        }

        StringBuilder builder = null;

        if (LOGGER.isLoggable(Level.FINEST)) {
            builder = new StringBuilder("[");
        }
        if (results != null) {
            for (final String result : results) {
                uninstalledSharedLibraries.add(result);

                if (LOGGER.isLoggable(Level.FINEST)) {
                    builder.append(result).append(",");
                }
            }
        }

        if (LOGGER.isLoggable(Level.FINEST)) {
            builder.append("] shared libraries uninstalled");
            LOGGER.finest(builder.toString());
        }
        return uninstalledSharedLibraries;
    }

    
    public List<String> uninstallAllServiceAssemblies() throws PetalsServiceTechnicalException {

        final DeploymentServiceClient dsc = this.getDeploymentServiceClient();

        if (dsc == null) {
            LOGGER.warning("Deployment service client not correctly recovered");
            throw new PetalsServiceTechnicalException(
                    "Deployment service client not correctly recovered");
        }

        final List<String> uninstalledServiceAssemblies = new ArrayList<String>();
        String[] sas = null;
        try {
            sas = dsc.getDeployedServiceAssemblies();

            StringBuilder builder = null;

            if (LOGGER.isLoggable(Level.FINEST)) {
                builder = new StringBuilder("[");
            }

            if (sas != null) {
                for (final String sa : sas) {
                    if (dsc.forceUndeploy(sa)) {
                        uninstalledServiceAssemblies.add(sa);
                        if (LOGGER.isLoggable(Level.FINEST)) {
                            builder.append(sa).append(",");
                        }
                    }
                }
            }

            if (LOGGER.isLoggable(Level.FINEST)) {
                builder.append("] service assemblies uninstalled");
                LOGGER.finest(builder.toString());
            }

        } catch (final DeploymentServiceErrorException e) {
            LOGGER.severe("Error occurred during uninstall all service assemblies");
            throw new PetalsServiceTechnicalException(e);
        }

        return uninstalledServiceAssemblies;
    }

    
    public String[] getAllLevels() throws PetalsServiceTechnicalException {
        final LoggerServiceClient lsc = this.getLoggerServiceClient();

        if (lsc == null) {
            LOGGER.warning("Logger service client not correctly recovered");
            throw new PetalsServiceTechnicalException(
                    "Logger service client not correctly recovered");
        }

        try {
            final String[] loggerLevels = lsc.getAllLevels();

            if (LOGGER.isLoggable(Level.FINEST)) {
                final StringBuilder builder = new StringBuilder();
                builder.append("[");
                for (final String loggerLevel : loggerLevels) {
                    builder.append(loggerLevel).append(",");
                }
                builder.append("] logger levels retrieved");
            }

            return loggerLevels;
        } catch (final LoggerServiceErrorException e) {
            LOGGER.severe("Error occurred during logger levels retrieving");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    
    public String getLevelForLogger(final String logger) throws PetalsServiceTechnicalException {

        final LoggerServiceClient lsc = this.getLoggerServiceClient();

        if (lsc == null) {
            LOGGER.warning("Logger service client not correctly recovered");
            throw new PetalsServiceTechnicalException(
                    "Logger service client not correctly recovered");
        }

        try {
            final String level = lsc.getLevelForLogger(logger);
            LOGGER.finest(logger + " logger have " + level + " for logging level");
            return level;
        } catch (final LoggerServiceErrorException e) {
            LOGGER.severe("Error occurred during logging level retrieving for " + logger + " logger");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    
    public void setLoggerLevel(final String logger, final String level)
            throws PetalsServiceTechnicalException {
        final LoggerServiceClient lsc = this.getLoggerServiceClient();

        if (lsc == null) {
            LOGGER.warning("Logger service client not correctly recovered");
            throw new PetalsServiceTechnicalException(
                    "Logger service client not correctly recovered");
        }

        try {
            lsc.setLevelForLogger(logger, level);
            LOGGER.finest("Set logger level action correctly performed for: " + logger + "/" + level
                    + " [logger/level]");
        } catch (final LoggerServiceErrorException e) {
            LOGGER.severe("Error occurred during the set of logger level: " + logger + "/" + level
                    + " [logger/level]");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    /**
     * Allow to recover client API of the logger service.
     * 
     * @return <LoggerServiceClient>
     * @throws <PetalsServiceException>
     */
    private LoggerServiceClient getLoggerServiceClient() throws PetalsServiceTechnicalException {

        LoggerServiceClient lsc = null;

        try {
            lsc = this.jmxClient.getLoggerServiceClient();
        } catch (final LoggerDoesNotExistException e) {
            LOGGER.severe("Error occurred during the first logger service client getting action");
            throw new PetalsServiceTechnicalException(e);
        } catch (final LoggerServiceErrorException e) {
            LOGGER.severe("Error occurred during the first logger service client getting action");
            throw new PetalsServiceTechnicalException(e);
        } catch (final ConnectionErrorException e) {

            LOGGER.finest("Error occurred during the first logger service client getting action");
            this.jmxClient = ConnectionHelper.connect(hostname, port, login, password, true);
            try {
                lsc = this.jmxClient.getLoggerServiceClient();
            } catch (final ConnectionErrorException ex) {
                LOGGER.severe("Error occurred during the second logger service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            } catch (final LoggerServiceErrorException ex) {
                LOGGER.severe("Error occurred during the second logger service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            } catch (final LoggerDoesNotExistException ex) {
                LOGGER.severe("Error occurred during the second logger service client getting action");
                throw new PetalsServiceTechnicalException(ex);
            }
        }

        if (lsc == null) {
            LOGGER.warning("Logger service client not correclty recovered");
            throw new PetalsServiceTechnicalException(
                    "Logger service client not correclty recovered");
        } else {
            return lsc;
        }
    }

    
    public String[] getComponentServiceAssembly(final String name)
            throws PetalsServiceTechnicalException {
        final DeploymentServiceClient dsc = this.getDeploymentServiceClient();

        try {
            final String[] sas = dsc.getDeployedServiceAssembliesForComponent(name);
            LOGGER.finest(sas.length + " service assemblies recovered for [" + name + "] component");
            return sas;
        } catch (final DeploymentServiceErrorException e) {
            LOGGER.severe("Error occurred during service assemblies recovery for [" + name
                    + "] component");
            throw new PetalsServiceTechnicalException(e);
        }
    }

    
    public List<String> getLoadedComponentInstallers() throws PetalsServiceTechnicalException {

        final String[] lcis;
        try {
            lcis = this.jmxClient.getInstallationServiceClient().getInstallers();
        } catch (final InstallationServiceErrorException e) {
            LOGGER.severe("Error occurred during the loaded component installers recovering");
            throw new PetalsServiceTechnicalException(e);
        } catch (final InstallationServiceDoesNotExistException e) {
            LOGGER.severe("Error occurred during the loaded component installers recovering");
            throw new PetalsServiceTechnicalException(e);
        } catch (final ConnectionErrorException e) {
            LOGGER.severe("Error occurred during the loaded component installers recovering");
            throw new PetalsServiceTechnicalException(e);
        }

        if (lcis != null && lcis.length > 0) {
            LOGGER.finest(lcis.length + " loaded component installers recovered");
        } else {
            LOGGER.finest("No loaded component installers recovered");
        }

        return Arrays.asList(lcis);
    }

}
