/**
 * petalsbpm-domain - Domain Objects for BPMN standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard;

import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.WithFlowElementsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.choreography.IChoreographyBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.choreography.IChoreographyTaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.choreography.ISubChoreographyBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.choreography.WithChoreographyFlowElementsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ICollaborationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.IInteractionNodeBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ILaneBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ILaneSetBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IErrorBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IEscalationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IItemDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IMessageBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IMessageFlowBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IParticipantBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IPartnerRoleBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.artifact.IAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.artifact.ITextAnnotationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.artifact.WithArtifactsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.infrastructure.IDefinitionsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.infrastructure.IImportBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.IProcessBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.ICallActivityBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.ISubProcessBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.ITaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IAssignmentBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataInputAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataInputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataObjectBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataOutputAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataOutputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IExpressionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IIOSpecificationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IInputOutputBinding;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.WithDataInputAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.WithDataInputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.WithDataOutputAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.WithDataOutputBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.ICatchEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IEndEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IIntermediateCatchEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IIntermediateThrowEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.ISignalBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IStartEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IThrowEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.definition.IEventDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.gateway.IGatewayBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.gateway.ISequenceFlowBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.service.IEndPointBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.service.IInterfaceBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.service.IOperationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.data.DataAssociationBean;

public abstract class DefinitionsBeanVisitor {

	private IDefinitionsBean defs; 
	
	public DefinitionsBeanVisitor(IDefinitionsBean defs) {
		this.defs = defs;
	}
	
	
	public void visitDefinitionsByPools() {
		this.visitDefinitions(defs);
		//IMPORTS
		for(IImportBean ib : defs.getImports()){
			this.visitImport(ib);
		}
		
		//SIGNALS
		for(ISignalBean sb : defs.getSignals()){
		    this.visitSignal(sb);
		}
		
		for(IErrorBean eb : defs.getErrors()){
		    this.visitError(eb);
		}
		
		for(IEscalationBean eb : defs.getEscalations()) {
		    this.visitEscalation(eb);
		}
		
		for(IEndPointBean ep : defs.getEndPoints()){
		    this.visitEndPoint(ep);
		}
		
		//PARTNER ROLES
		for(IPartnerRoleBean pr : defs.getPartnerRoles()){
			this.visitPartnerRole(pr);
			
			if(pr.getParticipants()!=null){
				for(IParticipantBean pb : pr.getParticipants()){
					this.visitPartnerRoleParticipant(pb, pr);
				}
			}
		}
		
		//ITEM DEFINITIONS
		for(IItemDefinitionBean idb : defs.getItemDefinitions()){
			this.visitItemDefinition(idb);
		}
		
		//MESSAGES
		for(IMessageBean msg : defs.getMessages()){
			this.visitMessage(msg);
			
			IItemDefinitionBean idb = msg.getItemDefinition();
			if(idb!=null){
				this.visitMessageItemDefinition(idb);
			}
		}
		
		//INTERFACES
        for(IInterfaceBean itf : defs.getInterfaces()){
            this.visitInterface(itf);
            
            //OPERATIONS
            if(itf.getOperations()!=null){
                for(IOperationBean op : itf.getOperations()){
                    this.visitOperation(op);
                    
                    IMessageBean msg = op.getMessageIn();
                    this.visitOperationMessageIn(msg,op);
                    
                    msg = op.getMessageOut();
                    if(msg!=null){
                        this.visitOperationMessageOut(msg,op);
                    }
                    
                    if(op.getErrors()!=null){
                        for(IErrorBean mb : op.getErrors()){
                            this.visitOperationError(mb,op);
                        }
                    }
                }
            }
        }
		
		//COLLABORATIONS
		for(ICollaborationBean collab : defs.getCollaborations()){
			this.visitCollaboration(collab);
			
			parseCollaboration(collab);
			
			parseWithArtifacts(collab);
		}
		
		
		for(IChoreographyBean chor : defs.getChoreographies()){
		    this.visitChoreography(chor);
		    
		    parseCollaboration(chor);
		    
		    parseChoreographyElements(chor);
		}
		
		
		//PROCESSES
		for(IProcessBean proc : defs.getProcesses()){
			this.visitProcess(proc);
			
			parseFlowNodes(proc);
			
			if(proc.getLaneSets()!=null){
				for(ILaneSetBean laneSet : proc.getLaneSets()){
					this.visitLaneSet(laneSet);
					
					parseLaneSet(laneSet);
				}
			}
			
			if(proc.getSequenceFlows()!=null){
				for(ISequenceFlowBean sfb : proc.getSequenceFlows()){
					this.visitSequenceFlow(sfb);
					
					if(sfb.getExpression()!=null){
						this.visitSFExpression(sfb.getExpression());
					}
				}
			}
			
			parseWithArtifacts(proc);
			
			parseIOSpec(proc.getIoSpecification());
			
			if(proc.getIOBindings()!=null) {
			    for(IInputOutputBinding iobinding : proc.getIOBindings()) {
			        this.visitIOBinding(iobinding);
			    }
			}
			
			this.endProcessVisit(proc);
		}
		
	}
	
	



    private void parseCollaboration(ICollaborationBean collab){
	    //POOLS
	    if(collab.getParticipants()!=null){
	        for(IParticipantBean p : collab.getParticipants()){
	            this.visitParticipant(p);

	            if(p.getProcess()!=null){
	                IProcessBean proc = p.getProcess();
	                this.visitParticipantProcess(proc,p);
	            }

	            if(p.getInterfaces()!=null){
	                for(IInterfaceBean itf : p.getInterfaces()){
	                    this.visitParticipantInterface(itf);
	                }
	            }
	            
	            if(p.getEndPoints()!=null){
	                for(IEndPointBean ep : p.getEndPoints()){
	                    this.visitParticipantEndPoint(ep,p);
	                }
	            }
	        }
	    }

	    //MESSAGE FLOWS
	    if(collab.getMessageFlows()!=null){
	        for(IMessageFlowBean mfb : collab.getMessageFlows()){
	            this.visitMessageFlow(mfb);

	            IMessageBean mb = mfb.getMessage();
	            if(mb!=null){
	                this.visitMessageFlowMessage(mb);
	            }

	            IInteractionNodeBean node = mfb.getSource();
	            if(node!=null){
	                this.visitMessageFlowSource(node);
	            }

	            node = mfb.getTarget(); 
	            if(node!=null){
	                this.visitMessageFlowTarget(node);
	            }
	        }
	    }
	}



    private void parseChoreographyElements(WithChoreographyFlowElementsBean chor) {
        if(chor.getStartEvents()!=null){
            for(IStartEventBean seb : chor.getStartEvents()){
                this.visitStartEvent(seb);
                parseCatchEventData(seb);
                if(seb.getTriggers()!=null){
                    for(IEventDefinitionBean ed : seb.getTriggers()){
                        visitEventDefinition(ed);
                    }
                }
            }
        }
        if(chor.getIntermediateCatchEvents()!=null){
            for(IIntermediateCatchEventBean e : chor.getIntermediateCatchEvents()){
                this.visitIntermediateCatchEvent(e);
                parseCatchEventData(e);
                if(e.getTriggers()!=null){
                    for(IEventDefinitionBean ed : e.getTriggers()){
                        visitEventDefinition(ed);
                    }
                }
            }
        }
        if(chor.getIntermediateThrowEvents()!=null){
            for(IIntermediateThrowEventBean e : chor.getIntermediateThrowEvents()){
                this.visitIntermediateThrowEvent(e);
                parseThrowEventData(e);
                if(e.getResults()!=null){
                    for(IEventDefinitionBean ed : e.getResults()){
                        visitEventDefinition(ed);
                    }
                }
            }
        }
        if(chor.getEndEvents()!=null){
            for(IEndEventBean eeb : chor.getEndEvents()){
                this.visitEndEvent(eeb);
                parseThrowEventData(eeb);
                if(eeb.getResults()!=null){
                    for(IEventDefinitionBean ed : eeb.getResults()){
                        visitEventDefinition(ed);
                    }
                }
            }
        }
        if(chor.getGateways()!=null){
            for(IGatewayBean gb : chor.getGateways()){
                this.visitGateway(gb);
            }
        }
        if(chor.getChoreographyTasks()!=null){
            for(IChoreographyTaskBean ct : chor.getChoreographyTasks()){
                this.visitChoreographyTask(ct);
            }
        }
        if(chor.getSubChoreographies()!=null){
            for(ISubChoreographyBean subChor : chor.getSubChoreographies()){
                this.visitSubChoreography(subChor);
                
                parseChoreographyElements(subChor);
                
                parseWithArtifacts(subChor);
            }
        }
        if(chor.getSequenceFlows()!=null){
            for(ISequenceFlowBean sfb : chor.getSequenceFlows()){
                this.visitSequenceFlow(sfb);
                
                if(sfb.getExpression()!=null){
                    this.visitSFExpression(sfb.getExpression());
                }
            }
        }
    }


    private void parseFlowNodes(WithFlowElementsBean proc) {
		if(proc.getStartEvents()!=null){
			for(IStartEventBean seb : proc.getStartEvents()){
				this.visitStartEvent(seb);
				parseCatchEventData(seb);
				if(seb.getTriggers()!=null){
				    for(IEventDefinitionBean ed : seb.getTriggers()){
				        visitEventDefinition(ed);
				    }
				}
			}
		}
		
		if(proc.getTasks()!=null){
			for(ITaskBean tb : proc.getTasks()){
				this.visitTask(tb);
				
				if(tb.getDataInputAssociations()!=null){
					for(IDataInputAssociationBean dia : tb.getDataInputAssociations()){
						this.visitDataInputAssociation(dia, tb);
						
						if(dia.getAssignements()!=null){
							for(IAssignmentBean a : dia.getAssignements()){
								visitAssignement(a,(DataAssociationBean)dia);
							}
						}
					}
				}
				if(tb.getDataOutputAssociations()!=null){
					for(IDataOutputAssociationBean doa : tb.getDataOutputAssociations()){
						this.visitDataOutputAssociation(doa, tb);
						
						if(doa.getAssignements()!=null){
							for(IAssignmentBean a : doa.getAssignements()){
								visitAssignement(a,(DataAssociationBean)doa);
							}
						}
					}
				}
				
				parseIOSpec(tb.getIoSpecification());
			}
		}
		if(proc.getCallActivities()!=null) {
		    for(ICallActivityBean ca : proc.getCallActivities()){
		        this.visitCallActivity(ca);
		        
		        if(ca.getDataInputAssociations()!=null){
                    for(IDataInputAssociationBean dia : ca.getDataInputAssociations()){
                        this.visitDataInputAssociation(dia, ca);
                        
                        if(dia.getAssignements()!=null){
                            for(IAssignmentBean a : dia.getAssignements()){
                                visitAssignement(a,(DataAssociationBean)dia);
                            }
                        }
                    }
                }
                if(ca.getDataOutputAssociations()!=null){
                    for(IDataOutputAssociationBean doa : ca.getDataOutputAssociations()){
                        this.visitDataOutputAssociation(doa, ca);
                        
                        if(doa.getAssignements()!=null){
                            for(IAssignmentBean a : doa.getAssignements()){
                                visitAssignement(a,(DataAssociationBean)doa);
                            }
                        }
                    }
                }
                
                parseIOSpec(ca.getIoSpecification());
		    }
		}
		if(proc.getDataObjects()!=null){
			for(IDataObjectBean dobj : proc.getDataObjects()){
				this.visitDataObject(dobj);
			}
		}
		if(proc.getIntermediateCatchEvents()!=null){
			for(IIntermediateCatchEventBean e : proc.getIntermediateCatchEvents()){
				this.visitIntermediateCatchEvent(e);
				parseCatchEventData(e);
				if(e.getTriggers()!=null){
                    for(IEventDefinitionBean ed : e.getTriggers()){
                        visitEventDefinition(ed);
                    }
                }
			}
		}
		if(proc.getIntermediateThrowEvents()!=null){
			for(IIntermediateThrowEventBean e : proc.getIntermediateThrowEvents()){
				this.visitIntermediateThrowEvent(e);
				parseThrowEventData(e);
				if(e.getResults()!=null){
                    for(IEventDefinitionBean ed : e.getResults()){
                        visitEventDefinition(ed);
                    }
                }
			}
		}
		if(proc.getEndEvents()!=null){
			for(IEndEventBean eeb : proc.getEndEvents()){
				this.visitEndEvent(eeb);
				parseThrowEventData(eeb);
				if(eeb.getResults()!=null){
                    for(IEventDefinitionBean ed : eeb.getResults()){
                        visitEventDefinition(ed);
                    }
                }
			}
		}
		if(proc.getGateways()!=null){
			for(IGatewayBean gb : proc.getGateways()){
				this.visitGateway(gb);
			}
		}
		if(proc.getSubprocesses()!=null){
            for(ISubProcessBean sp : proc.getSubprocesses()){
                this.visitSubProcess(sp);
                
                parseFlowNodes(sp);
                
                if(sp.getLaneSets()!=null){
                    for(ILaneSetBean laneSet : sp.getLaneSets()){
                        this.visitLaneSet(laneSet);
                        
                        parseLaneSet(laneSet);
                        
                        this.endLaneSetVisit(laneSet);
                    }
                }
                
                if(sp.getSequenceFlows()!=null){
                    for(ISequenceFlowBean sfb : sp.getSequenceFlows()){
                        this.visitSequenceFlow(sfb);
                        
                        if(sfb.getExpression()!=null){
                            this.visitSFExpression(sfb.getExpression());
                        }
                    }
                }
                
                parseWithArtifacts(sp);
                
                parseIOSpec(sp.getIoSpecification());
                
                this.endSubProcessVisit(sp);
            }
        }
	}
	
	



    private void parseLaneSet(ILaneSetBean laneSet) {
		//LANES
		if(laneSet.getLanes()!=null){
			for(ILaneBean lane: laneSet.getLanes()){
				this.visitLane(lane);
				
				ILaneSetBean lsb = lane.getChildLaneSet();
				if(lsb!=null){
					this.visitChildLaneSet(lsb);
					
					parseLaneSet(lsb);
					
					endLaneSetVisit(lsb);
				}
			}
		}
	}
	
	
	private void parseWithArtifacts(WithArtifactsBean wa){
        if(wa.getAssociations()!=null){
            for(IAssociationBean a : wa.getAssociations()){
                this.visitAssociation(a);
            }
        }
        if(wa.getTextAnnotations()!=null){
            for(ITextAnnotationBean ta : wa.getTextAnnotations()){
                this.visitTextAnnotation(ta);
            }
        }
	}
	
	private void parseIOSpec(IIOSpecificationBean ios){
	    if(ios!=null){
            this.visitIOSpecification(ios);
            
            if(ios.getDataInputs()!=null){
                for(IDataInputBean dib : ios.getDataInputs()){
                    this.visitDataInput(dib,ios);
                }
            }
            if(ios.getDataOutputs()!=null){
                for(IDataOutputBean dib : ios.getDataOutputs()){
                    this.visitDataOutput(dib,ios);
                }
            }
        }
	}
	
	
    private void parseCatchEventData(ICatchEventBean ceb){
		if(ceb.getDataOutputs()!=null){
			for(IDataOutputBean dob : ceb.getDataOutputs()){
				this.visitDataOutput(dob,ceb);
			}
		}
		if(ceb.getDataOutputAssociations()!=null){
			for(IDataOutputAssociationBean a : ceb.getDataOutputAssociations()){
				this.visitDataOutputAssociation(a,ceb);
				
				if(a.getAssignements()!=null){
					for(IAssignmentBean ab : a.getAssignements()){
						visitAssignement(ab,a);
					}
				}
			}
		}
	}
	
	private void parseThrowEventData(IThrowEventBean teb){
		if(teb.getDataInputs()!=null){
			for(IDataInputBean dob : teb.getDataInputs()){
				this.visitDataInput(dob,teb);
			}
		}
		if(teb.getDataInputAssociations()!=null){
			for(IDataInputAssociationBean a : teb.getDataInputAssociations()){
				this.visitDataInputAssociation(a,teb);
				
				if(a.getAssignements()!=null){
					for(IAssignmentBean ab : a.getAssignements()){
						visitAssignement(ab,a);
					}
				}
			}
		}
	}

	
	public void visitIOBinding(IInputOutputBinding iobinding) {}
	public void visitEscalation(IEscalationBean eb) {}
	public void visitError(IErrorBean eb) {}
	public void visitEndPoint(IEndPointBean ep) {}
	public void visitCallActivity(ICallActivityBean ca) {}
	public void visitTextAnnotation(ITextAnnotationBean ta) {}
    public void visitAssociation(IAssociationBean a) {}
	public void visitChoreographyTask(IChoreographyTaskBean chor){}
	public void visitSubChoreography(ISubChoreographyBean chor){}
	public void endSubChoreographyVisit(ISubChoreographyBean chor){}
	public void visitChoreography(IChoreographyBean chor){}
	public void endChoreographyVisit(IChoreographyBean chor){}
	public void visitSignal(ISignalBean sb) {}
	public void visitSubProcess(ISubProcessBean sp){}
	public void endSubProcessVisit(ISubProcessBean proc) {}
	public void visitEventDefinition(IEventDefinitionBean ed){}
	public void visitDataObject(IDataObjectBean dobj) {}
	public void visitIOSpecification(IIOSpecificationBean ios) {}
	public void visitDataInput(IDataInputBean dob, WithDataInputBean owner) {}
	public void visitDataOutput(IDataOutputBean dob, WithDataOutputBean owner) {}
	public void visitIntermediateThrowEvent(IIntermediateThrowEventBean e) {}
	public void visitIntermediateCatchEvent(IIntermediateCatchEventBean e) {}
	public void visitMessageItemDefinition(IItemDefinitionBean idb) {}
	public void endProcessVisit(IProcessBean proc) {}
	public void visitMessage(IMessageBean msg) {}
	public void visitPartnerRole(IPartnerRoleBean pr) {}
	public void visitPartnerRoleParticipant(IParticipantBean pb, IPartnerRoleBean pr) {}
	public void visitProcess(IProcessBean proc) {}
	public void visitInterface(IInterfaceBean proc) {}
	public void visitAssignement(IAssignmentBean ab, IDataAssociationBean a) {}
	public void visitDataInputAssociation(IDataInputAssociationBean dia, WithDataInputAssociationBean owner){}
	public void visitDataOutputAssociation(IDataOutputAssociationBean doa, WithDataOutputAssociationBean owner){}
	public void visitSFExpression(IExpressionBean expression) {}
	public void visitChildLaneSet(ILaneSetBean lsb) {}
	public void visitDefinitions(IDefinitionsBean defs) {}
	public void visitParticipantInterface(IInterfaceBean interface1) {}
	public void visitParticipantProcess(IProcessBean process, IParticipantBean participant) {}
	public void visitParticipantEndPoint(IEndPointBean ep, IParticipantBean participant) {}
	public void visitParticipant(IParticipantBean participant) {}
	public void visitMessageFlowSource(IInteractionNodeBean node) {}
	public void visitMessageFlowTarget(IInteractionNodeBean node) {}
	public void visitMessageFlowMessage(IMessageBean mb) {}
	public void visitMessageFlow(IMessageFlowBean mfb) {}
	public void visitCollaboration(ICollaborationBean collab) {}
	public void visitSequenceFlow(ISequenceFlowBean sfb) {}
	public void visitGateway(IGatewayBean gb) {}
	public void visitEndEvent(IEndEventBean eeb) {}
	public void visitTask(ITaskBean tb) {}
	public void visitStartEvent(IStartEventBean seb) {}
	public void visitLane(ILaneBean lane) {}
	public void visitLaneSet(ILaneSetBean laneSet) {}
	public void endLaneSetVisit(ILaneSetBean laneSet) {}
	public void visitOperationMessageIn(IMessageBean msg,IOperationBean opBean) {}
	public void visitOperationMessageOut(IMessageBean msg,IOperationBean opBean) {}
	public void visitOperationError(IErrorBean msg,IOperationBean opBean) {}
	public void visitOperation(IOperationBean op) {}
	public void visitItemDefinition(IItemDefinitionBean itemDefinition) {}
	public void visitImport(IImportBean ib) {}
}
