/**
 * petalsbpm-domain - Domain Objects for BPMN standard - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard;

import java.util.ArrayList;
import java.util.List;

import com.ebmwebsourcing.geasyschema.domain.api.IElement;
import com.ebmwebsourcing.geasywsdl.domain.api.IOperation;
import com.ebmwebsourcing.geasywsdl.domain.api.IPortType;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ICollaborationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ILaneBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ILaneSetBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.ICallableElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IErrorBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IFlowElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IFlowNodeBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IItemDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IMessageBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IMessageFlowBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IParticipantBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IPartnerRoleBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.foundation.IBaseElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.infrastructure.IDefinitionsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.infrastructure.IImportBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.infrastructure.xsd.IBPMN20ImportBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.IProcessBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.ICallActivityBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.activity.ITaskBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IAssignmentBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataInputAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IDataOutputAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.IExpressionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.WithDataInputAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.data.WithDataOutputAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IEndEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IIntermediateCatchEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IIntermediateThrowEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.IStartEventBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.event.definition.IEventDefinitionBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.gateway.IGatewayBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.gateway.ISequenceFlowBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.service.IEndPointBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.service.IInterfaceBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.service.IOperationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.foundation.BaseElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure.DefinitionsBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure.xsd.BPMN20ImportBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure.xsd.WSDLImportBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.infrastructure.xsd.XSDImportBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.process.ProcessBean;

public class DefinitionsHelper {

	private static DefinitionsHelper instance;
	
	public static DefinitionsHelper getInstance(){
		if(instance==null){
			instance = new DefinitionsHelper();
		}
		return instance;
	}
	
	
	public List<String> getImportedStructureRefQnames(IDefinitionsBean defs){

		List<String> result = new ArrayList<String>();
		
		for(IImportBean ib:defs.getImports()){
			
			if (ib instanceof XSDImportBean){
				
				XSDImportBean xsdIb = (XSDImportBean) ib;
				
				for(IElement element:xsdIb.getSchema().getElements()){
					
					result.add(ib.getNSDeclaration().getPrefix()+":"+element.getName());
				
				}
			
			}
			
		}
		
		return result;
	}
	
	
	public List<String> getImportedWSDLImplementationRefQnames(IDefinitionsBean defs){
        
        List<String> result = new ArrayList<String>();
        
        for(IImportBean ib:defs.getImports()){
            
            if (ib instanceof WSDLImportBean){
                
                WSDLImportBean wsdlIb = (WSDLImportBean) ib;
                
                for(IPortType pt:wsdlIb.getWSDLDefinitions().getPortTypes()){
                    
                    result.add(ib.getNSDeclaration().getPrefix()+":"+pt.getName());
                    
                }
                
            }
            
        }
        
        return result;
    }
    
    public List<String> getImportedWSDLOperationsRefQnames(IDefinitionsBean defs){
        
        List<String> result = new ArrayList<String>();
        
        for(IImportBean ib:defs.getImports()){
            
            if (ib instanceof WSDLImportBean){
                
                WSDLImportBean wsdlIb = (WSDLImportBean) ib;
                
                for(IPortType pt:wsdlIb.getWSDLDefinitions().getPortTypes()){
                    
                    for(IOperation o:pt.getOperations()){
                        
                        result.add(ib.getNSDeclaration().getPrefix()+":"+o.getName());  
                        
                    }
                    
                    
                }
                
            }
            
        }
        
        return result;
    }
	
	
	public IProcessBean getParentProcess(IFlowElementBean fe, IDefinitionsBean defs){
        for(IProcessBean p : defs.getProcesses()){
            for(IFlowElementBean elt : p.getFlowNodes()){
                if(fe.getId().equals(elt.getId())){
                    return p;
                }
            }
            for(ISequenceFlowBean sf : p.getSequenceFlows()) {
                if(fe.getId().equals(sf.getId())){
                    return p;
                }
            }
        }
        return null;
    }
	
	
	public List<ISequenceFlowBean> getOutgoingSequenceFlows(IFlowNodeBean fn, IDefinitionsBean defs) {
	    IProcessBean proc = getParentProcess(fn, defs);
	    if(proc==null) {
	        return null;
	    }
	    List<ISequenceFlowBean> result = new ArrayList<ISequenceFlowBean>();
	    for(ISequenceFlowBean sf : proc.getSequenceFlows()) {
	    	
	    	if (sf.getSourceNode()==null) continue;
	        if(sf.getSourceNode().getId().equals(fn.getId())) {
	            result.add(sf);
	        }
	    
	    }
	    return result;
	}
	
	public List<ICallableElementBean> getCallableElements(IDefinitionsBean defs){
		
		List<ICallableElementBean> result = new ArrayList<ICallableElementBean>();
		
		//add processes that are embedded in current definitions
		result.addAll(defs.getProcesses());
		
		//add process that have been imported
		result.addAll(getImportedCallableElements(defs));
		
		return result;
	}
	
	
	public List<ICallableElementBean> getImportedCallableElements(IDefinitionsBean defs) {
	    List<ICallableElementBean> result = new ArrayList<ICallableElementBean>();
	    for(IImportBean i:defs.getImports()){
	        if (i instanceof IBPMN20ImportBean){
	            IBPMN20ImportBean bpmnImport = (IBPMN20ImportBean) i;
	            result.addAll(bpmnImport.getBPMN20Definitions().getProcesses());
	        }
	    }
	    return result;
	}
	
	
	public BaseElementBean getElementById(DefinitionsBean defs, String id){
		try{
			return (BaseElementBean) new SearchingVisitor(defs).getElementById(id);
		}
		catch(Exception e){
			e.printStackTrace();
			return null;
		}
	}
	
	public BaseElementBean getElementById(IDefinitionsBean defs, String id, String namespace) {
	    if(namespace==null || namespace.isEmpty()) {
	        return getElementById((DefinitionsBean) defs, id);
	    }
	    for(IImportBean impt : defs.getImports()) {
	        if(impt instanceof BPMN20ImportBean && impt.getNamespace().equals(namespace)) {
	           return getElementById((DefinitionsBean) ((BPMN20ImportBean)impt).getBPMN20Definitions(), id); 
	        }
	    }
	    return null;
    }
	
	
	public IFlowElementBean getFlowNodeById(String id, ProcessBean proc){
		if(proc.getLaneSets()!=null){
			for(ILaneSetBean lsb : proc.getLaneSets()){
				for(ILaneBean l : lsb.getLanes()){
					for(IFlowElementBean fe : l.getFlowNodes()){
						if(fe.getId().equals(id)){
							return fe;
						}
					}
				}
			}
		}
		return null;
	}
	
	public List<ILaneBean> findAllChildLanes(IProcessBean process) {
        List<ILaneBean> result = new ArrayList<ILaneBean>();
        if(process.getLaneSets()!=null) {
            for(ILaneSetBean laneSet : process.getLaneSets()) {
                result.addAll(laneSet.getLanes());
            }
        }
        return result;
    }
	
	
	
	private class SearchingVisitor extends DefinitionsBeanVisitor{
		private IBaseElementBean bean;
		private String id;
		
		public SearchingVisitor(DefinitionsBean defs){
			super(defs);
		}
		
		public IBaseElementBean getElementById(String id) throws Exception{
			this.id = id;
			this.bean = null;
			visitDefinitionsByPools();
			return bean;
		}
		
		@Override
		public void visitEndPoint(IEndPointBean epb) {
		    if(epb.getId().equals(this.id) && bean==null){bean = epb;}
		}
		@Override
		public void visitInterface(IInterfaceBean itf) {
		    if(itf.getId().equals(this.id) && bean==null){bean = itf;}
		}
		@Override
		public void visitProcess(IProcessBean pb) {
		    if(pb.getId().equals(this.id) && bean==null){bean = pb;}
		}
		@Override
		public void visitParticipant(IParticipantBean participant) {
			if(participant.getId().equals(this.id) && bean==null){bean = participant;}
		}
//		@Override
//		public void visitParticipantInterface(IInterfaceBean itf){
//			if(itf.getId().equals(this.id) && bean==null){bean = itf;}
//		}
		@Override
		public void visitParticipantProcess(IProcessBean proc, IParticipantBean participant){
			if(proc.getId().equals(this.id) && bean==null){bean = proc;}
		}
		@Override
		public void visitMessage(IMessageBean msg) {
		    if(msg.getId().equals(this.id) && bean==null){bean = msg;}
		}
		@Override
		public void visitMessageFlow(IMessageFlowBean mfb) {
			if(mfb.getId().equals(this.id) && bean==null){bean = mfb;}
		}
		@Override
		public void visitCollaboration(ICollaborationBean collab) {
			if(collab.getId().equals(this.id) && bean==null){bean = collab;}
		}
		@Override
		public void visitSequenceFlow(ISequenceFlowBean sfb) {
			if(sfb.getId().equals(this.id) && bean==null){bean = sfb;}
		}
		@Override
		public void visitGateway(IGatewayBean gb) {
			if(gb.getId().equals(this.id) && bean==null){bean = gb;}
		}
		@Override
		public void visitEndEvent(IEndEventBean eeb) {
			if(eeb.getId().equals(this.id) && bean==null){bean = eeb;}
		}
		@Override
		public void visitTask(ITaskBean tb) {
			if(tb.getId().equals(this.id) && bean==null){bean = tb;}
		}
		@Override
		public void visitCallActivity(ICallActivityBean ca) {
		    if(ca.getId().equals(this.id) && bean==null){bean = ca;}
		}
		@Override
		public void visitIntermediateThrowEvent(IIntermediateThrowEventBean e) {
			if(e.getId().equals(this.id) && bean==null){bean = e;}
		}
		@Override
		public void visitIntermediateCatchEvent(IIntermediateCatchEventBean e) {
			if(e.getId().equals(this.id) && bean==null){bean = e;}
		}
		@Override
		public void visitDataInputAssociation(IDataInputAssociationBean dia, WithDataInputAssociationBean owner){
			if(dia.getId().equals(this.id) && bean==null){bean = dia;}
		}
		@Override
		public void visitDataOutputAssociation(IDataOutputAssociationBean doa, WithDataOutputAssociationBean owner){
			if(doa.getId().equals(this.id) && bean==null){bean = doa;}
		}
		@Override
		public void visitAssignement(IAssignmentBean a, IDataAssociationBean da) {
			if(a.getId().equals(this.id) && bean !=null){bean = a;}
		}
		@Override
		public void visitStartEvent(IStartEventBean seb) {
			if(seb.getId().equals(this.id) && bean==null){bean = seb;}
		}
		@Override
		public void visitLane(ILaneBean lane) {
			if(lane.getId().equals(this.id) && bean==null){bean = lane;}
		}
		@Override
		public void visitLaneSet(ILaneSetBean laneSet) {
			if(laneSet.getId().equals(this.id) && bean==null){bean = laneSet;}
		}
		@Override
		public void visitOperation(IOperationBean op) {
			if(op.getId().equals(this.id) && bean==null){bean = op;}
		}
		@Override
		public void visitItemDefinition(IItemDefinitionBean itemDefinition) {
			if(itemDefinition.getId().equals(this.id) && bean==null){bean = itemDefinition;}
		}
		@Override
		public void visitSFExpression(IExpressionBean exp){
			if(exp.getId().equals(this.id) && bean==null){bean = exp;}
		}
		@Override
		public void visitPartnerRole(IPartnerRoleBean partnerRole) {
			if(partnerRole.getId().equals(this.id) && bean==null){bean = partnerRole;}
		}
		@Override
		public void visitChildLaneSet(ILaneSetBean lsb) {
			if(lsb.getId().equals(this.id) && bean==null){bean = lsb;}
		}
		@Override
		public void visitMessageFlowMessage(IMessageBean msg) {
			if(msg.getId().equals(this.id) && bean==null){bean = msg;}
		}
		@Override
		public void visitOperationMessageIn(IMessageBean msg,IOperationBean opBean) {
			if(msg.getId().equals(this.id) && bean==null){bean = msg;}
		}
		@Override
		public void visitOperationMessageOut(IMessageBean msg,IOperationBean opBean) {
			if(msg.getId().equals(this.id) && bean==null){bean = msg;}
		}
		@Override
		public void visitOperationError(IErrorBean msg,IOperationBean opBean) {
			if(msg.getId().equals(this.id) && bean==null){bean = msg;}
		}
		@Override
		public void visitEventDefinition(IEventDefinitionBean ed){
		    if(ed.getId().equals(this.id) && bean==null){bean = ed;}
		}
	}

}
