/**
 * petalsbpm-service - BPMN Editor service - Copyright (C) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.ebmwebsourcing.petalsbpm.server.service.bpmndi.clientToServer;

import java.util.HashMap;
import java.util.Map;

import javax.xml.namespace.QName;

import com.ebmwebsourcing.easybox.api.XmlContextFactory;
import com.ebmwebsourcing.easybox.api.XmlObjectFactory;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.BPMNDiagram;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.BPMNEdge;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.BPMNLabel;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.BPMNLabelStyle;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.BPMNPlane;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.BPMNShape;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.Bounds;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.DiagramElement;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.Font;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.element.Point;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.type.TDiagramElement.Extension;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.type.TMessageVisibleKind;
import com.ebmwebsourcing.easybpmn.bpmn20diagram.api.type.TParticipantBandKind;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.diagramcommon.layout.IBounds;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.diagramcommon.layout.IPoint;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.extension.client.AttributeExtension;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.extension.client.ObjectExtension;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IDiagramElement;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.ILabel;
import com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.interchange.api.IStyle;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.collaboration.ILaneBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IMessageFlowBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.IParticipantBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.common.artifact.IAssociationBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.foundation.IBaseElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.api.standard.process.gateway.ISequenceFlowBean;
import com.ebmwebsourcing.petalsbpm.business.domain.bpmn2.to.standard.common.FlowElementBean;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.IBPMNDiagram;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.IBPMNEdge;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.IBPMNLabel;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.IBPMNLabelStyle;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.IBPMNPlane;
import com.ebmwebsourcing.petalsbpm.business.domain.di.api.IBPMNShape;
import com.ebmwebsourcing.petalsbpm.server.service.extension.ExtensionBindingManager;

public class BPMNDiagramGenerator {
	
	private static XmlObjectFactory objectFactory;
	private static ExtensionBindingManager bindingManager;
	private static Map<IParticipantBean,IBounds> pools;
	private static Map<ILaneBean,Bounds> lanes;

	public static BPMNDiagram generateBPMNDiagram(IBPMNDiagram diagram){
		objectFactory = new XmlContextFactory().newContext().getXmlObjectFactory();
		bindingManager = new ExtensionBindingManager();
		pools = new HashMap<IParticipantBean, IBounds>();
		lanes = new HashMap<ILaneBean, Bounds>();
		
		BPMNDiagram result = objectFactory.create(BPMNDiagram.class);
		result.setName(diagram.getName());
		result.setResolution(diagram.getResolution());
		result.setDocumentation(diagram.getDocument());
		
		result.setBPMNPlane(getPlane(diagram.getRootElement()));
		
		if(diagram.getStyles()!=null){
			for(IStyle style : diagram.getStyles()){
				result.addBPMNLabelStyle(getLabelStyle((IBPMNLabelStyle)style));
			}
		}
		
		return result;
	}
	
	
	private static BPMNLabelStyle getLabelStyle(IBPMNLabelStyle style) {
		BPMNLabelStyle result = objectFactory.create(BPMNLabelStyle.class);
		
		result.setId(style.getId());
		result.setFont(getFont(style.getFont()));
		
		return result;
	}


	private static Font getFont(
			com.ebmwebsourcing.geasytools.diagrameditor.domain.diagramdefinition.diagramcommon.layout.Font font) {
		Font result = objectFactory.create(Font.class);
		
		result.setIsBold(font.isBold());
		result.setIsItalic(font.isItalic());
		result.setIsStrikeThrough(font.isStrikeThrough());
		result.setIsUnderline(font.isUnderline());
		result.setName(font.getName());
		result.setSize(font.getSize());
		
		return result;
	}


	private static BPMNPlane getPlane(IBPMNPlane plane){
		BPMNPlane result = objectFactory.create(BPMNPlane.class);
		result.setId(plane.getId());
		result.setBpmnElement(new QName(plane.getModelElement().getId()));
		
		adaptExtensions(plane, result);
		
		for(IDiagramElement de : plane.getOwnedElements()){
			result.addDiagramElement(getDiagramElement(de));
		}
		
		return result;
	}


	private static DiagramElement getDiagramElement(IDiagramElement de) {
		if(de instanceof IBPMNEdge){
			return getEdge((IBPMNEdge)de);
		}
		else if(de instanceof IBPMNShape){
			return getShape((IBPMNShape)de);
		}
		return null;
	}


	private static BPMNShape getShape(IBPMNShape shape) {
		BPMNShape result = objectFactory.create(BPMNShape.class);
		
		result.setId(shape.getId());
		result.setBpmnElement(new QName(shape.getModelElement().getId()));
		result.setBounds(getBounds(shape.getBounds()));
		if(shape.getChoreographyActivityShape()!=null){
			result.setChoreographyActivityShape(new QName(shape.getChoreographyActivityShape().getId()));
		}
		result.setIsExpanded(shape.isExpanded());
		result.setIsHorizontal(shape.isHorizontal());
		result.setIsMarkerVisible(shape.isMarkerVisible());
		result.setIsMessageVisible(shape.isMessageVisible());
		
		adaptExtensions(shape, result);
		
		//make the flow elements coordinates absolute
		if(shape.getModelElement() instanceof IParticipantBean) {
			//first possibility : the shape is pool -> store its coordinates for its children
			if(shape.getParticipantBandKind()==null) {
				if(((IParticipantBean) shape.getModelElement()).getProcess()!=null) {
					pools.put((IParticipantBean) shape.getModelElement(), shape.getBounds());
				}
			}
			//second possibility : the shape is a participant band of a choreography task 
			// -> make its coordinates absolute using its parent choreography task
			else {
				Bounds shapeBounds = result.getBounds();
				IBounds choreoShapeBounds = shape.getChoreographyActivityShape().getBounds();
				shapeBounds.setX(shapeBounds.getX()+choreoShapeBounds.getX());
                shapeBounds.setY(shapeBounds.getY()+choreoShapeBounds.getY());
			}
		}
		else if(shape.getModelElement() instanceof ILaneBean) {
		    for(IParticipantBean p : pools.keySet()) {
		        if(p.getProcess().getLanes().contains(shape.getModelElement())) {
		            IBounds poolBounds = pools.get(p);
                    Bounds laneBounds = result.getBounds();
                    laneBounds.setX(laneBounds.getX()+poolBounds.getX());
                    laneBounds.setY(laneBounds.getY()+poolBounds.getY());
                    lanes.put((ILaneBean)shape.getModelElement(), laneBounds);
                    break;
		        }
		    }
		}
		else if(shape.getModelElement() instanceof FlowElementBean) {
		    for(ILaneBean lane : lanes.keySet()) {
                if(lane.getFlowNodes().contains(shape.getModelElement())) {
                    Bounds laneBounds = lanes.get(lane);
                    Bounds shapeBounds = result.getBounds();
                    shapeBounds.setX(shapeBounds.getX()+laneBounds.getX());
                    shapeBounds.setY(shapeBounds.getY()+laneBounds.getY());
                    break;
                }
            }
		}
		
		if(shape.getParticipantBandKind()!=null){
			switch(shape.getParticipantBandKind()){
			case bottom_initiating:
				result.setParticipantBandKind(TParticipantBandKind.bottom_initiating);
				break;
			case bottom_non_initiating:
				result.setParticipantBandKind(TParticipantBandKind.bottom_non_initiating);
				break;
			case middle_initiating:
				result.setParticipantBandKind(TParticipantBandKind.middle_initiating);
				break;
			case middle_non_initiating:
				result.setParticipantBandKind(TParticipantBandKind.middle_non_initiating);
				break;
			case top_initiating:
				result.setParticipantBandKind(TParticipantBandKind.top_initiating);
				break;
			case top_non_initiating:
				result.setParticipantBandKind(TParticipantBandKind.top_non_initiating);
				break;
			}
		}
		
		for(ILabel label : shape.getOwnedLabels()){
			result.setBPMNLabel(getLabel((IBPMNLabel)label));
		}
		
		return result;
	}


	private static BPMNEdge getEdge(IBPMNEdge edge) {
		BPMNEdge result = objectFactory.create(BPMNEdge.class);
		
		result.setBpmnElement(new QName(edge.getModelElement().getId()));
		result.setId(edge.getId());

		QName sourceRef = getEdgeSourceRef(edge);
		if(sourceRef!=null) result.setSourceElement(sourceRef);
		QName targetRef = getEdgeTargetRef(edge);
		if(targetRef!=null) result.setTargetElement(targetRef);
		
		adaptExtensions(edge, result);
		
		if(edge.getMessageVisibleKind()!=null){
			switch(edge.getMessageVisibleKind()){
			case initiating:
				result.setMessageVisibleKind(TMessageVisibleKind.initiating);
				break;
			case non_initiating:
				result.setMessageVisibleKind(TMessageVisibleKind.non_initiating);
				break;
			}
		}
		
		for(ILabel label : edge.getOwnedLabels()){
			result.setBPMNLabel(getLabel((IBPMNLabel)label));
		}
		
		for(IPoint point : edge.getWayPoints()){
			result.addWayPoint(getPoint(point));
		}
		
		return result;
	}


	private static BPMNLabel getLabel(IBPMNLabel label) {
		BPMNLabel result = objectFactory.create(BPMNLabel.class);
		
		result.setId(label.getId());
		result.setLabelStyle(new QName(label.getBPMNLabelStyle().getId()));
		
		adaptExtensions(label, result);
		
		result.setBounds(getBounds((com.ebmwebsourcing.petalsbpm.business.domain.di.impl.BPMNLabel)label));
		
		return result;
	}


	private static Bounds getBounds(IBounds bounds) {
		Bounds result = objectFactory.create(Bounds.class);
		
		result.setHeight(bounds.getHeight());
		result.setWidth(bounds.getWidth());
		result.setX(bounds.getX());
		result.setY(bounds.getY());
		
		return result;
	}
	
	
	private static Point getPoint(IPoint point) {
		Point result = objectFactory.create(Point.class);
		
		result.setX(point.getX());
		result.setY(point.getY());
		
		return result;
	}
	
	
	private static QName getEdgeSourceRef(IBPMNEdge edge) {
        IBaseElementBean edgeModelElement = edge.getModelElement();
        String sourceId = null;
        if(edgeModelElement instanceof ISequenceFlowBean) {
            sourceId = ((ISequenceFlowBean)edgeModelElement).getSourceNode().getId();
        }
        else if(edgeModelElement instanceof IMessageFlowBean) {
            sourceId = ((IMessageFlowBean)edgeModelElement).getSource().getId();
        }
        else if(edgeModelElement instanceof IAssociationBean) {
            sourceId = ((IAssociationBean)edgeModelElement).getSource().getId();
        }
        
        String edgeSourceElementId = edge.getSource().getModelElement().getId();
        
        if(sourceId!=null && sourceId.equals(edgeSourceElementId)) {
            return null;
        }
        return new QName(edge.getSource().getId());
	}
	
	private static QName getEdgeTargetRef(IBPMNEdge edge) {
        IBaseElementBean edgeModelElement = edge.getModelElement();
        String TargetId = null;
        if(edgeModelElement instanceof ISequenceFlowBean) {
            TargetId = ((ISequenceFlowBean)edgeModelElement).getTargetNode().getId();
        }
        else if(edgeModelElement instanceof IMessageFlowBean) {
            TargetId = ((IMessageFlowBean)edgeModelElement).getTarget().getId();
        }
        else if(edgeModelElement instanceof IAssociationBean) {
            TargetId = ((IAssociationBean)edgeModelElement).getTarget().getId();
        }
        
        String edgeTargetElementId = edge.getTarget().getModelElement().getId();
        
        if(TargetId!=null && TargetId.equals(edgeTargetElementId)) {
            return null;
        }
        return new QName(edge.getTarget().getId());
    }
	
	
	private static void adaptExtensions(IDiagramElement bean, DiagramElement elt){
		if(!bean.getObjectExtensions().isEmpty()){
			Extension ee = objectFactory.create(Extension.class);
			for(ObjectExtension obj : bean.getObjectExtensions()){
				ee.addAnyXmlObject(bindingManager.clientToServer(obj));
			}
			elt.setExtension(ee);
		}
		
		for(AttributeExtension att : bean.getAttributeExtensions()){
			elt.addOtherAttribute(new QName(att.getAttributeQNameNS(), att.getAttributeQNameLocalPart()), att.getAttributeValue());
		}
	}
}
