/**
 * Agreement Descriptor - SOA Tools Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * AgreementDescriptorBuilder.java
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.sla.agreement.descriptor.original;

/**
 * @author nsalatge - eBM WebSourcing
 *
 */
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.net.URL;

import javax.xml.XMLConstants;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.ggf.schemas.graap._2007._03.ws_agreement.Agreement;
import org.ggf.schemas.graap._2007._03.ws_agreement.AgreementTemplate;
import org.ggf.schemas.graap._2007._03.ws_agreement.GuaranteeTermType;
import org.w3._2005._08.addressing.EndpointReferenceType;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import sla.ebmwebsourcing.com.agreementextensions.TExpression;

import com.ebmwebsourcing.addressing.descriptor.AddressingException;
import com.ebmwebsourcing.addressing.descriptor.Endpoint;
import com.ebmwebsourcing.sla.agreement.descriptor.AgreementDescriptorException;


/**
 * This Class defines the Agreement Descriptor builder for the kernel.
 * <li> It can validate EndpointReference descriptor in XML form.
 * <li> It can build class bindings from a XML form.
 * <li> It can build the XML binding from classes form.
 * <p>
 * Note that this Builder is based on the original Agreement schema; it doesn't
 * control the EndpointReference extension parts.</p>
 * <p> This class is thread-safe.</p>
 *
 * @author Roland NAUDIN - EBM WebSourcing
 * @since PEtALS 2.2
 */
final public class AgreementDescriptorBuilder {

	/**
	 * Agreeement Namespace
	 */
	public static final String AGREEMENT_NAMESPACE = "http://schemas.ggf.org/graap/2007/03/ws-agreement";

	/**
	 * Agreeement schema name
	 */
	public static final String XSD_AGREEMENT = "agreement.xsd";

	/**
	 * Agreeement Extensions schema name
	 */
	public static final String XSD_AGREEMENT_EXTENSIONS = "agreementExtensions.xsd";


	private static JAXBContext jaxbContext;

	/**
	 * The JAXB unique unmarshaller.
	 */
	private static Unmarshaller unmarshaller = null;

	/**
	 * The exception raised during JAXB unique unmarshaller creation.
	 */
	private static AgreementDescriptorException unmarshallerCreationEx = null;

	/**
	 * The JAXB unique marshaller.
	 */
	private static Marshaller marshaller = null;

	/**
	 * The exception raised during JAXB unique marshaller creation.
	 */
	private static AgreementDescriptorException marshallerCreationEx = null;

	/*
	 * Private object initializations
	 */
	static {
		final SchemaFactory factory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
		try {
			// The EndpointReference schema resource is in the Jar where the class is
			// loaded
			final URL schemaUrl = AgreementDescriptorBuilder.class.getResource("/"
					+ XSD_AGREEMENT);
			final URL schemaUrlExt = AgreementDescriptorBuilder.class.getResource("/"
					+ XSD_AGREEMENT_EXTENSIONS);

			final URL schemaUrlEndpoint = Endpoint.class.getResource("/"
					+ Endpoint.XSD_ADDRESSING);
			final URL schemaUrlEndpointPolicy = Endpoint.class.getResource("/"
					+ Endpoint.XSD_POLICY);

			final Schema schema = factory
			.newSchema(new StreamSource[] {
					new StreamSource(schemaUrl.openStream()), new StreamSource(schemaUrlExt.openStream()),
					new StreamSource(schemaUrlEndpoint.openStream()), new StreamSource(schemaUrlEndpointPolicy.openStream())});

			// Unmarshaller initialization
			try {
				jaxbContext = JAXBContext
				.newInstance(new Class[] {
						org.ggf.schemas.graap._2007._03.ws_agreement.ObjectFactory.class,
						org.w3._2005._08.addressing.ObjectFactory.class,
						sla.ebmwebsourcing.com.agreementextensions.ObjectFactory.class
				});

				unmarshaller = jaxbContext.createUnmarshaller();
				unmarshaller.setSchema(schema);
			} catch (JAXBException e) {
				unmarshallerCreationEx = new AgreementDescriptorException(
						"Failed to create the JAXB unmarshaller", e);
			}

			// Marshaller initialization
			try {
				final JAXBContext jaxbContext = JAXBContext
				.newInstance(new Class[] {
						org.ggf.schemas.graap._2007._03.ws_agreement.ObjectFactory.class,
						org.w3._2005._08.addressing.ObjectFactory.class,
						sla.ebmwebsourcing.com.agreementextensions.ObjectFactory.class });
				marshaller = jaxbContext.createMarshaller();
				marshaller.setSchema(schema);
			} catch (JAXBException e) {
				marshallerCreationEx = new AgreementDescriptorException(
						"Failed to create the JAXB marshaller", e);
			}

		} catch (final IOException e) {
			final AgreementDescriptorException schemaCreationEx = new AgreementDescriptorException(
					"Failed to get resource '" + XSD_AGREEMENT
					+ "'from the current class-path", e);
			unmarshallerCreationEx = new AgreementDescriptorException(
					"Failed to create the JAXB unmarshaller",
					schemaCreationEx);

			marshallerCreationEx = new AgreementDescriptorException(
					"Failed to create the JAXB marshaller",
					schemaCreationEx);
		} catch (final SAXException e) {
			final AgreementDescriptorException schemaCreationEx = new AgreementDescriptorException(
					"Failed to parse resource '" + XSD_AGREEMENT + "'", e);
			unmarshallerCreationEx = new AgreementDescriptorException(
					"Failed to create the JAXB unmarshaller",
					schemaCreationEx);

			marshallerCreationEx = new AgreementDescriptorException(
					"Failed to create the JAXB marshaller",
					schemaCreationEx);
		}
	}

	/**
	 * The class is static, so no constructor available.
	 */
	private AgreementDescriptorBuilder() {
		// NOP
	}

	/**
	 * Validate the given Agreement descriptor against the EndpointReference schema.
	 *
	 * @param EndpointReferenceDescriptorStream
	 * @throws EndpointReferenceException
	 *             The validation has failed
	 * @throws EndpointReferenceDescriptorException
	 *             Failed to create the validator
	 */
	public static void validateAgreementDescriptor(final InputStream agreementDescriptorStream)
	throws AgreementDescriptorException {

		final Validator validator = getUnMarshaller().getSchema().newValidator();

		try {
			validator.validate(new StreamSource(agreementDescriptorStream));
		} catch (final SAXException e) {
			throw new AgreementDescriptorException(
					"Failed to validate EndpointReference descriptor against Agreement schema", e);
		} catch (final IOException e) {
			throw new AgreementDescriptorException(
					"Failed to validate EndpointReference descriptor against Agreement schema", e);
		}
	}

	/**
	 * Build the Java bindings from the Agreement descriptor in XML form. The
	 * validation against the Agreement schema is processed during the transformation.
	 *
	 * @param AgreementDescriptorStream
	 *            The Agreement descriptor in XML form
	 * @return The root object of the EndpointReference descriptor
	 * @throws AgreementDescriptorException
	 *             The exception raised during the unmarshaller creation or the
	 *             exception raised during the build of the java bindings.
	 */
	public static Agreement buildJavaAgreementDescriptor(final InputStream agreementDescriptorStream)
	throws AgreementDescriptorException {

		try {
			// TODO : Check if it is a Thread safe method
			final JAXBElement<Agreement> agreementBinding = getUnMarshaller().unmarshal(
					new StreamSource(agreementDescriptorStream), Agreement.class);

			return agreementBinding.getValue();

		} catch (JAXBException e) {
			throw new AgreementDescriptorException(
					"Failed to build Java bindings from Agreement descriptor XML document", e);
		}
	}

	public static JAXBElement<Agreement> buildJavaJAxbAgreementDescriptor(final InputStream agreementDescriptorStream)
	throws AgreementDescriptorException {

		try {
			// TODO : Check if it is a Thread safe method
			final JAXBElement<Agreement> agreementBinding = getUnMarshaller().unmarshal(
					new StreamSource(agreementDescriptorStream), Agreement.class);

			return agreementBinding;

		} catch (JAXBException e) {
			throw new AgreementDescriptorException(
					"Failed to build Java bindings from Agreement descriptor XML document", e);
		}
	}

	/**
	 * Build the XML nodes from the EndpointReference descriptor in Java classes form.
	 *
	 * @param EndpointReferenceDescriptorClass
	 *            The EndpointReference Descriptor root class
	 * @param EndpointReferenceDescriptorNode
	 *            The XML Node to fill with the EndpointReference descriptor XML nodes
	 * @throws EndpointReferenceDescriptorException
	 *             The exception raised during the unmarshaller creation or the
	 *             exception raised during the build of the XML nodes.
	 */
	public static Element buildXmlAgreementDescriptor(final Agreement agreementDescriptor) throws AgreementDescriptorException {
		Document doc = null;
		try {
			doc = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();


			// TODO : Check if it is a Thread safe method
			getMarshaller().marshal(agreementDescriptor, doc);

		} catch (JAXBException ex) {
			throw new AgreementDescriptorException(
					"Failed to build XML binding from Agreement descriptor Java classes", ex);
		} catch (ParserConfigurationException ex) {
			throw new AgreementDescriptorException(
					"Failed to build XML binding from Agreement descriptor Java classes", ex);

		}
		return doc.getDocumentElement();
	}


	public static Element buildXmlGuaranteeDescriptor(final GuaranteeTermType guarantee) throws AgreementDescriptorException {
		Document doc = null;
		try {
			doc = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();


			// TODO : Check if it is a Thread safe method
			getMarshaller().marshal(guarantee, doc);

		} catch (JAXBException ex) {
			throw new AgreementDescriptorException(
					"Failed to build XML binding from Agreement descriptor Java classes", ex);
		} catch (ParserConfigurationException ex) {
			throw new AgreementDescriptorException(
					"Failed to build XML binding from Agreement descriptor Java classes", ex);

		}
		return doc.getDocumentElement();
	}


	public static Element buildXmlEndpointDescriptor(final Endpoint epr) throws AgreementDescriptorException {
		Document doc = null;
		try {
			doc = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();

			// TODO : Check if it is a Thread safe method
			getMarshaller().marshal(epr, doc);

		} catch (JAXBException ex) {
			throw new AgreementDescriptorException(
					"Failed to build XML binding from Agreement descriptor Java classes", ex);
		} catch (ParserConfigurationException ex) {
			throw new AgreementDescriptorException(
					"Failed to build XML binding from Agreement descriptor Java classes", ex);

		}
		return doc.getDocumentElement();
	}

	public static JAXBElement<EndpointReferenceType> buildXmlJaxbEndpointDescriptor(final Endpoint epr)
	throws AgreementDescriptorException {

		try {
			InputStream input = new ByteArrayInputStream(epr.marshall2String().getBytes());
			// TODO : Check if it is a Thread safe method
			final JAXBElement<EndpointReferenceType> endpointReferenceBinding = getUnMarshaller().unmarshal(
					new StreamSource(input), EndpointReferenceType.class);

			return endpointReferenceBinding;

		} catch (JAXBException e) {
			throw new AgreementDescriptorException(
					"Failed to build Java bindings from EndpointReference descriptor XML document", e);
		} catch (AgreementDescriptorException e) {
			throw new AgreementDescriptorException(
					"Failed to build Java bindings from EndpointReference descriptor XML document", e);
		} catch (AddressingException e) {
			throw new AgreementDescriptorException(
					"Failed to build Java bindings from EndpointReference descriptor XML document", e);
		}
	}

	/**
	 * Build the XML String from the Agreement descriptor in Java classes form.
	 *
	 * @param AgreementDescriptorClass
	 *            The Agreement Descriptor root class
	 * @return The String to fill with the Agreement descriptor XML
	 * @throws AgreementDescriptorException
	 *             The exception raised during the marshaller creation or the
	 *             exception raised during the build of the XML string.
	 */
	public static String buildXmlStringAgreementdescriptor(final Agreement agreementDescriptor)
	throws AgreementDescriptorException {

		try {
			final StringWriter stringWriter = new StringWriter();
			// TODO : Check if it is a Thread safe method
			getMarshaller().marshal(agreementDescriptor, stringWriter);
			return stringWriter.toString();
		} catch (JAXBException e) {
			throw new AgreementDescriptorException(
					"Failed to build XML binding from Agreement descriptor Java classes", e);
		}
	}

	/**
	 * Get the unmarshaller instance.
	 *
	 * @return the unmarshaller instance
	 * @throws AgreementDescriptorException
	 *             The exception raised during the unmarshaller creation.
	 */
	private static Unmarshaller getUnMarshaller() throws AgreementDescriptorException {

		if (unmarshallerCreationEx != null) {
			throw unmarshallerCreationEx;
		}

		return unmarshaller;
	}

	/**
	 * Get the marshaller instance.
	 *
	 * @return the marshaller instance
	 * @throws AgreementDescriptorException
	 *             The exception raised during the marshaller creation.
	 */
	public static Marshaller getMarshaller() throws AgreementDescriptorException {

		if (marshallerCreationEx != null) {
			throw marshallerCreationEx;
		}

		return marshaller;
	}

	/**
	 * @return the unmarshaller
	 */
	public static Unmarshaller getUnmarshaller() throws AgreementDescriptorException  {
		if (unmarshallerCreationEx != null) {
			throw unmarshallerCreationEx;
		}

		return unmarshaller;
	}

	/**
	 * @param marshaller the marshaller to set
	 */
	public static void setMarshaller(Marshaller marshaller) {
		AgreementDescriptorBuilder.marshaller = marshaller;
	}

	/**
	 * @return the jaxbContext
	 */
	public static JAXBContext getJaxbContext() {
		return jaxbContext;
	}

	/**
	 * @param jaxbContext the jaxbContext to set
	 */
	public static void setJaxbContext(JAXBContext jaxbContext) {
		AgreementDescriptorBuilder.jaxbContext = jaxbContext;
	}
}