package com.ebmwebsourcing.webcommons.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import org.springframework.util.ResourceUtils;

public class InputStreamUtil {
    private static final String FILE_SCHEME = "file";

    /**
     * Default value is 2048.
     */
    public static final int DEFAULT_BUFFER_SIZE = 2048;

    public static InputStream getInputStream(String resourceLocation) throws IOException,
            URISyntaxException {
        return getInputStream(ResourceUtils.getURL(resourceLocation));
    }

    public static InputStream getInputStream(URI resourceLocation) throws IOException,
            URISyntaxException {
        return getInputStream(ResourceUtils.getURL(resourceLocation.toString()));
    }

    public static InputStream getInputStream(URL resourceLocation) throws IOException,
            URISyntaxException {
        InputStream result = null;
        if (FILE_SCHEME.equals(resourceLocation.getProtocol())) {
            final File f = new File(resourceLocation.toURI());
            result = new FileInputStream(f);
        } else {
            result = ResourceUtils.getURL(resourceLocation.toString()).openStream();
        }
        return result;
    }

    /**
     * Copies information from the input stream to the output stream using a
     * default buffer size of 2048 bytes.
     * 
     * @throws java.io.IOException
     */
    public static void copy(InputStream input, OutputStream output) throws IOException {
        copy(input, output, DEFAULT_BUFFER_SIZE);
    }

    /**
     * Copies information from the input stream to the output stream using the
     * specified buffer size
     * 
     * @throws java.io.IOException
     */
    public static void copy(InputStream input, OutputStream output, int bufferSize)
            throws IOException {
        byte[] buf = new byte[bufferSize];
        int bytesRead = input.read(buf);
        while (bytesRead != -1) {
            output.write(buf, 0, bytesRead);
            bytesRead = input.read(buf);
        }
        output.flush();
    }

    /**
     * Copies information between specified streams and then closes both of the
     * streams.
     * 
     * @throws java.io.IOException
     */
    public static void copyThenClose(InputStream input, OutputStream output) throws IOException {
        copy(input, output);
        input.close();
        output.close();
    }

    /**
     * @returns a byte[] containing the information contained in the specified
     *          InputStream.
     * @throws java.io.IOException
     */
    public static byte[] getBytes(InputStream input) throws IOException {
        ByteArrayOutputStream result = new ByteArrayOutputStream();
        copy(input, result);
        result.close();
        return result.toByteArray();
    }
}
