/**
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.wsstar.notification.service.persistence;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import com.ebmwebsourcing.wsstar.notification.definition.basenotification.api.SubscriptionManagerRP;
import com.ebmwebsourcing.wsstar.notification.definition.brokerednotification.api.PublisherRegistrationRP;
import com.ebmwebsourcing.wsstar.notification.definition.inout.WSNotificationReader;
import com.ebmwebsourcing.wsstar.notification.definition.inout.WSNotificationWriter;
import com.ebmwebsourcing.wsstar.notification.definition.utils.WSNotificationException;

/**
 * The persistence manager used to persist consumer subscription and producer registration so
 * that they can be reloaded when needed.
 * 
 * @author Thierry Déjean - eBM WebSourcing
 */
public class WsnPersistence {

    /**
     * The persistence folders path
     */
    private File subscriptions_folder;
    private File registrations_folder;
   
    /**
     * 
     * @param subscriptions_path
     * @param registrations_path
     */
    public WsnPersistence(File subscriptions_path,File registrations_path) {

    	if (subscriptions_path == null) {
    		throw new NullPointerException("Persistence folder must not be null");
    	}
    	
    	if (!subscriptions_path.exists()) {
    		subscriptions_path.mkdirs();
    	}
    	this.subscriptions_folder = subscriptions_path;    	    	

    	// --- registration persistance is not used by WsnNotificationProducer
    	// --- component.It is only used by WsnNotification Broker component
    	if (registrations_path != null) {

    		if (!registrations_path.exists()) {
    			registrations_path.mkdirs();
    		}
    		
    		this.registrations_folder = registrations_path;
    	}
    }

    /**
     * 
     * @param subscription
     * @throws Exception
     */
    public void persist(SubscriptionManagerRP subscriptionRP, String uuid) throws WSNotificationException {
        
    	File persist = new File(this.subscriptions_folder.getAbsoluteFile(),uuid);
        if (persist == null) {
            throw new WSNotificationException("The persistance file can not be created");
        }
        if (!persist.exists()) {
            try {
				persist.createNewFile();
			} catch (IOException e) {
				throw new WSNotificationException(e);
			}
        }        
        WSNotificationWriter.getInstance().writeSubscriptionManagerRP(subscriptionRP, persist);
    }

    /**
     * 
     * @param subscription
     * @throws Exception
     */
    public void persist(PublisherRegistrationRP registrationRP, String uuid) throws WSNotificationException {
        
    	File persist = new File(this.registrations_folder,uuid);
        if (persist == null) {
            throw new WSNotificationException("The persistance file can not be created");
        }
        if (!persist.exists()) {
            try {
				persist.createNewFile();
			} catch (IOException e) {
				throw new WSNotificationException(e);
			}
        }        
        WSNotificationWriter.getInstance().writePublisherRegistrationRP(registrationRP, persist);
    }
    
    /**
     * 
     * @return
     * @throws WSNotificationException 
     */
    public Map<String,PublisherRegistrationRP> getRegistrationsToRestore() throws WSNotificationException{

    	Map<String, PublisherRegistrationRP> result = new HashMap<String, PublisherRegistrationRP>();

    	if (this.registrations_folder.exists()){
    		File[] persistedRegistrations = this.registrations_folder.listFiles();
    		for (int i = 0; i < persistedRegistrations.length; i++) {
    			result.put(persistedRegistrations[i].getName(),
    					WSNotificationReader.getInstance().readPublisherRegistrationRP(persistedRegistrations[i]));
    		}
    	}	
    	return result;
    }

    /**
     * 
     * @return
     * @throws WSNotificationException 
     */
    public Map<String,SubscriptionManagerRP> getSubscriptionsToRestore() throws WSNotificationException{

    	Map<String, SubscriptionManagerRP> result = new HashMap<String, SubscriptionManagerRP>();
    	if (this.subscriptions_folder.exists()){
    		File[] persistedSubsriptionFiles = this.subscriptions_folder.listFiles();
    		for (int i = 0; i < persistedSubsriptionFiles.length; i++) {
    			result.put(persistedSubsriptionFiles[i].getName(),
    					WSNotificationReader.getInstance().readSubscriptionManagerRP(persistedSubsriptionFiles[i]));
    		}
    	}	    	
    	return result;
    }
    
    
    /**
     * 
     * @param uuid
     * @throws WSNotificationException
     */
    public void removeSubscription(String uuid) throws WSNotificationException{
    	File subscriptionToRemove = new File(this.subscriptions_folder,uuid);
    	if (subscriptionToRemove.exists())
    		if (!subscriptionToRemove.delete())
    			throw new WSNotificationException("Deletion of subscription \"" + uuid + "\" failed !");
    }
    
    /**
     * 
     * @param uuid
     * @throws WSNotificationException
     */
    public void removeRegistration(String uuid) throws WSNotificationException{
    	File registrationToRemove = new File(this.registrations_folder,uuid);
    	if (registrationToRemove.exists())
    		if (!registrationToRemove.delete())
    			throw new WSNotificationException("Deletion of registration \"" + uuid + "\" failed !");
    }
      
}
