/**
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */
package com.ebmwebsourcing.wsstar.notification.service.topic;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.xml.namespace.QName;

import org.jdom.Content;
import org.jdom.Document;
import org.jdom.JDOMException;
import org.jdom.Namespace;
import org.jdom.input.DOMBuilder;
import org.jdom.input.SAXBuilder;
import org.jdom.output.DOMOutputter;
import org.ow2.easywsdl.schema.api.extensions.NamespaceMapperImpl;
import org.ow2.easywsdl.schema.util.XMLPrettyPrinter;

import com.ebmwebsourcing.wsstar.notification.definition.WSNotificationFactory;
import com.ebmwebsourcing.wsstar.notification.definition.basenotification.api.TopicExpressionType;
import com.ebmwebsourcing.wsstar.notification.definition.inout.WSNotificationReader;
import com.ebmwebsourcing.wsstar.notification.definition.inout.WSNotificationWriter;
import com.ebmwebsourcing.wsstar.notification.definition.topics.WstopConstants;
import com.ebmwebsourcing.wsstar.notification.definition.topics.api.TopicNamespaceType;
import com.ebmwebsourcing.wsstar.notification.definition.topics.api.TopicSetType;
import com.ebmwebsourcing.wsstar.notification.definition.topics.api.TopicType;
import com.ebmwebsourcing.wsstar.notification.definition.utils.WSNotificationException;
import com.ebmwebsourcing.wsstar.notification.extension.utils.WsnExtensionConstants;
import com.ebmwebsourcing.wsstar.notification.extension.utils.WsnSpecificTypeHelper;
import com.ebmwebsourcing.wsstar.notification.service.fault.ResourceUnknownFault;
import com.ebmwebsourcing.wsstar.notification.service.fault.TopicExpressionDialectUnknownFault;
import com.ebmwebsourcing.wsstar.notification.service.fault.TopicNotSupportedFault;
import com.ebmwebsourcing.wsstar.notification.service.fault.WSNotificationFault;
import com.ebmwebsourcing.wsstar.notification.service.fault.WsnFaultMessageConstants;


/**
 * 
 * This class must implement the singleton pattern
 * (solution chosen : see http://en.wikipedia.org/wiki/Singleton_pattern)
 * 
 * @author tdejean - eBM WebSourcing
 *
 */
public class WstopTopicManager {

	private final List<String> supportedDialectsForSetOfTopics = Collections.synchronizedList(new ArrayList<String>());

	private Document supportedTopics = null;

	private Logger logger = null;

	private final XPathAnalyzer xpathAnalyser;
	
	/*
	private WstopTopicManager() {
		this.supportedDialectsForSetOfTopics.add(WstopConstants.FULL_TOPIC_EXPRESSION_DIALECT_URI);
		this.supportedDialectsForSetOfTopics.add(WstopConstants.XPATH_TOPIC_EXPRESSION_DIALECT_URI);

		this.logger = Logger.getLogger(WstopTopicManager.class.getName());
		this.xpathAnalyser = new XPathAnalyzer();
	}
	*/
	
	/**
	 * SingletonHolder is loaded on the first execution of Singleton.getInstance()
	 * or the first access to SingletonHolder.INSTANCE, not before.
	 */
	/*private static class WstopTopicManagerHolder {
		private final static WstopTopicManager INSTANCE = new WstopTopicManager();
	}
	*/
	public WstopTopicManager(final InputStream supportedTopicSetConfig) {
		this.supportedDialectsForSetOfTopics.add(WstopConstants.FULL_TOPIC_EXPRESSION_DIALECT_URI);
		this.supportedDialectsForSetOfTopics.add(WstopConstants.XPATH_TOPIC_EXPRESSION_DIALECT_URI);

		this.logger = Logger.getLogger(WstopTopicManager.class.getName());
		this.xpathAnalyser = new XPathAnalyzer();
		try {
			this.supportedTopics = this.convertFromStreamToJDocument(supportedTopicSetConfig);
		}  catch (final JDOMException e) {
			this.logger.log(Level.WARNING, "There is a problem with your configuration file of \"supported TopicSet\". (does it  well formatted?)",e);
			//throw new WSNotificationException(e);
		} catch (final IOException e) {
			this.logger.log(Level.WARNING, "There is a problem with your configuration file of \"supported TopicSet\". (does it  well formatted?)",e);
			//throw new WSNotificationException(e);
		}
	}
	/*
	public static WstopTopicManager getInstance(final InputStream supportedTopicSetConfig) {
		final WstopTopicManager instance = WstopTopicManagerHolder.INSTANCE;

		if (instance.getSupportedTopics() == null) {
			try {
				instance.supportedTopics = WstopTopicManager.convertFromStreamToJDocument(supportedTopicSetConfig);
			} catch (final JDOMException e) {
				instance.logger.log(Level.WARNING, "There is a problem with your configuration file of \"supported TopicSet\". (does it  well formatted?)",e);
				//throw new WSNotificationException(e);
			} catch (final IOException e) {
				instance.logger.log(Level.WARNING, "There is a problem with your configuration file of \"supported TopicSet\". (does it  well formatted?)",e);
				//throw new WSNotificationException(e);
			} finally {
				if (instance.supportedTopics == null) {
					instance.supportedTopics = new Document(instance.createEmptyTopicSet());
				}
			}
		}
		return instance;
	}*/
	
	
	public WstopTopicManager(final InputStream topicNamespaceStream, final List<String> supportedTopics) throws WSNotificationException{
		this.supportedDialectsForSetOfTopics.add(WstopConstants.FULL_TOPIC_EXPRESSION_DIALECT_URI);
		this.supportedDialectsForSetOfTopics.add(WstopConstants.XPATH_TOPIC_EXPRESSION_DIALECT_URI);

		this.logger = Logger.getLogger(WstopTopicManager.class.getName());
		this.xpathAnalyser = new XPathAnalyzer();
		try {
			InputStream supportedTopicSetConfig = null;
			if(topicNamespaceStream != null) {
				final Document jdom = WstopTopicManager.convertFromStreamToJDocument(topicNamespaceStream);
				// convert jdom to dom
				org.w3c.dom.Document domDocument = null;
				final DOMOutputter converter = new DOMOutputter();

				domDocument = converter.output(jdom);

				final TopicNamespaceType topicNamespace = WSNotificationReader.getInstance().readTopicNamespaceType(domDocument);
				final TopicSetType topicSet = this.createTopicSetFromTopicNamespace(topicNamespace, supportedTopics);
				final org.w3c.dom.Document topicSetDom = WSNotificationWriter.getInstance().writeTopicSetType(topicSet);
				supportedTopicSetConfig = new ByteArrayInputStream(XMLPrettyPrinter.prettyPrint(topicSetDom).getBytes());
				this.supportedTopics = WstopTopicManager.convertFromStreamToJDocument(supportedTopicSetConfig);
			}
		}  catch (final JDOMException e1) {
			throw new WSNotificationException(e1);
		} catch (final IOException e1) {
			throw new WSNotificationException(e1);
		}
	}
	
	/*
	public static WstopTopicManager getInstance(final InputStream topicNamespaceStream, final List<String> supportedTopics) throws WSNotificationException {
		final WstopTopicManager instance = WstopTopicManagerHolder.INSTANCE;
		try {
			InputStream supportedTopicSetConfig = null;
			if(topicNamespaceStream != null) {
				final Document jdom = WstopTopicManager.convertFromStreamToJDocument(topicNamespaceStream);
				// convert jdom to dom
				org.w3c.dom.Document domDocument = null;
				final DOMOutputter converter = new DOMOutputter();

				domDocument = converter.output(jdom);

				final TopicNamespaceType topicNamespace = WSNotificationFactory.getInstance().createWSNotificationReader().readTopicNamespaceType(domDocument);
				final TopicSetType topicSet = instance.createTopicSetFromTopicNamespace(topicNamespace, supportedTopics);
				final org.w3c.dom.Document topicSetDom = WSNotificationFactory.getInstance().createWSNotificationWriter().writeTopicSetType(topicSet);
				supportedTopicSetConfig = new ByteArrayInputStream(XMLPrettyPrinter.prettyPrint(topicSetDom).getBytes());
				instance.supportedTopics = WstopTopicManager.convertFromStreamToJDocument(supportedTopicSetConfig);
			}
		} catch (final JDOMException e1) {
			throw new WSNotificationException(e1);
		} catch (final IOException e1) {
			throw new WSNotificationException(e1);
		}
		return instance;
	}*/

	/**
	 *  Parse a xml file to confugure supported Topics
	 * @param InputStream supported Topics config as stream
	 * @return A Document representation of a topicNamspaceType object
	 * @throws IOException
	 * @throws JDOMException
	 */
	private static Document convertFromStreamToJDocument(final InputStream supportedTopicsConfig) throws JDOMException, IOException{
		Document result = null;
		final SAXBuilder builder = new SAXBuilder();
		result = builder.build(supportedTopicsConfig);
		return result;
	}

	/**
	 *
	 * Method use to display current supported topic tree
	 * with stored registration/subscription uuids.
	 * Really usefull in "DEBUD" mode ;)
	 *
	 * @return a {@link String} representation of supported topic tree
	 */
	private String displayCurrentSupportedTopicList(){
		String log = "\n ~~~ Supported Topics List (with stored registration/subscription uuids): ~~~ \n";

		try {
			// create JDOM to DOM converter:
			final DOMOutputter output = new DOMOutputter();
			// here we have a DOM-document:
			final org.w3c.dom.Document dom = output.output(this.supportedTopics);

			log += XMLPrettyPrinter.prettyPrint(dom);

		} catch (final Exception ex) {
			ex.printStackTrace();
		}

		log +="\n \t\t ~~~~~~ \n";

		return log;

	}

	public void setLogger(final Logger logger) {
		this.logger = logger;
	}

	public void addSupportedDialect(final String dialactURI){
		this.supportedDialectsForSetOfTopics.add(dialactURI);
	}

	public List<String> getSupportedDialectForSetOfTopics() {
		return this.supportedDialectsForSetOfTopics;
	}

	public Document getSupportedTopics() {
		return this.supportedTopics ;
	}

	/**
	 * Use specifically for test Unit (called in setUp() method)
	 */
	public void RAZSupportedTopicSet(final InputStream defaultConfig){
		try {
			this.supportedTopics = WstopTopicManager.convertFromStreamToJDocument(defaultConfig);
		} catch (final JDOMException e) {
			this.logger.log(Level.WARNING, "There is a problem with your configuration file of \"supported TopicSet\". (does it  well formatted?)",e);
			//throw new WSNotificationException(e);
		} catch (final IOException e) {
			this.logger.log(Level.WARNING, "There is a problem with your configuration file of \"supported TopicSet\". (does it  well formatted?)",e);
			//throw new WSNotificationException(e);
		}
	}

	private boolean isValideDialect(final String topicExpressionDialect){
		boolean result = false;
		for (final String knownDialect : this.supportedDialectsForSetOfTopics) {
			result |= topicExpressionDialect.compareTo(knownDialect) == 0;
		}
		return result;
	}
	/**
	 * Check if the topic expression contains in a "Subscribe" or "RegisterPublisher" request
	 * is allowed (Is Dialect supported ? Is Expression Well formed ? , does the topic
	 * expression describe a sub-set of supported topics ? ...) according to TopicNamesapceType
	 * 
	 * @param expression The topic expression contains in the request payload
	 * @param topics Supported topics organized as a tree (see TopicNamespaceType)
	 * @param isSubscribeRequest boolean is to build fault
	 * @return true if the expression is accepted, false if it is not
	 * @throws WSNotificationException
	 * @throws WSNotificationFault
	 */
	public boolean acceptSubscribeOrRegisterTopicExpression(final TopicExpressionType topicExprToCheck, final boolean isSubscribeRequest) throws WSNotificationException, WSNotificationFault{
		boolean result = true;

		final String faultMsg = isSubscribeRequest ?
				WsnFaultMessageConstants.WsnbSubcribeFaultDescriptions.TOPIC_EXPRESSION_DIALECT_UNKNOWN_FAULT_DESC :
					WsnFaultMessageConstants.WsnbrRegisterPublisherFaultDescriptions.TOPIC_EXPRESSION_DIALECT_UNKNOWN_FAULT_DESC;

		if (!this.isValideDialect(topicExprToCheck.getDialect())) {
			throw new TopicExpressionDialectUnknownFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,faultMsg);
		}

		//String supportedNS = supportedTopics.getRootElement().getAttributeValue("targetNamespace");

		/*for (QName topExprNs : topicExprToCheck.getTopicNameSpace()) {
			 isNamespaceCompilant |= (((topExprNs.getNamespaceURI()).compareTo(supportedNS)) == 0);
		}*/
		/*if (!isNamespaceCompilant(topicExprToCheck.getTopicNameSpace()))
		//if (topicExprToCheck.getTopicNameSpace().compareTo(supportedNS) != 0)
			//return false;
			throw new InvalidTopicExpressionFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,
					WsnFaultMessageConstants.WsnbrRegisterPublisherFaultDescriptions.INVALID_TOPIC_EXPRESSION_FAULT_DESC);
		 */
		// ----- XPATH Analyser Part ------
		final String topicExprStringForm = topicExprToCheck.getContent();
		final NamespaceMapperImpl context = new NamespaceMapperImpl();

		final String formattedTopicExpression = "";
		for (final QName ns : topicExprToCheck.getTopicNameSpace()) {
			context.addNamespace(ns.getLocalPart(), ns.getNamespaceURI());
		}/*
			if (formattedTopicExpression.length()>0)
				formattedTopicExpression += " or ";
			formattedTopicExpression += topicExprStringForm +"[@wstop:topic='true']";
		}*/
		context.addNamespace(WstopConstants.WS_TOPICS_PREFIX,WstopConstants.WS_TOPICS_NAMESPACE_URI);
		context.addNamespace(WsnExtensionConstants.EBM_RESOURCEIDS_PREFIX,WsnExtensionConstants.EBM_RESOURCEIDS_NAMESPACE_URI);

		final List<?> res = this.xpathAnalyser.evaluate(topicExprStringForm/*formattedTopicExpression*/ , this.supportedTopics.getRootElement(), context);
		result = res != null && res.size() > 0 ;
		/*if (result) {
			org.jdom.Element currentResId = null;
			for (Object topicNodeItem : res) {
				currentResId = ((org.jdom.Element)topicNodeItem).getChild(WstopConstants.RESOURCE_IDS_QNAME.getLocalPart(),
						Namespace.getNamespace(WstopConstants.RESOURCE_IDS_QNAME.getNamespaceURI()));
				if(currentResId != null) {
					currentResId.getChildren().add((new org.jdom.Element(WstopConstants.SUBSCRIPTION_ID_QNAME.getLocalPart(),
							WstopConstants.EBM_RESOURCEIDS_PREFIX,WstopConstants.SUBSCRIPTION_ID_QNAME.getNamespaceURI())).setText("fictiveResourceId"));
				}
			}
			//List<?> resourceIdsList = ((org.jdom.Element)res.get(0)).getChild("ResourcesIds", "")
		}
		 */

		//List<?> res = xpathAnalyser.evaluate(formattedTopicExpression , this.supportedTopics.getRootElement(), context);
		//result = (res != null) && res.size() > 0 ;
		// --------------------------------
		return result;
	}


	/**
	 * 
	 * @param topicExpr
	 * @param registrationId
	 * @return
	 * @throws WSNotificationException
	 * @throws WSNotificationFault
	 */
	public void storeNewRegistration(final TopicExpressionType topicExprToCheck, final String registrationId) throws WSNotificationException, WSNotificationFault{

		boolean resourceStored = false;

		final String faultMsg = WsnFaultMessageConstants.WsnbrRegisterPublisherFaultDescriptions.TOPIC_EXPRESSION_DIALECT_UNKNOWN_FAULT_DESC;

		if (!this.isValideDialect(topicExprToCheck.getDialect())) {
			throw new TopicExpressionDialectUnknownFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,faultMsg);
		}

		// ----- XPATH Analyser Part ------
		final String topicExprStringForm = topicExprToCheck.getContent();
		final NamespaceMapperImpl context = new NamespaceMapperImpl();

		final String formattedTopicExpression = "";
		for (final QName ns : topicExprToCheck.getTopicNameSpace()) {
			context.addNamespace(ns.getLocalPart(), ns.getNamespaceURI());
		}
		/*	if (formattedTopicExpression.length()>0)
				formattedTopicExpression += " or ";
			formattedTopicExpression += topicExprStringForm +"[@wstop:topic='true']";
		}*/
		context.addNamespace(WstopConstants.WS_TOPICS_PREFIX,WstopConstants.WS_TOPICS_NAMESPACE_URI);
		context.addNamespace(WsnExtensionConstants.EBM_RESOURCEIDS_PREFIX,WsnExtensionConstants.EBM_RESOURCEIDS_NAMESPACE_URI);

		final List<?> res = this.xpathAnalyser.evaluate(topicExprStringForm/*formattedTopicExpression*/ , this.supportedTopics.getRootElement(), context);

		if (res != null && res.size() > 0) {
			org.jdom.Element currentResId = null;
			for (final Object topicNodeItem : res) {
				currentResId = ((org.jdom.Element)topicNodeItem).getChild(WsnExtensionConstants.RESOURCE_IDS_QNAME.getLocalPart(),
						Namespace.getNamespace(WsnExtensionConstants.RESOURCE_IDS_QNAME.getNamespaceURI()));
				if(currentResId == null) {
					currentResId = new org.jdom.Element(WsnExtensionConstants.RESOURCE_IDS_QNAME.getLocalPart(),
							Namespace.getNamespace(WsnExtensionConstants.RESOURCE_IDS_QNAME.getNamespaceURI()));
					((org.jdom.Element)topicNodeItem).addContent(currentResId);
				}
				//if(currentResId != null) {
				currentResId.getChildren().add(new org.jdom.Element(WsnExtensionConstants.REGISTRATION_ID_QNAME.getLocalPart(),
						WsnExtensionConstants.EBM_RESOURCEIDS_PREFIX,WsnExtensionConstants.REGISTRATION_ID_QNAME.getNamespaceURI()).setText(registrationId));
				resourceStored = true;
				//}
			}
		}

		if (!resourceStored) {
			throw new TopicNotSupportedFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,
					WsnFaultMessageConstants.WsnbrRegisterPublisherFaultDescriptions.TOPIC_NOT_SUPPORTED_FAULT_DESC);
		}
                if( this.logger.isLoggable(Level.FINE) )
                    this.logger.log(Level.FINE, this.displayCurrentSupportedTopicList());
	}

	/**
	 * 
	 * 
	 * @param registrationId
	 * @return
	 * @throws WSNotificationException
	 * @throws WSNotificationFault
	 */
	public void removeExistingRegistration(final String registrationId) throws WSNotificationException, WSNotificationFault{

		boolean resourceRemoved = false;
		// ----- XPATH Analyser Part ------
		final String resourceIdsXpatExpr = "*//" + WsnExtensionConstants.RESOURCE_IDS_QNAME.getPrefix() + ":" +WsnExtensionConstants.RESOURCE_IDS_QNAME.getLocalPart();
		final NamespaceMapperImpl context = new NamespaceMapperImpl();
		context.addNamespace(WsnExtensionConstants.EBM_RESOURCEIDS_PREFIX,WsnExtensionConstants.EBM_RESOURCEIDS_NAMESPACE_URI);

		final List<?> res = this.xpathAnalyser.evaluate(resourceIdsXpatExpr , this.supportedTopics.getRootElement(), context);

		if (res != null && res.size() > 0) {
			//final org.jdom.Element nodeToDelete = null;
			final String findIdXpathExpr = WsnExtensionConstants.REGISTRATION_ID_QNAME.getPrefix() +
				":" + WsnExtensionConstants.REGISTRATION_ID_QNAME.getLocalPart() +"[child::text()=\"" + registrationId+ "\"]";
			List<?> subRes = null;
			for (final Object nodeItem : res) {
				subRes = this.xpathAnalyser.evaluate(findIdXpathExpr , (org.jdom.Element)nodeItem, context);
				if (subRes != null && subRes.size() > 0) {
					((org.jdom.Element)nodeItem).removeContent((Content)subRes.get(0));
					resourceRemoved = true;
				}
			}
		}
		
		if (!resourceRemoved) {
			throw new ResourceUnknownFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,
					WsnFaultMessageConstants.WsnbrDestroyRegistrationFaultDescriptions.RESOURCE_UNKNOWN_FAULT_DESC);
		}
                if( this.logger.isLoggable(Level.FINE) )
                    this.logger.log(Level.FINE, this.displayCurrentSupportedTopicList());
	}


	/**
	 * 
	 * @param topicExpr
	 * @param subscriptionId
	 * @return
	 * @throws WSNotificationException
	 * @throws WSNotificationFault
	 */
	public void storeNewSubscription(final TopicExpressionType topicExprToCheck, final String subscriptionId) throws WSNotificationException, WSNotificationFault{

		boolean resourceStored = false;

		final String faultMsg = WsnFaultMessageConstants.WsnbSubcribeFaultDescriptions.TOPIC_EXPRESSION_DIALECT_UNKNOWN_FAULT_DESC;

		if (!this.isValideDialect(topicExprToCheck.getDialect())) {
			throw new TopicExpressionDialectUnknownFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,faultMsg);
		}

		// ----- XPATH Analyser Part ------
		final String topicExprStringForm = topicExprToCheck.getContent();
		final NamespaceMapperImpl context = new NamespaceMapperImpl();

		final String formattedTopicExpression = "";
		for (final QName ns : topicExprToCheck.getTopicNameSpace()) {
			context.addNamespace(ns.getLocalPart(), ns.getNamespaceURI());
		}
		/*	if (formattedTopicExpression.length()>0)
				formattedTopicExpression += " or ";
			formattedTopicExpression += topicExprStringForm +"[@wstop:topic='true']";
		}*/
		context.addNamespace(WstopConstants.WS_TOPICS_PREFIX,WstopConstants.WS_TOPICS_NAMESPACE_URI);
		context.addNamespace(WsnExtensionConstants.EBM_RESOURCEIDS_PREFIX,WsnExtensionConstants.EBM_RESOURCEIDS_NAMESPACE_URI);

		final List<?> res = this.xpathAnalyser.evaluate(/*formattedTopicExpression*/topicExprStringForm , this.supportedTopics.getRootElement(), context);

		if (res != null && res.size() > 0) {
			org.jdom.Element currentResId = null;
			for (final Object topicNodeItem : res) {
				currentResId = ((org.jdom.Element)topicNodeItem).getChild(WsnExtensionConstants.RESOURCE_IDS_QNAME.getLocalPart(),
						Namespace.getNamespace(WsnExtensionConstants.RESOURCE_IDS_QNAME.getNamespaceURI()));
				if(currentResId == null) {
					currentResId = new org.jdom.Element(WsnExtensionConstants.RESOURCE_IDS_QNAME.getLocalPart(),
							Namespace.getNamespace(WsnExtensionConstants.RESOURCE_IDS_QNAME.getNamespaceURI()));
					((org.jdom.Element)topicNodeItem).addContent(currentResId);
				}
				currentResId.getChildren().add(new org.jdom.Element(WsnExtensionConstants.SUBSCRIPTION_ID_QNAME.getLocalPart(),
						WsnExtensionConstants.SUBSCRIPTION_ID_QNAME.getPrefix(),
						WsnExtensionConstants.SUBSCRIPTION_ID_QNAME.getNamespaceURI()).setText(subscriptionId));
				resourceStored = true;
			}
		}

		if (!resourceStored) {
			throw new TopicNotSupportedFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,
					WsnFaultMessageConstants.WsnbSubcribeFaultDescriptions.TOPIC_NOT_SUPPORTED_FAULT_DESC);
		}
                if( this.logger.isLoggable(Level.FINE) )
                    this.logger.log(Level.FINE, this.displayCurrentSupportedTopicList());
	}

	/**
	 * 
	 * 
	 * @param subscriptionId
	 * @return
	 * @throws WSNotificationException
	 * @throws WSNotificationFault
	 */
	public void removeExistingSubscription(final String subscriptionId) throws WSNotificationException, WSNotificationFault{

		boolean resourceRemoved = false;
		// ----- XPATH Analyser Part ------
		final String resourceIdsXpatExpr = "*//" + WsnExtensionConstants.RESOURCE_IDS_QNAME.getPrefix() + ":" +WsnExtensionConstants.RESOURCE_IDS_QNAME.getLocalPart();
		final NamespaceMapperImpl context = new NamespaceMapperImpl();
		context.addNamespace(WsnExtensionConstants.EBM_RESOURCEIDS_PREFIX,WsnExtensionConstants.EBM_RESOURCEIDS_NAMESPACE_URI);

		final List<?> res = this.xpathAnalyser.evaluate(resourceIdsXpatExpr , this.supportedTopics.getRootElement(), context);

		if (res != null && res.size() > 0) {
			//final org.jdom.Element nodeToDelete = null;
			
			final String findIdXpathExpr = WsnExtensionConstants.SUBSCRIPTION_ID_QNAME.getPrefix() +
				":" + WsnExtensionConstants.SUBSCRIPTION_ID_QNAME.getLocalPart() +"[child::text()=\"" + subscriptionId+ "\"]";
			
			List<?> subRes = null;
			for (final Object nodeItem : res) {
				subRes = this.xpathAnalyser.evaluate(findIdXpathExpr , (org.jdom.Element)nodeItem, context);
				if (subRes != null && subRes.size() > 0) {
					((org.jdom.Element)nodeItem).removeContent((Content)subRes.get(0));
					resourceRemoved = true;
				}
			}
		}
		if (!resourceRemoved) {
			throw new ResourceUnknownFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,
					WsnFaultMessageConstants.WsnbUnsubcribeFaultDescriptions.RESOURCE_UNKNOWN_FAULT_DESC);
		}
                if( this.logger.isLoggable(Level.FINE) )
                    this.logger.log(Level.FINE, this.displayCurrentSupportedTopicList());
	}

	
	/**
	 * Store uuid corresponding to a notification message content on a specific topic
	 * Update existing previous uuid (only one notification message content uuid allowed per topic)
	 * 
	 * @param topic
	 * @param notifContentUuiId
	 * @throws WSNotificationException
	 * @throws WSNotificationFault
	 */
	public void storeNotifContentUuid(final TopicExpressionType topic, final String currentMessageContentUuid) throws WSNotificationException, WSNotificationFault{
		
		boolean resourceStored = false;
			
		if ((topic.getDialect().compareTo(WstopConstants.SIMPLE_TOPIC_EXPRESSION_DIALECT_URI) != 0) &&
				(topic.getDialect().compareTo(WstopConstants.CONCRETE_TOPIC_EXPRESSION_DIALECT_URI) != 0) &&
				(!this.isValideDialect(topic.getDialect()))){
			throw new TopicExpressionDialectUnknownFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,
					WsnFaultMessageConstants.WsnbGetCurrentMessageFaultDescriptions.TOPIC_EXPRESSION_DIALECT_UNKNOWN_FAULT_DESC);
		}

		// ----- XPATH Analyser Part ------
		final String topicExprStringForm = topic.getContent();
		final NamespaceMapperImpl context = new NamespaceMapperImpl();

		final String formattedTopicExpression = "";
		for (final QName ns : topic.getTopicNameSpace()) {
			context.addNamespace(ns.getLocalPart(), ns.getNamespaceURI());
		}
		
		context.addNamespace(WstopConstants.WS_TOPICS_PREFIX,WstopConstants.WS_TOPICS_NAMESPACE_URI);
		context.addNamespace(WsnExtensionConstants.EBM_RESOURCEIDS_PREFIX,WsnExtensionConstants.EBM_RESOURCEIDS_NAMESPACE_URI);

		final List<?> res = this.xpathAnalyser.evaluate(topicExprStringForm , this.supportedTopics.getRootElement(), context);

		if (res != null && res.size() == 1) {
			
			org.jdom.Element currentResIdNode,currentMsgIdNode = null;
			
			Object topicNode = res.get(0);
			
			currentResIdNode = ((org.jdom.Element)topicNode).getChild(WsnExtensionConstants.RESOURCE_IDS_QNAME.getLocalPart(),
					Namespace.getNamespace(WsnExtensionConstants.RESOURCE_IDS_QNAME.getNamespaceURI()));
			
			if(currentResIdNode == null) {
				
				currentResIdNode = new org.jdom.Element(WsnExtensionConstants.RESOURCE_IDS_QNAME.getLocalPart(),
						Namespace.getNamespace(WsnExtensionConstants.RESOURCE_IDS_QNAME.getNamespaceURI()));
				
				((org.jdom.Element)topicNode).addContent(currentResIdNode);
			}	
			
			currentMsgIdNode = currentResIdNode.getChild(WsnExtensionConstants.CURRENT_MESSAGE_ID_QNAME.getLocalPart(),
					Namespace.getNamespace(WsnExtensionConstants.CURRENT_MESSAGE_ID_QNAME.getNamespaceURI()));
			
			if(currentMsgIdNode == null) {
				
				currentMsgIdNode = new org.jdom.Element(WsnExtensionConstants.CURRENT_MESSAGE_ID_QNAME.getLocalPart(),
						Namespace.getNamespace(WsnExtensionConstants.CURRENT_MESSAGE_ID_QNAME.getNamespaceURI()));
				
				((org.jdom.Element)currentResIdNode).addContent(currentMsgIdNode);
			}	
			
			currentMsgIdNode.setText(currentMessageContentUuid);
			resourceStored = true;						
		}

		if (!resourceStored) {
			throw new TopicNotSupportedFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,
					WsnFaultMessageConstants.WsnbGetCurrentMessageFaultDescriptions.TOPIC_NOT_SUPPORTED_FAULT_DESC);
		}
                if( this.logger.isLoggable(Level.FINE) )
                    this.logger.log(Level.FINE, this.displayCurrentSupportedTopicList());
	}

	/**
	 * Return the Uuid associated to the last Notification message content sent on a given Topic 
	 *
	 * @param topic 
	 * @throws WSNotificationException
	 * @throws WSNotificationFault
	 */
	public String getNotifContentUuid(final TopicExpressionType topic, Boolean isSetRequest) throws WSNotificationException, WSNotificationFault{
		
		String result = null;
		
                if( this.logger.isLoggable(Level.FINE) )
                    this.logger.log(Level.FINE, this.displayCurrentSupportedTopicList());
		
		if ((topic.getDialect().compareTo(WstopConstants.SIMPLE_TOPIC_EXPRESSION_DIALECT_URI) != 0) &&
				(topic.getDialect().compareTo(WstopConstants.CONCRETE_TOPIC_EXPRESSION_DIALECT_URI) != 0) &&
				(!this.isValideDialect(topic.getDialect()))){
			throw new TopicExpressionDialectUnknownFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,
					WsnFaultMessageConstants.WsnbGetCurrentMessageFaultDescriptions.TOPIC_EXPRESSION_DIALECT_UNKNOWN_FAULT_DESC);
		}

		// ----- XPATH Analyser Part ------
		final String topicExprStringForm = topic.getContent();
		final NamespaceMapperImpl context = new NamespaceMapperImpl();

		String formattedTopicExpression = "";
		for (final QName ns : topic.getTopicNameSpace()) {
			context.addNamespace(ns.getLocalPart(), ns.getNamespaceURI());
		}
		/*if (formattedTopicExpression.length()>0)
				formattedTopicExpression += " or ";*/
		formattedTopicExpression = topicExprStringForm;
		final String topicTag = "[@wstop:topic='true']";

		if (!formattedTopicExpression.contains(topicTag)) {
			formattedTopicExpression = formattedTopicExpression + topicTag;
		}

		context.addNamespace(WstopConstants.WS_TOPICS_PREFIX,WstopConstants.WS_TOPICS_NAMESPACE_URI);
		context.addNamespace(WsnExtensionConstants.EBM_RESOURCEIDS_PREFIX,WsnExtensionConstants.EBM_RESOURCEIDS_NAMESPACE_URI);

		final List<?> res = this.xpathAnalyser.evaluate(formattedTopicExpression , this.supportedTopics.getRootElement(), context);
		if (res != null && res.size() == 1){

			//for (final Object item : res) {
			Object topicNode = res.get(0);

			final String findIdXpathExpr = WsnExtensionConstants.CURRENT_MESSAGE_ID_QNAME.getPrefix() + ":" + WsnExtensionConstants.CURRENT_MESSAGE_ID_QNAME.getLocalPart();

			final org.jdom.Element rIdsOffoundTopic = ((org.jdom.Element)topicNode).getChild(WsnExtensionConstants.RESOURCE_IDS_QNAME.getLocalPart(),
					Namespace.getNamespace(WsnExtensionConstants.RESOURCE_IDS_QNAME.getNamespaceURI()));

			List<?> currentMesgUuidRes = null;

			currentMesgUuidRes = this.xpathAnalyser.evaluate(findIdXpathExpr , rIdsOffoundTopic, context);
		
			if ((currentMesgUuidRes != null) && (currentMesgUuidRes.size() == 1)) {
				Object currentMessageUuidNode = currentMesgUuidRes.get(0);
				result = ((org.jdom.Element)currentMessageUuidNode).getText();					
			} else if (!isSetRequest)
				throw new TopicExpressionDialectUnknownFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,
						WsnFaultMessageConstants.WsnbGetCurrentMessageFaultDescriptions.NO_CURRENT_MESSAGE_ON_TOPIC_FAULT_DESC);
								
		} else if (res != null && res.size() > 0 ) {
			throw new TopicExpressionDialectUnknownFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,
					WsnFaultMessageConstants.WsnbGetCurrentMessageFaultDescriptions.MULTIPLE_TOPICS_SPECIFIED_FAULT_DESC);
		} else 
			throw new TopicExpressionDialectUnknownFault(WsnFaultMessageConstants.FAULT_DESCRIPTION_LANGUAGE,
					WsnFaultMessageConstants.WsnbGetCurrentMessageFaultDescriptions.NO_CURRENT_MESSAGE_ON_TOPIC_FAULT_DESC);				

		return result;
	}
	
	
	/**
	 * 
	 * @param topic
	 * @param isNotification
	 * @return
	 * @throws WSNotificationException
	 */
	public List<String> getSubscriptionIdsFromTopicsSet(final TopicExpressionType topic, final boolean isNotification) throws WSNotificationException{
		List<String> result = new ArrayList<String>();
                if( this.logger.isLoggable(Level.FINE) )
                    this.logger.log(Level.FINE, this.displayCurrentSupportedTopicList());
		if (isNotification && topic.getDialect().compareTo(WstopConstants.CONCRETE_TOPIC_EXPRESSION_DIALECT_URI) != 0 ||
				!isNotification && !this.isValideDialect(topic.getDialect())) {
			return null;
		}

		// ----- XPATH Analyser Part ------
		final String topicExprStringForm = topic.getContent();
		final NamespaceMapperImpl context = new NamespaceMapperImpl();

		String formattedTopicExpression = "";
		for (final QName ns : topic.getTopicNameSpace()) {
			context.addNamespace(ns.getLocalPart(), ns.getNamespaceURI());
		}
		/*if (formattedTopicExpression.length()>0)
				formattedTopicExpression += " or ";*/
		formattedTopicExpression = topicExprStringForm;
		final String topicTag = "[@wstop:topic='true']";

		if (!formattedTopicExpression.contains(topicTag)) {
			formattedTopicExpression = formattedTopicExpression + topicTag;
		}

		context.addNamespace(WstopConstants.WS_TOPICS_PREFIX,WstopConstants.WS_TOPICS_NAMESPACE_URI);
		context.addNamespace(WsnExtensionConstants.EBM_RESOURCEIDS_PREFIX,WsnExtensionConstants.EBM_RESOURCEIDS_NAMESPACE_URI);

		final List<?> res = this.xpathAnalyser.evaluate(formattedTopicExpression , this.supportedTopics.getRootElement(), context);
		if (res != null){
			for (final Object item : res) {
				final String findIdXpathExpr = WsnExtensionConstants.SUBSCRIPTION_ID_QNAME.getPrefix() + ":" + WsnExtensionConstants.SUBSCRIPTION_ID_QNAME.getLocalPart();//"ebm:SubscriptionId";
				final org.jdom.Element rIdsOffoundTopic = ((org.jdom.Element)item).getChild(WsnExtensionConstants.RESOURCE_IDS_QNAME.getLocalPart(),
						Namespace.getNamespace(WsnExtensionConstants.RESOURCE_IDS_QNAME.getNamespaceURI()));
				List<?> subRes = null;
				subRes = this.xpathAnalyser.evaluate(findIdXpathExpr , rIdsOffoundTopic, context);
				if (subRes != null && subRes.size() > 0) {
					for (final Object susbcriptionNodeItem : subRes) {
						final String uuidToStore = ((org.jdom.Element)susbcriptionNodeItem).getText();
						if (!result.contains(uuidToStore)) {
							result.add(uuidToStore);
						}
					}
				}
			}
		} else {
			result = null;
		}
		return result;
	}

	public List<String> getRegistrationIdsFromTopicsSet(final TopicExpressionType topic) throws WSNotificationException{
		List<String> result = new ArrayList<String>();
                if( this.logger.isLoggable(Level.FINE) )
                    this.logger.log(Level.FINE, this.displayCurrentSupportedTopicList());
		if (!this.isValideDialect(topic.getDialect())) {
			return null;
		}

		// ----- XPATH Analyser Part ------
		final String topicExprStringForm = topic.getContent();
		final NamespaceMapperImpl context = new NamespaceMapperImpl();

		String formattedTopicExpression = "";
		for (final QName ns : topic.getTopicNameSpace()) {
			context.addNamespace(ns.getLocalPart(), ns.getNamespaceURI());
		}
		/*if (formattedTopicExpression.length()>0)
				formattedTopicExpression += " or ";*/
		formattedTopicExpression = topicExprStringForm +"[@wstop:topic='true']";

		context.addNamespace(WstopConstants.WS_TOPICS_PREFIX,WstopConstants.WS_TOPICS_NAMESPACE_URI);
		context.addNamespace(WsnExtensionConstants.EBM_RESOURCEIDS_PREFIX,WsnExtensionConstants.EBM_RESOURCEIDS_NAMESPACE_URI);

		final List<?> res = this.xpathAnalyser.evaluate(formattedTopicExpression , this.supportedTopics.getRootElement(), context);
		if (res != null){
			for (final Object item : res) {
				final String findIdXpathExpr = WsnExtensionConstants.REGISTRATION_ID_QNAME.getPrefix() + ":" + WsnExtensionConstants.REGISTRATION_ID_QNAME.getLocalPart();//"ebm:RegistrationId";
				final org.jdom.Element rIdsOffoundTopic = ((org.jdom.Element)item).getChild(WsnExtensionConstants.RESOURCE_IDS_QNAME.getLocalPart(),
						Namespace.getNamespace(WsnExtensionConstants.RESOURCE_IDS_QNAME.getNamespaceURI()));
				//org.jdom.Element rIdsOffoundTopic = ((org.jdom.Element)res.get(0)).getChild(WstopConstants.RESOURCE_IDS_QNAME.getLocalPart(),
				//		Namespace.getNamespace(WstopConstants.RESOURCE_IDS_QNAME.getNamespaceURI()));
				List<?> subRes = null;
				subRes = this.xpathAnalyser.evaluate(findIdXpathExpr , rIdsOffoundTopic, context);
				if (subRes != null && subRes.size() > 0) {
					for (final Object susbcriptionNodeItem : subRes) {
						final String uuidToStore = ((org.jdom.Element)susbcriptionNodeItem).getText();
						if (!result.contains(uuidToStore)) {
							result.add(uuidToStore);
						}
					}
				}
			}
		} else {
			result = null;
		}
		return result;
	}

	public List<TopicExpressionType> getTopicSetPath(String subscriptionId) throws WSNotificationException{
		List<TopicExpressionType> result = new ArrayList<TopicExpressionType>();
		
		final NamespaceMapperImpl context = new NamespaceMapperImpl();

		context.addNamespace(WstopConstants.WS_TOPICS_PREFIX,WstopConstants.WS_TOPICS_NAMESPACE_URI);
		context.addNamespace(WsnExtensionConstants.EBM_RESOURCEIDS_PREFIX,WsnExtensionConstants.EBM_RESOURCEIDS_NAMESPACE_URI);

		String xpathExpression = "//*[@wstop:topic='true' and ebm:ResourceIds/ebm:SubscriptionId=\"" + subscriptionId +"\"]";
		
		List<?> nodeRes =  this.xpathAnalyser.evaluate(xpathExpression , this.supportedTopics.getRootElement(), context);		
		
		TopicExpressionType currentTopExpr = null;
		String currentTopExprContent = "";
		org.jdom.Element currentElt = null;
		
		if (nodeRes != null){
			for (final Object item : nodeRes) {
				
				currentElt =((org.jdom.Element)item); 
				
				currentTopExpr = WSNotificationFactory.getInstance().createTopicExpressionType();
				currentTopExpr.setDialect(WstopConstants.CONCRETE_TOPIC_EXPRESSION_DIALECT_URI);
				
				currentTopExprContent=currentElt.getName();
				
								
				while ((currentElt.getParentElement() != null) && (!currentElt.getParentElement().isRootElement())){										
					currentTopExprContent = (currentElt.getParentElement()).getName() +"/" + currentTopExprContent;					
					
					if ((currentElt.getParentElement().getNamespacePrefix() != null) && (currentElt.getParentElement().getNamespacePrefix().length()>0)){
						currentTopExprContent = currentElt.getParentElement().getNamespacePrefix() + ":" + currentTopExprContent;
						currentTopExpr.addTopicNameSpace(currentElt.getParentElement().getNamespacePrefix(),currentElt.getParentElement().getNamespace(currentElt.getParentElement().getNamespacePrefix()).getURI());						
					}
					currentElt = currentElt.getParentElement();					
				}
				currentTopExprContent += "[@wstop:topic='true']";	
				currentTopExpr.setContent(currentTopExprContent);
				result.add(currentTopExpr);
			}
		}
		
		return result;
	}
	
	
	
	public TopicSetType createTopicSetFromTopicNamespace(final TopicNamespaceType topicns, final List<String> topics) throws WSNotificationException {
		TopicSetType res = null;

		TopicNamespaceType topicNS = null;

		org.w3c.dom.Document domDocument = WSNotificationWriter.getInstance().writeTopicNamespaceType(topicns);

		// convert dom to jdom
		final DOMBuilder builder = new DOMBuilder();
		final org.jdom.Document jdomDocument = builder.build(domDocument);

		this.addSupportedTopicAttr(jdomDocument.getRootElement().getChildren(), topics);

		// convert jdom to dom
		final DOMOutputter converter = new DOMOutputter();
		try {
			domDocument = converter.output(jdomDocument);
		} catch (final JDOMException e) {
			throw new WSNotificationException(e);
		}

		// convert dom to topicSet
		topicNS = WSNotificationReader.getInstance().readTopicNamespaceType(domDocument);

		res = this.createTopicSetFromSupportedTopicNamespace(topicNS);

		return res;
	}

	private void addSupportedTopicAttr(final List<org.jdom.Element> children, final List<String> topics) {
		for(final org.jdom.Element child: children) {
			if(child.getName().equals(WstopConstants.TOPIC_QNAME.getLocalPart())&&child.getNamespaceURI().equals(WstopConstants.TOPIC_QNAME.getNamespaceURI())) {
				if(topics.contains(child.getAttribute("name").getValue())) {
					child.setAttribute(WsnExtensionConstants.SUPPORTED_QNAME_ATTR.getLocalPart(),
							"true", Namespace.getNamespace(WsnExtensionConstants.SUPPORTED_QNAME_ATTR.getPrefix(),
									WsnExtensionConstants.SUPPORTED_QNAME_ATTR.getNamespaceURI()));
				}
				if(child.getChildren() != null && child.getChildren().size() > 0) {
					this.addSupportedTopicAttr(child.getChildren(), topics);
				}
			}
		}
	}
	
	public TopicSetType createTopicSetFromSupportedTopicNamespace(final TopicNamespaceType topicns) throws WSNotificationException {
		TopicSetType res = null;

		/*
		Namespace wstop = Namespace.getNamespace("wstop", "http://docs.oasis-open.org/wsn/t-1");
		Namespace tns = Namespace.getNamespace("tns", topicns.getTargetNamespace());
		Namespace xsi = Namespace.getNamespace("xsi", "http://www.w3.org/2001/XMLSchema-instance");


		org.jdom.Element root = new org.jdom.Element("TopicSet", wstop);
		root.addNamespaceDeclaration(tns);
		root.addNamespaceDeclaration(xsi);
		root.setAttribute("schemaLocation", "http://docs.oasis-open.org/wsn/t-1 http://docs.oasis-open.org/wsn/t-1.xsd", xsi);
		Document doc = new Document(root);
		 */
		final Namespace tns = Namespace.getNamespace("tns", topicns.getTargetNamespace());
		final org.jdom.Element root = this.createEmptyTopicSet();
		final Document doc = new Document(root);
		this.createTopicSetTree(topicns.getTopics(), root, tns, true);

		// convert jdom to dom
		org.w3c.dom.Document domDocument = null;
		final DOMOutputter converter = new DOMOutputter();
		try {
			domDocument = converter.output(doc);
		} catch (final JDOMException e) {
			throw new WSNotificationException(e);
		}

		// convert dom to topicSet
		res = WSNotificationReader.getInstance().readTopicSetType(domDocument);

		return res;
	}

	private org.jdom.Element createEmptyTopicSet() {
		final Namespace wstop = Namespace.getNamespace(WstConstants.PREFIX, WstConstants.NAMESPACE_URI);
		final Namespace xsi = Namespace.getNamespace(WstConstants.XML_SCHEMA_PREFIX, WstConstants.XML_SCHEMA_NAMESPACE);

		final org.jdom.Element root = new org.jdom.Element(WstConstants.TOPIC_SET_QNAME.getLocalPart(), wstop);
		root.addNamespaceDeclaration(xsi);
		root.setAttribute("schemaLocation", "http://docs.oasis-open.org/wsn/t-1 http://docs.oasis-open.org/wsn/t-1.xsd", xsi);
		return root;
	}

	private void createTopicSetTree(final List<TopicType> topics,
			final org.jdom.Element root, final Namespace tns, final boolean first) {

		final Namespace wstop = Namespace.getNamespace("wstop", "http://docs.oasis-open.org/wsn/t-1");

		for(final TopicType topic: topics) {
			org.jdom.Element childTopic = null;
			if(first) {
				childTopic = new org.jdom.Element(topic.getName(), tns);
			} else {
				childTopic = new org.jdom.Element(topic.getName());
			}
			if(WsnSpecificTypeHelper.isTopicSupported(topic) != null && WsnSpecificTypeHelper.isTopicSupported(topic) == true) {
				childTopic.setAttribute("topic", "true", wstop);
			}
			if(topic.getChildren() != null && topic.getChildren().size() > 0) {
				this.createTopicSetTree(topic.getChildren(), childTopic, tns, false);
			}
			root.addContent(childTopic);
		}

	}

}
