/**
 * easySchema - easyWSDL toolbox Platform.
 * Copyright (c) 2008,  eBM Websourcing
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.ebmwebsourcing.wsstar.qml.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import javax.xml.transform.Source;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.sax.SAXSource;

import org.ow2.easywsdl.schema.api.SchemaException;
import org.ow2.easywsdl.schema.api.XmlException;
import org.ow2.easywsdl.schema.util.SourceHelper;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;

import com.ebmwebsourcing.wsstar.qml.api.Constraint;
import com.ebmwebsourcing.wsstar.qml.api.EquivalentTo;
import com.ebmwebsourcing.wsstar.qml.api.QOSDimension;
import com.ebmwebsourcing.wsstar.qml.api.WSQMLException;
import com.ebmwebsourcing.wsstar.qml.api.WSQMLReader;

/**
 * @author Nicolas Salatge - eBM WebSourcing
 * @author Le Duc Bao - France Telecom
 */
public class WSQMLReaderImpl implements WSQMLReader {

	private static Logger log = Logger.getLogger(WSQMLReaderImpl.class.getName());

	private Map<FeatureConstants, Object> features = new HashMap<FeatureConstants, Object>();

	/**
	 * The JAXB unique unmarshaller.
	 */
	private static Unmarshaller unmarshaller = null;

	/**
	 * The exception raised during JAXB unique unmarshaller creation.
	 */
	private static SchemaException unmarshallerCreationEx = null;

	/*
	 * Private object initializations
	 */
	static {
		try {

			WSQMLJAXBContext.createContext();

			WSQMLReaderImpl.unmarshaller = WSQMLJAXBContext.getJaxbContext().createUnmarshaller();
			// unmarshaller.setSchema(parent);
		} catch (final JAXBException e) {
			final WSQMLException jaxbContextCreationEx = new WSQMLException("Failed to create jaxb context", e);
			WSQMLReaderImpl.unmarshallerCreationEx = new SchemaException("Failed to create the JAXB unmarshaller", jaxbContextCreationEx);

		} catch (final SchemaException e) {
			final SchemaException jaxbContextCreationEx = new SchemaException("Failed to create jaxb context", e);
			WSQMLReaderImpl.unmarshallerCreationEx = new SchemaException("Failed to create the JAXB unmarshaller", jaxbContextCreationEx);

		}
	}

	public WSQMLReaderImpl() {
		this.features.put(FeatureConstants.VERBOSE, false);
		this.features.put(FeatureConstants.IMPORT_DOCUMENTS, true);
	}

	/**
	 * Get the unmarshaller instance.
	 * 
	 * @return the unmarshaller instance
	 * @throws SchemaException
	 *             The exception raised during the unmarshaller creation.
	 */
	public static Unmarshaller getUnMarshaller() throws SchemaException {

		if (WSQMLReaderImpl.unmarshallerCreationEx != null) {
			throw WSQMLReaderImpl.unmarshallerCreationEx;
		}

		return WSQMLReaderImpl.unmarshaller;
	}

	/**
	 * @return the jaxbContext
	 */
	public static JAXBContext getJaxbContext() {
		return WSQMLJAXBContext.getJaxbContext();
	}

	public QOSDimension readQOSDimension(final URI uri) throws WSQMLException {
		QOSDimension desc = null;
		try {
			InputStream input = null;
			final File f = new File(uri.getPath());
			if (f.exists()) {
				input = new FileInputStream(f);
			} else {
				input = uri.toURL().openStream();
			}
			desc = this.readQOSDimension(uri, new InputSource(input));
		} catch (final MalformedURLException e) {
			throw new WSQMLException("Can not get wsdl at: " + uri, e);
		} catch (final IOException e) {
			throw new WSQMLException("Can not get wsdl at: " + uri, e);
		} catch (final SchemaException e) {
			throw new WSQMLException("Can not get wsdl at: " + uri, e);
		} catch (final XmlException e) {
			throw new WSQMLException("Can not get wsdl at: " + uri, e);
		}
		return desc;
	}

	public QOSDimension readQOSDimension(final Document document) throws WSQMLException {
		QOSDimension desc = null;
		try {
			desc = this.readQOSDimension(new URI(document.getDocumentURI()), SourceHelper.convertDOMSource2InputSource(new DOMSource(document)));
		} catch (final URISyntaxException e) {
			throw new WSQMLException(e);
		} catch (final XmlException e) {
			throw new WSQMLException(e);
		}

		return desc;
	}

	public QOSDimension readQOSDimension(final URI uri, final InputSource input) throws WSQMLException {
		QOSDimension desc = null;
		try {

			final XMLReader xmlReader = XMLReaderFactory.createXMLReader();
			final SAXSource saxSource = new SAXSource(xmlReader, input);

			final JAXBElement<com.ebmwebsourcing.wsstar.qml.upmc.fr.QoSDimension> dimBinding = WSQMLReaderImpl.getUnMarshaller().unmarshal(saxSource, com.ebmwebsourcing.wsstar.qml.upmc.fr.QoSDimension.class);

			desc = new QOSDimensionImpl(uri, dimBinding.getValue(), this.getFeatures());
		} catch (final XmlException e) {
			throw new WSQMLException(e);
		} catch (final SAXException e) {
			throw new WSQMLException(e);
		} catch (final JAXBException e) {
			throw new WSQMLException(e);
		}

		return desc;
	}

	public void setFeature(final FeatureConstants name, final Object value) throws IllegalArgumentException {
		this.features.put(name, value);
		log.finest("set features: name = " + name + " - value = " + value);
	}

	public Object getFeature(final FeatureConstants name) throws IllegalArgumentException {
		return this.features.get(name);
	}

	public Map<FeatureConstants, Object> getFeatures() {
		return this.features;
	}

	public void setFeatures(final Map<FeatureConstants, Object> features) {
		this.features = features;
	}

	private static com.ebmwebsourcing.wsstar.qml.upmc.fr.ConstraintType convertStream2Constraint(final Source stream) throws SchemaException {

		try {
			// TODO : Check if it is a Thread safe method
			final JAXBElement<com.ebmwebsourcing.wsstar.qml.upmc.fr.ConstraintType> schemaBinding = WSQMLReaderImpl.getUnMarshaller().unmarshal(stream, com.ebmwebsourcing.wsstar.qml.upmc.fr.ConstraintType.class);

			return schemaBinding.getValue();

		} catch (final JAXBException e) {
			throw new SchemaException("Failed to build Java bindings from SchemaImpl descriptor XML document", e);
		}
	}

	public Constraint readConstraint(final URI uri) throws WSQMLException {
		Constraint desc = null;
		try {
			InputStream input = null;
			final File f = new File(uri.getPath());
			if (f.exists()) {
				input = new FileInputStream(f);
			} else {
				input = uri.toURL().openStream();
			}
			desc = this.readConstraint(uri, new InputSource(this.getInputStream(uri)));
		} catch (final MalformedURLException e) {
			throw new WSQMLException("Can not get wsdl at: " + uri, e);
		} catch (final IOException e) {
			throw new WSQMLException("Can not get wsdl at: " + uri, e);
		} catch (final SchemaException e) {
			throw new WSQMLException("Can not get wsdl at: " + uri, e);
		} catch (final XmlException e) {
			throw new WSQMLException("Can not get wsdl at: " + uri, e);
		}
		return desc;
	}

	private InputStream getInputStream(final URI uri) throws SchemaException {
		InputStream input = null;
		try {
			File f = null;
			if (uri.getPath() != null && uri.getScheme() == null) {
				f = new File(uri.getPath());
			}
			if (f != null) {
				input = new FileInputStream(f);
			} else {
				input = uri.toURL().openStream();
			}
		} catch (final MalformedURLException e) {
			throw new SchemaException("Can not get document at: " + uri, e);
		} catch (final IOException e) {
			throw new SchemaException("Can not get document at: " + uri, e);
		}
		return input;
	}

	public Constraint readConstraint(final Document document) throws WSQMLException {
		Constraint desc = null;
		try {
			desc = this.readConstraint(new URI(document.getDocumentURI()), SourceHelper.convertDOMSource2InputSource(new DOMSource(document)));
		} catch (final URISyntaxException e) {
			throw new WSQMLException(e);
		} catch (final XmlException e) {
			throw new WSQMLException(e);
		}

		return desc;
	}

	public Constraint readConstraint(final URI uri, final InputSource input) throws WSQMLException {
		Constraint desc = null;
		try {

			final XMLReader xmlReader = XMLReaderFactory.createXMLReader();
			final SAXSource saxSource = new SAXSource(xmlReader, input);

			final JAXBElement<com.ebmwebsourcing.wsstar.qml.upmc.fr.ConstraintType> constraintBinding = WSQMLReaderImpl.getUnMarshaller().unmarshal(saxSource, com.ebmwebsourcing.wsstar.qml.upmc.fr.ConstraintType.class);

			desc = new ConstraintImpl(uri, constraintBinding.getValue(), this.getFeatures());
		} catch (final XmlException e) {
			throw new WSQMLException(e);
		} catch (final SAXException e) {
			throw new WSQMLException(e);
		} catch (final JAXBException e) {
			throw new WSQMLException(e);
		}

		return desc;
	}

	public EquivalentTo readEquivalentTo(final URI uri) throws WSQMLException {
		EquivalentTo desc = null;
		try {
			InputStream input = null;
			final File f = new File(uri.getPath());
			if (f.exists()) {
				input = new FileInputStream(f);
			} else {
				input = uri.toURL().openStream();
			}
			desc = this.readEquivalentTo(uri, new InputSource(input));
		} catch (final MalformedURLException e) {
			throw new WSQMLException("Can not get wsdl at: " + uri, e);
		} catch (final IOException e) {
			throw new WSQMLException("Can not get wsdl at: " + uri, e);
		} catch (final SchemaException e) {
			throw new WSQMLException("Can not get wsdl at: " + uri, e);
		} catch (final XmlException e) {
			throw new WSQMLException("Can not get wsdl at: " + uri, e);
		}
		return desc;
	}

	public EquivalentTo readEquivalentTo(final URI uri, final InputSource input) throws WSQMLException {
		EquivalentTo desc = null;
		try {

			final XMLReader xmlReader = XMLReaderFactory.createXMLReader();
			final SAXSource saxSource = new SAXSource(xmlReader, input);

			final JAXBElement<com.ebmwebsourcing.wsstar.qml.upmc.fr.EquivalentTo> dimBinding = WSQMLReaderImpl.getUnMarshaller().unmarshal(saxSource, com.ebmwebsourcing.wsstar.qml.upmc.fr.EquivalentTo.class);

			desc = new EquivalentToImpl(uri, dimBinding.getValue(), this.getFeatures());
		} catch (final XmlException e) {
			throw new WSQMLException(e);
		} catch (final SAXException e) {
			throw new WSQMLException(e);
		} catch (final JAXBException e) {
			throw new WSQMLException(e);
		}

		return desc;
	}

}
