/**
 * Copyright (c) 2010 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * $id.java
 * -------------------------------------------------------------------------
 */
package com.ebmwebsourcing.wsstar.addressing.datatypes.api.refinedabstraction;

import java.net.URI;


import com.ebmwebsourcing.wsstar.addressing.datatypes.api.abstraction.EndpointReferenceType;
import com.ebmwebsourcing.wsstar.addressing.datatypes.api.abstraction.MetadataType;
import com.ebmwebsourcing.wsstar.addressing.datatypes.api.abstraction.ReferenceParametersType;
import com.ebmwebsourcing.wsstar.addressing.datatypes.api.abstraction.WsaFactory;
import com.ebmwebsourcing.wsstar.addressing.datatypes.api.abstraction.WsaReader;
import com.ebmwebsourcing.wsstar.addressing.datatypes.api.abstraction.WsaWriter;
import com.ebmwebsourcing.wsstar.addressing.datatypes.api.implementor.WsaModelFactory;
import com.ebmwebsourcing.wsstar.addressing.datatypes.api.utils.WsaException;

/**
 * Implementation, following the "Singleton" design pattern, of
 *  the {@link WsaFactory} factory interface.
 * 
 * NOTE : "RefinedWsaFactory" class is the "refined abstraction"
 * part of the "Bridge" design pattern.
 * 
 * @author Thierry Déjean - EBM Websourcing
 *
 */
public final class RefinedWsaFactory implements WsaFactory {
	
	private WsaModelFactory model;
		
	// ################# EXPERITMENTAL SINGLETON PATTERN FORM ######################
	
	private RefinedWsaFactory () {}	

	/**
	 * SingletonHolder is loaded on the first execution of Singleton.getInstance() 
	 * or the first access to SingletonHolder.INSTANCE, not before.
	 */
	private static final class WsaFactoryHolder { 		
		private static final RefinedWsaFactory INSTANCE = new RefinedWsaFactory();
		
		private WsaFactoryHolder(){};
		
	}

	/**
	 * Get the singleton instance of the class once the model's factory
	 * has been initialized.
	 * 
	 * @return the singleton instance of the class
	 * @throws WsaException
	 */
	public static WsaFactory getInstance() throws WsaException {
		RefinedWsaFactory singleton = WsaFactoryHolder.INSTANCE;
		if (singleton.model == null){ 			
			throw new WsaException("\n\t/!\\ WARNING /!\\\n" +
					"The WsaModelFactory have not been initialized !!!\n" +
					"Please create a \"WsaModelFactory\" instance and \n" +
					"call the \"getInstance(WsaModelFactory)\" method instead." +
					"\n\t/!\\ WARNING /!\\\n");
		}
		return singleton;		
	}	
	
	/**
	 * Initialize model's factory and get the singleton instance of the class
	 * 
	 * @param modelFactory then model's factory implementation.
	 * @return the singleton instance of the class
	 */
	public static WsaFactory getInstance(WsaModelFactory modelFactory) {		
		RefinedWsaFactory singleton = WsaFactoryHolder.INSTANCE;
		singleton.model = modelFactory;		 	
		return singleton;		
	}	 		
	// ##############################################################################

	/**
	 * Get the model's factory implementation used.
	 * In "Bridge" design pattern language it is the 
	 * "ConcreteImplementor" 
	 * 
	 * @return the model's factory implementation used
	 */
	public WsaModelFactory getModel() {
		return this.model;
	}
	
	/**
	 * Set the model's factory implementation to used.
	 * In "Bridge" design pattern language it is the 
	 * "ConcreteImplementor" 
	 * @param modelFactory an instance of model's factory implementation
	 */
	protected void setModel(WsaModelFactory modelFactory) {
		this.model = modelFactory;
	}
	
	/*
	 * (non-Javadoc)
	 * @see com.ebmwebsourcing.wsstar.addressing.api.abstraction.WsaFactory#createEndpointReferenceType(java.net.URI)
	 */
	@Override
	public EndpointReferenceType createEndpointReferenceType(URI address) {
		return this.model.createWsaModelEndpointReferenceType(address);
	}
		
	/*
	 * (non-Javadoc)
	 * @see com.ebmwebsourcing.wsstar.addressing.api.abstraction.WsaFactory#createReferenceParametersType()
	 */
	@Override
	public ReferenceParametersType createReferenceParametersType() {
		return this.model.createWsaModelReferenceParametersType();
	}

	/*
	 * (non-Javadoc)
	 * @see com.ebmwebsourcing.wsstar.addressing.api.abstraction.WsaFactory#createMetadataType()
	 */
	@Override
	public MetadataType createMetadataType() {
		return this.model.createWsaModelMetadataType();
	}

	/*
	 * (non-Javadoc)
	 * @see com.ebmwebsourcing.wsstar.addressing.api.abstraction.WsaFactory#getWsaReader()
	 */
	@Override
	public WsaReader getWsaReader() {
			
//		WsaReader result;		
//		try {
//			result = RefinedWsaReader.getInstance();
//		} catch (WsaException e) {
//			result = RefinedWsaReader.getInstance(this.model.getWsaModelReader());
//		}		
//		return result;
		return this.model.getWsaModelReader();
		
	}
	
	/*
	 * (non-Javadoc)
	 * @see com.ebmwebsourcing.wsstar.addressing.api.abstraction.WsaFactory#getWsaWriter()
	 */
	@Override
	public WsaWriter getWsaWriter() {
		
//		WsaWriter result;
		
//		try {
//			result = RefinedWsaWriter.getInstance();
//		} catch (WsaException e) {
//			result = RefinedWsaWriter.getInstance(this.model.getWsaModelWriter());
//		}		
//		return result;
		return this.model.getWsaModelWriter();
		
	}
	
}
