/*******************************************************************************
 * Copyright (c) 2011 EBM Websourcing.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v2.1
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 * 
 * Contributors:
 *     EBM Websourcing - initial API and implementation
 ******************************************************************************/
package com.ebmwebsourcing.easierbsm.sla.manager.impl.thread;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import java.util.logging.Logger;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.ParserConfigurationException;

import org.jdom.JDOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import com.ebmwebsourcing.easierbsm.contant.EasierBSMFramework;
import com.ebmwebsourcing.easierbsm.sla.manager.api.SLAManagerComponentBehaviour;
import com.ebmwebsourcing.easybox.api.XmlObjectFactory;
import com.ebmwebsourcing.easybox.api.XmlObjectReadException;
import com.ebmwebsourcing.easybox.api.XmlObjectWriteException;
import com.ebmwebsourcing.easycommons.research.util.SOAException;
import com.ebmwebsourcing.easycommons.research.util.dom.DOMUtil;
import com.ebmwebsourcing.easycommons.research.util.easybox.SOAUtil;
import com.ebmwebsourcing.easycommons.research.util.esb.ESBUtil;
import com.ebmwebsourcing.easycommons.research.util.esb.EndpointAddress;
import com.ebmwebsourcing.easycommons.research.util.jaxb.SOAJAXBContext;
import com.ebmwebsourcing.easycommons.soap.handler.SOAPException;
import com.ebmwebsourcing.easycommons.soap.handler.SOAPSender;
import com.ebmwebsourcing.easycommons.xml.DocumentBuilders;
import com.ebmwebsourcing.easycommons.xml.XMLPrettyPrinter;
import com.ebmwebsourcing.easyesb.exchange10.api.element.Exchange;
import com.ebmwebsourcing.easyesb.rawreport10.api.element.Report;
import com.ebmwebsourcing.easyesb.rawreport10.api.element.ReportList;
import com.ebmwebsourcing.easyesb.rawreport10.api.type.TimeStampType;
import com.ebmwebsourcing.easyesb.soa.api.ESBException;
import com.ebmwebsourcing.easyesb.soa.api.endpoint.ProviderEndpoint;
import com.ebmwebsourcing.easyesb.soa.api.registry.RegistryEndpointBehaviour;
import com.ebmwebsourcing.easyesb.soa.impl.endpoint.thread.NotificationProducerThreadImpl;
import com.ebmwebsourcing.easyesb.soa10.api.type.EndpointType;
import com.ebmwebsourcing.easyesb.transporter.api.transport.TransportException;
import com.ebmwebsourcing.escad10.api.element.AlertDefinition;
import com.ebmwebsourcing.escad10.api.element.Protocol;
import com.ebmwebsourcing.escad10.api.element.To;
import com.ebmwebsourcing.escad10.protocol.soapnotif.api.element.Binding;
import com.ebmwebsourcing.escapnote10.api.element.AlertNoteDefinition;
import com.ebmwebsourcing.escapnote10.api.element.InitiatorIdentifier;
import com.ebmwebsourcing.escapnote10.api.element.ResponderIdentifier;
import com.ebmwebsourcing.esqml10.api.type.ConstraintType;
import com.ebmwebsourcing.wsagreement10.api.element.GuaranteeTerm;
import com.ebmwebsourcing.wsagreement10.api.element.ServiceLevelObjective;
import com.ebmwebsourcing.wsagreement10.api.type.AgreementType;
import com.ebmwebsourcing.wscap12.api.anonymoustype.Alert;
import com.ebmwebsourcing.wscap12.api.anonymoustype.Info;
import com.ebmwebsourcing.wsstar.basenotification.datatypes.api.abstraction.Notify;
import com.ebmwebsourcing.wsstar.basenotification.datatypes.api.refinedabstraction.RefinedWsnbFactory;
import com.ebmwebsourcing.wsstar.basenotification.datatypes.api.utils.WsnbException;

import easybox.org.ggf.schemas.graap._2007._03.ws_agreement.EJaxbAgreementContextType;
import easybox.org.w3._2005._08.addressing.EJaxbEndpointReferenceType;

public class SLAViolationThread extends Thread implements Runnable {

    static {
        try {
            SOAJAXBContext.getInstance().addOtherObjectFactory(com.ebmwebsourcing.wsstar.jaxb.notification.base.ObjectFactory.class);
        } catch (SOAException e) {
            // do nothing
            e.printStackTrace();
        }
    }

    private final Logger log = Logger.getLogger(SLAViolationThread.class
            .getName());

    private DocumentBuilder db = DocumentBuilders.takeDocumentBuilder();

    private enum InegalityOperator {
        GREATER_THAN("&gt;"),
        LESS_THAN("&lt;"),
        GREATER_THAN_OR_EQUALS("&gt;="),
        LESS_THAN_OR_EQUALS("&lt;=");

        private String value = null;

        InegalityOperator(String val) {
            this.value = val;
        }

        boolean equals(String val) {
            if(value.equals(val)) {
                return true;
            }
            return false;
        }

    }

    private enum Level {
        WARNING("Warning"),
        SEVERE("Severe");

        private String value = null;

        Level(String val) {
            this.value = val;
        }

        public boolean equals(String val) {
            if(value.equals(val)) {
                return true;
            }
            return false;
        }

        public String toString() {
            return value;
        }
    }

    private ReportList reports;

    private AgreementType agreement;

    private final SOAPSender soapSender;

    private final SLAManagerComponentBehaviour behaviour;

    //private final long TIMEOUT = 2000;

    public SLAViolationThread(SLAManagerComponentBehaviour behaviour, ReportList reports, AgreementType agreement) {
        this.reports = reports;
        this.agreement = agreement;
        this.soapSender = new SOAPSender();
        this.behaviour = behaviour;
    }

    public void addReport(Report report){
        this.reports.addReport(report);
    }

    public ReportList getReports(){
        return this.reports;
    }

    @Override
    public void run() {
        try {

            List<Alert> alerts = detectViolations(this.reports, this.agreement);
            if(alerts != null && alerts.size() > 0) {
                log.info("ALERTS FOUND");
                // get alert definition
                AlertDefinition alertDef = this.agreement.getAgreementContext().findExtendedElement(AlertDefinition.class);
                if(alertDef == null) {
                    // FIXME: EASYBOX BUG
                    Object obj = ((EJaxbAgreementContextType)this.agreement.getAgreementContext().getModelObject()).getAny().iterator().next();
                    if(obj != null && obj instanceof Element) {
                        Element elmt = (Element)obj;
                        Document doc = DOMUtil.getInstance().getDocumentBuilderFactory().newDocumentBuilder().newDocument();
                        doc.appendChild(doc.importNode(elmt, true));
                        alertDef = SOAUtil.getInstance().getReader(EasierBSMFramework.getInstance()).get().readDocument(doc, AlertDefinition.class);
                    }
                }

                if(alertDef != null) {
                    // send alerts
                    try {
                        sendAlerts(alerts, alertDef);
                    } catch (ESBException e) {
                        log.severe(e.getMessage());
                    }
                } else {
                    log.severe("No alertDefinition defined in agreement: " + agreement.getName());
                }
            }
        } catch (ParserConfigurationException e) {
            e.printStackTrace();
            log.severe(e.getMessage());
        } catch (XmlObjectReadException e) {
            e.printStackTrace();
            log.severe(e.getMessage());
        } catch (ESBException e) {
            e.printStackTrace();
            log.severe(e.getMessage());
        }
    }

    private void sendAlerts(List<Alert> alerts, AlertDefinition alertDef) throws ESBException {
        for(To to: alertDef.getTos()) {
            for(Protocol protocol: to.getProtocols()) {
                Binding soapNotifBinding = protocol.findBinding(Binding.class);
                if(soapNotifBinding != null) {
                    sendAlerts(alerts, to, soapNotifBinding);
                } else {
                    log.severe("Only soap-notification binding is supported to send alert for the moment!!!");
                }

            }
        }
    }

    private void sendAlerts(List<Alert> alerts, To to, Binding soapNotifBinding) throws ESBException {
        for(Alert alert: alerts) {
            sendAlert(alert, to, soapNotifBinding);
        }
    }

    private void sendAlert(Alert alert, To to, Binding soapNotifBinding) throws ESBException {
        try {
            final EndpointType ep = ((RegistryEndpointBehaviour)this.behaviour.getEndpoint().getNode()
                    .getRegistryEndpoint().findBehaviour(RegistryEndpointBehaviour.class)).getEndpoint(
                            QName.valueOf(soapNotifBinding.getSoapAddress()));

            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            SOAUtil.getInstance().getWriter(EasierBSMFramework.getInstance()).get().writeDocument(alert, baos);
            ByteArrayInputStream writtenStream = new ByteArrayInputStream(baos.toByteArray());
            Document notifPayload = DOMUtil.getInstance().getDocumentBuilderFactory().newDocumentBuilder().parse(writtenStream);

            String producerAddress = ((ProviderEndpoint) this.behaviour.getEndpoint()).getQName().getNamespaceURI() + "::" + ((ProviderEndpoint) this.behaviour.getEndpoint()).getQName().getLocalPart();

            final Notify notify = NotificationProducerThreadImpl.createNotification(producerAddress.trim(), soapNotifBinding.getSoapAddress().trim(), UUID.randomUUID().toString(), null, "http://www.w3.org/TR/1999/REC-xpath-19991116", notifPayload);

            if (ep != null) {
                final Exchange ex = NotificationProducerThreadImpl.createMessageExchange(this.behaviour, soapNotifBinding.getSoapAddress(), notify);
                this.log.info("send internal notification to: "	+ ex.getDestination());

                this.behaviour.getEndpoint().getNode()
                .getTransportersManager().push(ex,
                        ep.getNode());
            } else {
                Document request = RefinedWsnbFactory.getInstance().getWsnbWriter().writeNotifyAsDOM(notify);
                //				final Document request = WSNotificationWriter
                //				.getInstance().writeNotify(notify);

                this.soapSender.sendSoapRequest(SOAPSender.createSOAPMessageRequest(request), soapNotifBinding.getSoapAddress(), null);
                this.log.info("Alert sended to " + soapNotifBinding.getSoapAddress());
            }
        } catch (TransportException e) {
            throw new ESBException(e);
        } catch (SOAPException e) {
            throw new ESBException(e);
        } catch (JDOMException e) {
            throw new ESBException(e);
        } catch (XmlObjectWriteException e) {
            throw new ESBException(e);
        } catch (SAXException e) {
            throw new ESBException(e);
        } catch (IOException e) {
            throw new ESBException(e);
        } catch (ParserConfigurationException e) {
            throw new ESBException(e);
        } catch (WsnbException e) {
            throw new ESBException(e);
        } 
    }

    private List<Alert> detectViolations(ReportList reports, AgreementType ag) throws ESBException {
        List<Alert> alerts  = new ArrayList<Alert>();

        Object initiator = ag.getAgreementContext().getAgreementInitiator();
        Object responder = ag.getAgreementContext().getAgreementResponder();

        for(GuaranteeTerm g: ag.getTerms().getAll().getGuaranteeTerms()) {
            Alert alertOnGuarantee = detectViolationsOnGuarantee(reports, g, initiator, responder);

            if(alertOnGuarantee != null ) {
                alerts.add(alertOnGuarantee);
            }
        }

        return alerts;
    }

    private Alert detectViolationsOnGuarantee(ReportList reports,
            GuaranteeTerm g, Object initiator, Object responder) throws ESBException {
        Alert alert  = null;

        ServiceLevelObjective slo = g.getServiceLevelObjective();

        if(slo.getKPITarget().getKPIName().equals("latency")) {
            alert = detectLatencyViolation(g, slo, reports, initiator, responder);
        } else {
            alert = createAlert("Unknow KPI Target: " + slo.getKPITarget().getKPIName(), g, reports, Level.WARNING, initiator, responder);
        }

        return alert;
    }

    private Alert detectLatencyViolation(GuaranteeTerm g,
            ServiceLevelObjective slo, ReportList reports, Object initiator, Object responder) throws ESBException {
        Alert alert  = null;
        long currentLatency = Long.MAX_VALUE;
        try {

            Object obj = slo.getKPITarget().getCustomServiceLevel();
            if(obj != null && obj instanceof Element) {
                Element elmt = (Element)obj;
                Document doc;

                doc = DOMUtil.getInstance().getDocumentBuilderFactory().newDocumentBuilder().newDocument();
                doc.appendChild(doc.importNode(elmt, true));

                ConstraintType target = SOAUtil.getInstance().getReader(EasierBSMFramework.getInstance()).get().readFragment(doc, ConstraintType.class);
                long expectedLatency = (long) (target.getValue().getValue() * 1000);

                String alertType = "";
                String certainty = "";

                /**
                 * Manage non-responding service by checking sla only with t1 and t2 reports... able to send Potential alerts.
                 */
                if(reports.getReports().length == 2){

                    long current = new Date().getTime();
                    long t2 = reports.getReports()[1].getDateInGMT().getTime();
                    //if received reports are T1/T2 reports
                    if(reports.getReports()[0].getTimeStamp().equals(TimeStampType.t1)){
                        try {
                            synchronized (this) {
                                wait(expectedLatency);
                            }

                        } catch (InterruptedException e) {
                            // TODO Auto-generated catch block
                            e.printStackTrace();
                        }

                        alertType = "(Potential)";
                        certainty = "Possible";
                    }

                    //In  order to be sure the potential SLA violation will be sent
                    currentLatency = expectedLatency + 1;

                }

                /**
                 * Manage all services... able to send Confirmed alerts.
                 */
                if(reports.getReports().length == 4){
                    if(reports.getReports()[0].getTimeStamp().equals(TimeStampType.t1) && 
                            reports.getReports()[1].getTimeStamp().equals(TimeStampType.t2) &&
                            reports.getReports()[2].getTimeStamp().equals(TimeStampType.t3) &&
                            reports.getReports()[3].getTimeStamp().equals(TimeStampType.t4)){

                        Report providerOut = reports.getReports()[2];
                        Report providerIn = reports.getReports()[1];

                        currentLatency = providerOut.getDateInGMT().getTime() - providerIn.getDateInGMT().getTime();

                        alertType = "(Confirmed)";
                        certainty = "Observed";
                    }
                }


                if(InegalityOperator.GREATER_THAN.equals(target.getOperator()) && currentLatency <= expectedLatency) {
                    alert = createAlert(slo.getKPITarget().getKPIName() + " violation: " + "current " + slo.getKPITarget().getKPIName() + "(" + currentLatency + ")" + " is not > to " + expectedLatency + " " + alertType, g, reports, Level.SEVERE, initiator, responder);
                } else if(InegalityOperator.GREATER_THAN_OR_EQUALS.equals(target.getOperator()) && currentLatency < expectedLatency) {
                    alert = createAlert(slo.getKPITarget().getKPIName() + " violation: " + "current " + slo.getKPITarget().getKPIName() + "(" + currentLatency + ")" + " is not >= to " + expectedLatency + " " + alertType, g, reports, Level.SEVERE, initiator, responder);
                } else if(InegalityOperator.LESS_THAN.equals(target.getOperator()) && currentLatency >= expectedLatency) {
                    alert = createAlert(slo.getKPITarget().getKPIName() + " violation: " + "current " + slo.getKPITarget().getKPIName() + "(" + currentLatency + ")" + " is not < to " + expectedLatency + " " + alertType, g, reports, Level.SEVERE, initiator, responder);
                } else if(InegalityOperator.LESS_THAN_OR_EQUALS.equals(target.getOperator()) && currentLatency > expectedLatency) {
                    alert = createAlert(slo.getKPITarget().getKPIName() + " violation: " + "current " + slo.getKPITarget().getKPIName() + "(" + currentLatency + ")" + " is not <= to " + expectedLatency + " " + alertType, g, reports, Level.SEVERE, initiator, responder);
                } else if(!InegalityOperator.GREATER_THAN.equals(target.getOperator())
                        && !InegalityOperator.LESS_THAN.equals(target.getOperator())
                        && !InegalityOperator.GREATER_THAN_OR_EQUALS.equals(target.getOperator())
                        && !InegalityOperator.LESS_THAN_OR_EQUALS.equals(target.getOperator())){
                    throw new ESBException("Unknown agreement operator: " + target.getOperator()); 
                }

                Info info = SOAUtil.getInstance().getXmlContext(EasierBSMFramework.getInstance()).getXmlObjectFactory().create(Info.class);
                info.addCategory("Infra");
                info.setEvent("SLA Violation");
                info.setUrgency("Unknown");
                info.setSeverity("Unknown");
                info.setCertainty(certainty);
                alert.addInfo(info);

            }else{
                this.log.warning("Wrong number of reports to check SLAs!");
            }
        } catch (ParserConfigurationException e) {
            throw new ESBException(e);
        } catch (XmlObjectReadException e) {
            throw new ESBException(e);
        } 
        return alert;
    }


    private Alert createAlert(String msg, GuaranteeTerm g,
            ReportList reports, Level level, Object initiator, Object responder) {

        Alert alert = SOAUtil.getInstance().getXmlContext(EasierBSMFramework.getInstance()).getXmlObjectFactory().create(Alert.class);

        alert.setIdentifier(reports.getReports()[0].getExchangeId());
        alert.setMsgType("Alert");
        alert.setSent(Calendar.getInstance().getTime());

        /* Set Alert note with internal capnote spec. */

        XmlObjectFactory factory = SOAUtil.getInstance().getXmlContext(EasierBSMFramework.getInstance()).getXmlObjectFactory();
        AlertNoteDefinition note = factory.create(AlertNoteDefinition.class);
        note.setMessage(msg);

        if(initiator instanceof EJaxbEndpointReferenceType){
            try{
                EJaxbEndpointReferenceType initiatorEPR = (EJaxbEndpointReferenceType)initiator; 
                EndpointAddress ea = ESBUtil.analyzeURI(new URI(initiatorEPR.getAddress().getValue()));

                InitiatorIdentifier initiatorEndpoint = SOAUtil.getInstance().getXmlContext(EasierBSMFramework.getInstance()).getXmlObjectFactory().create(InitiatorIdentifier.class);
                initiatorEndpoint.setEndpointName(ea.getEndpointname());
                initiatorEndpoint.setNamespaceURI(ea.getNamespace());
                if(ea.getServicename() != null){
                    initiatorEndpoint.setServiceName(ea.getServicename());
                }
                note.setInitiatorIdentifier(initiatorEndpoint);

            }catch(Exception e){
                log.severe("Failed to parse initiator URI ...");
            }
        }
        if(responder instanceof EJaxbEndpointReferenceType){
            try{
                EJaxbEndpointReferenceType responderEPR = (EJaxbEndpointReferenceType)responder; 
                EndpointAddress ea = ESBUtil.analyzeURI(new URI(responderEPR.getAddress().getValue()));

                ResponderIdentifier responderEndpoint = SOAUtil.getInstance().getXmlContext(EasierBSMFramework.getInstance()).getXmlObjectFactory().create(ResponderIdentifier.class);
                responderEndpoint.setEndpointName(ea.getEndpointname());
                responderEndpoint.setNamespaceURI(ea.getNamespace());
                responderEndpoint.setServiceName(ea.getServicename());

                note.setResponderIdentifier(responderEndpoint);

            }catch(Exception e){
                log.severe("Failed to parse responder URI ...");
            }
        }

        String noteAsString = "";//"<![CDATA[";
        Document noteAsDOM = db.newDocument();
        try {
            SOAUtil.getInstance().getWriter(EasierBSMFramework.getInstance()).get().writeDocument(note, noteAsDOM);
        } catch (XmlObjectWriteException e) {
            log.severe("Failed to write note as dom in alert.");
        }
        
        noteAsString += XMLPrettyPrinter.prettyPrint(noteAsDOM);
        //noteAsString += "]]>";
        alert.setNote(noteAsString);

        return alert;
    }

    public void launchNotify(){
        this.notify();
    }


    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        DocumentBuilders.releaseDocumentBuilder(db);
    }

}
