/*******************************************************************************
 * Copyright (c) 2011 EBM Websourcing.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v2.1
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 * 
 * Contributors:
 *     EBM Websourcing - initial API and implementation
 ******************************************************************************/
/**
 * EasyESB: EasyESB Services Platform Copyright (C) 2009 EBM WebSourcing
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 * 
 * Initial developer(s): EBM WebSourcing
 */
package com.petalslink.easiergov.core.impl.config;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.petalslink.easiergov.core.config.Configuration;
import com.petalslink.easiergov.resources.api.GovException;
import com.petalslink.easiersbs.matching.service.api.matcher.similarity.SimilarityType;



/**
 * @author chamerling - eBM WebSourcing
 * 
 */
public class ConfigurationImpl implements Configuration {

    private static Logger log = Logger.getLogger(ConfigurationImpl.class.getName());

    private Map<String, String> properties = new HashMap<String, String>();

    private static String CONFIG = "config.properties";



    public ConfigurationImpl() throws GovException {
        this.properties.put(NAME, DEFAULT_NAME);
        this.properties.put(NAMESPACE, DEFAULT_NAMESPACE);
        this.properties.put(HOST, DEFAULT_HOST);
        this.properties.put(PORT, DEFAULT_PORT);
        this.properties.put(NOTIFICATION_PORT, DEFAULT_NOTIFICATION_PORT);
        
        this.properties.put(EASIERSBS_SYNTACTIC_ACCEPTANCE_THRESHOLD, DEFAULT_EASIERSBS_SYNTACTIC_ACCEPTANCE_THRESHOLD);
        this.properties.put(EASIERSBS_SYNTACTIC_LEVENSHTEIN_LIMIT, DEFAULT_EASIERSBS_SYNTACTIC_LEVENSHTEIN_LIMIT);
        this.properties.put(EASIERSBS_SYNTACTIC_SIMILARITY_TYPE, DEFAULT_EASIERSBS_SYNTACTIC_SIMILARITY_TYPE);
        loadConfigurationFile();

    }

    public ConfigurationImpl(Map<String, String> data) throws GovException {
        this.properties = data;
    }

    public ConfigurationImpl(String host, int port, int notif_port) throws GovException {
        this.properties.put(HOST, String.valueOf(host));
        this.properties.put(PORT, String.valueOf(port));
        this.properties.put(NOTIFICATION_PORT, String.valueOf(notif_port));
        
        this.properties.put(NAME, DEFAULT_NAME);
        this.properties.put(NAMESPACE, DEFAULT_NAMESPACE);
        this.properties.put(EASIERSBS_SYNTACTIC_ACCEPTANCE_THRESHOLD, DEFAULT_EASIERSBS_SYNTACTIC_ACCEPTANCE_THRESHOLD);
        this.properties.put(EASIERSBS_SYNTACTIC_LEVENSHTEIN_LIMIT, DEFAULT_EASIERSBS_SYNTACTIC_LEVENSHTEIN_LIMIT);
        this.properties.put(EASIERSBS_SYNTACTIC_SIMILARITY_TYPE, DEFAULT_EASIERSBS_SYNTACTIC_SIMILARITY_TYPE);

    }

    private void loadConfigurationFile() throws GovException {
        Properties props = new Properties();
        File f = new File(CONFIG);
        if (log.isLoggable(Level.FINE)) {
            log.fine("Loading config from " + f.getAbsoluteFile());
        }
        InputStream is = null;
        if(f.exists()) {
            try {
                is = new FileInputStream(f);
            } catch (FileNotFoundException e) {
                throw new GovException(e);
            }
        }
        if (is != null) {
            try {
                props.load(is);
                for (Object k : props.keySet()) {
                    properties.put(k.toString().trim(), props.get(k).toString()
                            .trim());

                }
            } catch (IOException e) {
                throw new GovException(e);
            } 
        } else {
            log.warning("No configuration file found...");
        }
    }


    /**
     * @return the data
     */
    public Map<String, String> getProperties() {
        return properties;
    }

    public void addProperty(String key, String value) {
        this.properties.put(key, value);
    }

    public static String getCONFIG() {
        return CONFIG;
    }

    public static void setCONFIG(String config) {
        CONFIG = config;
    }

    public Integer getPort() {
        return Integer.valueOf(this.getProperties().get(PORT));
    }

    public String getHost() {
        return String.valueOf(this.getProperties().get(HOST));
    }

    public void setHost(String host) {
        this.properties.put(HOST, String.valueOf(host));
    }

    public void setPort(int port) {
        this.properties.put(PORT, String.valueOf(port));
    }

    @Override
    public Integer getNotificationPort() {
        return Integer.valueOf(this.getProperties().get(NOTIFICATION_PORT));
    }

    @Override
    public void setNotificationPort(int port) {
        this.properties.put(NOTIFICATION_PORT, String.valueOf(port));
    }

    @Override
    public String getName() {
        return String.valueOf(this.getProperties().get(NAME));
    }

    @Override
    public void setName(String name) {
        this.properties.put(NAME, String.valueOf(name));
    }

    @Override
    public String getNamespace() {
        return String.valueOf(this.getProperties().get(NAMESPACE));
    }

    @Override
    public void setNamespace(String namespace) {
        this.properties.put(NAMESPACE, String.valueOf(namespace));
    }

	@Override
	public Float getEasiersbsSyntacticAcceptanceThreshold() {
		return Float.valueOf(this.getProperties().get(EASIERSBS_SYNTACTIC_ACCEPTANCE_THRESHOLD));
	}

	@Override
	public void setEasiersbsSyntacticAcceptanceThreshold(Float f) {
		this.properties.put(EASIERSBS_SYNTACTIC_ACCEPTANCE_THRESHOLD, String.valueOf(f));
	}

	@Override
	public Integer getEasiersbsSyntacticLevenshteinLimit() {
		return Integer.valueOf(this.getProperties().get(EASIERSBS_SYNTACTIC_LEVENSHTEIN_LIMIT));
	}

	@Override
	public void setEasiersbsSyntacticLevenshteinLimit(Integer f) {
		this.properties.put(EASIERSBS_SYNTACTIC_LEVENSHTEIN_LIMIT, String.valueOf(f));
	}

	@Override
	public SimilarityType getEasiersbsSyntacticSimilarityType() {
		String similarity = this.getProperties().get(EASIERSBS_SYNTACTIC_SIMILARITY_TYPE);
		if(similarity != null) {
			if(similarity.toLowerCase().equals("jaccard")) {
				return SimilarityType.JACCARD;
			} else if(similarity.toLowerCase().equals("dice")) {
				return SimilarityType.DICE;
			} else if(similarity.toLowerCase().equals("cosine")) {
				return SimilarityType.COSINE;
			} else if(similarity.toLowerCase().equals("jensen-shannon")) {
				return SimilarityType.JENSEN_SHANNON;
			}
		}
		return null;
	}

	@Override
	public void setEasiersbsSyntacticSimilarityType(SimilarityType f) {
		if(f != null) {
			if(f == SimilarityType.JACCARD) {
				this.properties.put(EASIERSBS_SYNTACTIC_SIMILARITY_TYPE, "jaccard");
			} else if(f == SimilarityType.DICE) {
				this.properties.put(EASIERSBS_SYNTACTIC_SIMILARITY_TYPE, "dice");
			} else if(f == SimilarityType.COSINE) {
				this.properties.put(EASIERSBS_SYNTACTIC_SIMILARITY_TYPE, "cosine");
			} else if(f == SimilarityType.JENSEN_SHANNON) {
				this.properties.put(EASIERSBS_SYNTACTIC_SIMILARITY_TYPE, "jensen-shannon");
			}
		}
	}




}
