/*******************************************************************************
 * Copyright (c) 2011 EBM Websourcing.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser Public License v2.1
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 * 
 * Contributors:
 *     EBM Websourcing - initial API and implementation
 ******************************************************************************/
/**
 * PETALS - PETALS Services Platform. Copyright (c) 2008 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */

package com.ebmwebsourcing.easiergov.launcher;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.logging.LogManager;

import com.ebmwebsourcing.easiergov.launcher.tasks.InfoTask;
import com.ebmwebsourcing.easiergov.launcher.util.CommandReader;
import com.ebmwebsourcing.easiergov.launcher.util.Locker;
import com.ebmwebsourcing.easiergov.launcher.util.SystemExitHook;
import com.petalslink.easiergov.core.container.Container;
import com.petalslink.easiergov.resources.api.GovException;

/**
 * 
 * Created on 29 janv. 08
 * 
 * @author Christophe HAMERLING, Roland Naudin - eBM WebSourcing
 * @since 1.0
 * 
 */
public abstract class AbstractContainerLauncher implements Launcher {

	private static final String START_COMMAND = "start";

	private static final String SHUTDOWN_COMMAND = "shutdown";

	private static final String VERSION_COMMAND = "version";


	private final static String LOGGER_FILE = "logger.properties";

	static {
		try {
			InputStream inputStream = null;
			File logger = new File(LOGGER_FILE);
			if (logger.exists()) {
				inputStream = new FileInputStream(logger);
			} else {
				// ty to get from classpath
				inputStream = AbstractContainerLauncher.class.getClass()
				.getResourceAsStream("/" + LOGGER_FILE);
			}
			if (inputStream != null) {
				LogManager.getLogManager().readConfiguration(inputStream);
			}
		} catch (final Exception e) {
			throw new RuntimeException("couldn't initialize logging properly",
					e);
		}
	}

	/**
	 * The container server instance
	 */
	private Container container;

	/**
	 * System exit hook used if the command line is used
	 */
	private SystemExitHook systemExitHook;

	/**
	 * The PEtALS locker indicating that PEtALS is running
	 */
	private final Locker locker;

	/**
	 * Default constructor
	 */
	public AbstractContainerLauncher(Container c) {
		this.locker = new Locker(new File("."));
		this.showBanner();

		this.container = c;
	}

	/**
	 * 
	 */
	protected void showBanner() {
		System.out.println();
		System.out
		.println(" -----------------------------------------------------------");
		System.out
		.println("|                                                           |");
		System.out
		.println("|                        EasierGOV                          |");
		System.out
		.println("|             EBM Research Service Governance               |");
		System.out
		.println("|             http://research.petalslink.org                |");
		System.out
		.println("|                                                           |");
		System.out
		.println(" -----------------------------------------------------------");
		System.out.println();
	}

	/**
	 * The main program
	 * 
	 * @param args
	 */
	public void launch(String[] args) {
		List<String> command = new ArrayList<String>();
		boolean console = true;

		if (args == null || args.length == 0) {
			command.add("start");
		}

		if(args != null) {
			for (String arg : args) {
				command.add(arg);
			}
		}

		try {
			if (command.contains(SHUTDOWN_COMMAND)) {
				System.out.println("Container is stopping...");
				this.shutdown();

			} else if (command.contains(VERSION_COMMAND)) {
				this.version();

			} else if (command.contains(START_COMMAND)) {
				System.out.println("Container is starting...");
				this.start();

				// show the commandLine mode if asked
				if (console) {
					this.commandLineMode(this.container);
				}
				if (this.systemExitHook != null) {
					Runtime.getRuntime()
					.removeShutdownHook(this.systemExitHook);
				}
				this.systemExitHook.run();

				System.exit(0);
			} else {
				System.out.println("Command '" + command + "' is unknown");
				this.printUsage();
				System.exit(-1);
			}
		} catch (Throwable e) {
			System.out.println("Command processing error : " + command);
			e.printStackTrace(System.err);
			if (this.systemExitHook != null) {
				Runtime.getRuntime().removeShutdownHook(this.systemExitHook);
			}
			System.exit(-1);
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 */
	public void start() throws Exception {
		// check if started or not stopped correctly
		if (this.locker.isLocked()) {
			throw new GovException(
			"Can not start the Container server, remove lock file from Container root path or stop server");
		}

		this.locker.lock();

		this.container.start();

		new InfoTask(this.container).doProcess(null);
		
		// add a hook if a terminate signal is sent from the command
		// line
		this.systemExitHook = new SystemExitHook(this.locker);
		Runtime.getRuntime().addShutdownHook(this.systemExitHook);

	}

	
	/*
	 * (non-Javadoc)
	 * 
	 */
	public void shutdown() throws Exception {
		if(this.systemExitHook == null) {
			this.systemExitHook = new SystemExitHook(this.locker);
		} 
		this.systemExitHook.start();
	}

	/*
	 * (non-Javadoc)
	 * 
	 */
	public void version() throws Exception {
		System.out.println("No version");
	}

	/**
	 * Print the launcher usage
	 */
	protected void printUsage() {
		System.out.println("usage:");
		System.out
		.println(" -start                  start the container");
		System.out
		.println(" -shutdowm               shutdown the container");
		System.out
		.println(" -version                get the container version");
	}

	/**
	 * Print the petals welcome message
	 */
	private void printStartedMessage() {
		this.printSpecificStartMessage();
		Date date = new Date(System.currentTimeMillis());
		SimpleDateFormat sdf = new SimpleDateFormat();
		System.out.println();
		System.out.println("Container " + this.getDistributionName()
				+ " distribution successfully started - " + sdf.format(date));
	}

	/**
	 * To be overrided for specific message
	 */
	protected void printSpecificStartMessage() {
	}

	/**
	 * Show a command line to interact with Petals. TODO we pass the PetalsAdmin
	 * object to the CommandReader to perform some actions. Find a better way.
	 * 
	 * @param node
	 * 
	 * @throws Exception
	 */
	protected void commandLineMode(Container node) throws Exception {
		CommandReader console = new CommandReader(node);
		console.read();
	}

	/**
	 * Get the distribution name
	 * 
	 * @return
	 */
	protected abstract String getDistributionName();
}
