/**
 * EasierSBS project - Java file
 * Copyright (C) 2011 EBM WebSourcing - Petals Link
 * 
 * EasierSBS is free project: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * EasierSBS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this program.
 * If not, see <http://www.gnu.org/licenses/lgpl-3.0.txt>.	
 * 
 */ 
package com.petalslink.easiersbs.registry.service.impl;

import java.net.URL;
import java.util.List;
import java.util.Set;

import javax.xml.namespace.QName;

import org.petalslink.abslayer.Factory;
import org.petalslink.abslayer.service.api.Description;
import org.petalslink.abslayer.service.api.Operation;
import org.petalslink.abslayer.service.api.Service;

import com.ebmwebsourcing.easybox.api.XmlContext;
import com.ebmwebsourcing.easybox.api.XmlContextFactory;
import com.ebmwebsourcing.easybox.api.XmlObjectReadException;
import com.ebmwebsourcing.easybox.api.XmlObjectReader;
import com.ebmwebsourcing.easywsdl11.api.element.Definitions;
import com.petalslink.easiersbs.registry.service.api.RegistryException;
import com.petalslink.easiersbs.registry.service.api.SemanticRegistryManager;
import com.petalslink.easiersbs.registry.service.api.model.Partner;
import com.petalslink.easiersbs.registry.service.api.model.SemanticProfile;
import com.petalslink.easiersbs.registry.service.api.registry.SemanticProfileRegistry;
import com.petalslink.easiersbs.registry.service.api.registry.TechnicalServiceRegistry;
import com.petalslink.easiersbs.registry.service.impl.registry.SemanticProfileRegistryImpl;
import com.petalslink.easiersbs.registry.service.impl.registry.TechnicalServiceRegistryImpl;
import com.petalslink.easiersbs.registry.service.impl.util.ServiceUtil;

/**
 * @author Nicolas Boissel-Dallier - Petals Link
 */
public class SemanticRegistryManagerImpl implements SemanticRegistryManager {

	SemanticProfileRegistry profiles = new SemanticProfileRegistryImpl();
	TechnicalServiceRegistry services = new TechnicalServiceRegistryImpl();
	
	XmlContextFactory xmlContextFactory = new XmlContextFactory();
	XmlContext xmlContext = xmlContextFactory.newContext();
	XmlObjectReader reader = xmlContext.createReader();
	
	public SemanticRegistryManagerImpl(){
		
	}

	public void addSemanticService(QName operationQName, Description wsdlDesc, Partner partner, URL wsdlUrl) {
		SemanticProfile profile = ServiceUtil.extractSemanticProfile(operationQName, wsdlDesc);
		profile.setPartner(partner);
		List<Service> listServices = ServiceUtil.findServicesByOperation(operationQName, wsdlDesc);
		for(Service service : listServices){
			services.addTechnicalService(service.getQName(), wsdlDesc, wsdlUrl);
		}
		profiles.addSemanticProfile(profile);	
	}
	
	public void addSemanticService(QName operationQName, Description wsdlDesc, URL wsdlUrl) {
		addSemanticService(operationQName, wsdlDesc, null, wsdlUrl);
	}

	public void addSemanticService(QName operationQName, URL wsdlUrl,
			Partner partner) throws RegistryException {
		Description wsdlDesc = this.getDescription(wsdlUrl);
		this.addSemanticService(operationQName, wsdlDesc, partner, wsdlUrl);
	}
	
	public void addSemanticService(QName operationQName, URL wsdlUrl) throws RegistryException {
		addSemanticService(operationQName, wsdlUrl, null);
	}

	public void addSemanticServices(Description wsdlDesc, Partner partner, URL wsdlUrl) {
		for(Operation op : ServiceUtil.getOperations(wsdlDesc)){
			this.addSemanticService(op.inferQName(), wsdlDesc, partner, wsdlUrl);
		}
	}
	
	public void addSemanticServices(Description wsdlDesc, URL wsdlUrl) {
		addSemanticServices(wsdlDesc, null, wsdlUrl);
	}
	
	public void addSemanticServices(URL wsdlUrl, Partner partner) throws RegistryException {
		Description wsdlDesc = this.getDescription(wsdlUrl);
		this.addSemanticServices(wsdlDesc, partner, wsdlUrl);
	}
	
	public void addSemanticServices(URL wsdlUrl) throws RegistryException {
		addSemanticServices(wsdlUrl, null);
	}
	
	public void clearRegistry() {
		services.removeAllTechnicalServices();
		profiles.removeAllSemanticProfiles();
	}


	public SemanticProfile getSemanticProfile(QName operationQName) {
		return profiles.getSemanticProfile(operationQName);
	}
	
	public Set<SemanticProfile> getAllSemanticProfiles() {
		return profiles.getAllSemanticProfiles();
	}

	public Set<SemanticProfile> findSemanticProfilesByPartner(Partner partner) {
		return profiles.findSemanticProfilesByPartner(partner);
	}

	public Description getTechnicalService(QName serviceQName) {
		return services.getTechnicalService(serviceQName);
	}
	
	public URL getWsdlUrl(QName serviceQName) {
		return services.getWsdlUrl(serviceQName);
	}
	
	public Set<URL> getWsdlUrls() {
		return services.getWsdlUrls();
	}

	public Set<Description> getAllTechnicalServices() {
		return services.getAllTechnicalServices();
	}

	public void removeSemanticServices(Description wsdlDesc) {
		// Remove all relative technical services
		for(Service service: wsdlDesc.getServices()){
			services.removeTechnicalService(service.getQName());
		}
		// Remove all relative semantic profiles
		for(Operation op : ServiceUtil.getOperations(wsdlDesc)){
			profiles.removeSemanticProfile(op.inferQName());
		}
	}

	public void removeSemanticService(QName operationQName) {
		// List relative services
		SemanticProfile profile = profiles.getSemanticProfile(operationQName);
		profiles.removeSemanticProfile(operationQName);
		// Is relative technical services always used?
		for(QName serviceQName : profile.getServiceQNames()){
			boolean used = false;
			Description wsdlDesc = services.getTechnicalService(serviceQName);
			Set<Operation> ops = ServiceUtil.findOperationsByService(serviceQName, wsdlDesc);
			for(Operation op : ops){
				if(profiles.getSemanticProfile(op.inferQName()) != null){
					used = true;
					break;
				}
			}
			// Remove unused technical services
			if(!used){
				services.removeTechnicalService(serviceQName);
			}
		}
	}
	
	public void removeSemanticProfilesByPartner(Partner partner) {
		profiles.removeSemanticProfilesByPartner(partner);
	}

	public Description getDescription(URL wsdlUrl) throws RegistryException {
		Definitions def = null;
		try {
			def = reader.readDocument(wsdlUrl, Definitions.class);
		} catch (XmlObjectReadException e) {
			throw new RegistryException("Impossible to parse WSDL file at " + wsdlUrl.toString(), e);
		}
		return (Description) Factory.getInstance().wrap(def);
	}

}
