/**
 * EasierSBS project - Java file
 * Copyright (C) 2011 EBM WebSourcing - Petals Link
 * 
 * EasierSBS is free project: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * EasierSBS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this program.
 * If not, see <http://www.gnu.org/licenses/lgpl-3.0.txt>.	
 * 
 */ 
package com.petalslink.easiersbs.registry.service.impl.util;

import java.net.URI;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.xml.namespace.QName;

import org.petalslink.abslayer.service.api.Description;
import org.petalslink.abslayer.service.api.Endpoint;
import org.petalslink.abslayer.service.api.Interface;
import org.petalslink.abslayer.service.api.Operation;
import org.petalslink.abslayer.service.api.Part;
import org.petalslink.abslayer.service.api.Service;

import com.ebmwebsourcing.easybox.api.XmlObject;
import com.ebmwebsourcing.easybox.api.XmlObjectNode;
import com.ebmwebsourcing.easysawsdl10.api.SawsdlHelper;
import com.ebmwebsourcing.easyschema10.api.SchemaHelper;
import com.ebmwebsourcing.easyschema10.api.element.Choice;
import com.ebmwebsourcing.easyschema10.api.element.ComplexType;
import com.ebmwebsourcing.easyschema10.api.element.Element;
import com.ebmwebsourcing.easyschema10.api.element.Sequence;
import com.ebmwebsourcing.easyschema10.api.type.Type;
import com.ebmwebsourcing.easyschema10.api.with.WithElements;
import com.petalslink.easiersbs.registry.service.api.model.SemanticElement;
import com.petalslink.easiersbs.registry.service.api.model.SemanticPart;
import com.petalslink.easiersbs.registry.service.api.model.SemanticProfile;
import com.petalslink.easiersbs.registry.service.impl.model.SemanticElementImpl;
import com.petalslink.easiersbs.registry.service.impl.model.SemanticPartImpl;
import com.petalslink.easiersbs.registry.service.impl.model.SemanticProfileImpl;

/**
 * @author Nicolas Boissel-Dallier - Petals Link
 */
public class ServiceUtil {
	
	public static Set<Operation> getOperations(Description wsdlDesc){
		Set<Operation> operations = new HashSet<Operation>();
		for(Interface inter : wsdlDesc.getInterfaces()){
			operations.addAll(Arrays.asList(inter.getOperations()));
		}
		return operations;
	}
	
	public static Operation findOperation(QName operationQName, Description wsdlDesc) {
		for(Interface inter : wsdlDesc.getInterfaces()){
			Operation op = inter.getOperation(operationQName);
			if(op != null){
				return op;
			}
		}
		return null;
	}

	public static List<Service> findServicesByOperation(QName operationQName, Description wsdlDesc){
		Operation op = findOperation(operationQName, wsdlDesc);
		List<Service> services = wsdlDesc.findServicesImplementingInterface(op.getParentInterface());
		return services;
	}
	
	public static Set<Operation> findOperationsByService(QName serviceQName, Description wsdlDesc){
		Set<Operation> ops = new HashSet<Operation>();
		Service serv = wsdlDesc.findService(serviceQName);
		for(Endpoint end : serv.getEndpoints()){
			ops.addAll(Arrays.asList(end.getBinding().getInterface().getOperations()));
		}
		return ops;
	}
	
	public static Set<URI> getFlattenConcepts(Set<SemanticElement> elements){
		Set<URI> concepts = new HashSet<URI>();
		for(SemanticElement element : elements){
			concepts.addAll(element.getSemanticConcepts());
		}
		return concepts;
	}
	
	public static SemanticProfile extractSemanticProfile(QName operationQName, Description wsdlDesc) {
		SemanticProfile profile = new SemanticProfileImpl();
		
		Operation op = findOperation(operationQName, wsdlDesc);
		
		// Technical information
		profile.setOperationQName(operationQName);
		for(Service serv : findServicesByOperation(operationQName, wsdlDesc)){
			profile.addServiceQName(serv.getQName());
		}
		
		// Interface
		SemanticPart interPart = extractSemanticPart(op.getParentInterface().getModel());
		profile.setSemanticInterface(interPart);
		
		// Operation
		SemanticPart opPart = extractSemanticPart(op.getModel());
		profile.setSemanticOperation(opPart);
		
		// Input
		Set<SemanticElement> inputElements = new HashSet<SemanticElement>();
		if(op.getInput() != null) {
			for(Part part : op.getInput().getParts()){
				if(part != null){
					extractAllSemanticElements(part, inputElements);
				}
			}
		}
		profile.setInputSemanticElements(inputElements);
		
		// Output
		Set<SemanticElement> outputElements = new HashSet<SemanticElement>();
		if(op.getOutput() != null) {
			for(Part part : op.getOutput().getParts()){
				if(part != null){
					extractAllSemanticElements(part, outputElements);
				}
			}
		}
		
		profile.setOutputSemanticElements(outputElements);
			
		return profile;
		
	}
	
	public static SemanticPart extractSemanticPart(XmlObject object){
		SemanticPart part = new SemanticPartImpl();
		URI[] concepts = SawsdlHelper.getModelReference(object);
		for(URI concept : concepts){
			part.addSemanticConcept(concept);
		}
		return part;
	}
	
	public static SemanticElement extractSemanticElement(Element element){
		SemanticElement semantic = new SemanticElementImpl();

		URI[] concepts = SawsdlHelper.getModelReference(element);
		
		for(URI concept : concepts){
			semantic.addSemanticConcept(concept);
		}
		
		URI[] typeConcepts = SawsdlHelper.getModelReference(element.findType());
		for(URI concept : typeConcepts){
			semantic.addSemanticConcept(concept);
		}
		
		if(element.getMinOccurs() == 0){
			semantic.setRequired(false);
		} else {
			semantic.setRequired(true);
		}
		
		semantic.setElement(element);
		
		return semantic;
	}
	
	private static void extractAllSemanticElements(Element element, Set<SemanticElement> elementList){
		if(element.hasRef()){
			QName ref = element.getRef();
			Element realElement = SchemaHelper.findElementByQName(element.getXmlObjectBaseRoot(), ref);
			extractAllSemanticElements(realElement, elementList);
		} else {
			SemanticElement semanticElement = extractSemanticElement(element);
			if(!elementList.contains(semanticElement)){
				elementList.add(semanticElement);
				Type type = element.findType();
				if(type instanceof ComplexType){
					XmlObjectNode[] nodes = type.getXmlObjectChildren();
					for(XmlObjectNode node : nodes){
						extractAllSemanticElements(node, elementList);
					}
				}
			}
		}
	}
	
	private static void extractAllSemanticElements(XmlObjectNode node, Set<SemanticElement> elementList){
		if(node instanceof WithElements){
			for(Element element : ((WithElements)node).getElements()){
				extractAllSemanticElements(element, elementList);
			}
		}
		if(node instanceof Sequence || node instanceof Choice){
			for(XmlObjectNode child : node.getXmlObjectChildren()){
				extractAllSemanticElements(child, elementList);
			}
		}
	}
	
	private static void extractAllSemanticElements(Part part, Set<SemanticElement> elementList){
		Element element = null;
		if(part.getElement() != null){
			element = part.getElement();
		} else if(part.getType() != null) { 
			// Create an element representing the expected one
			element = part.createElement();
			element.setName(part.getQName().getLocalPart());
			element.setType(part.getType());
			// TODO : find why each XPath request is executed twice
			Type type = SchemaHelper.findTypeByQName(part.getDescription().getModel(), part.getType());
			SchemaHelper.findParentSchema(type).addElement(element);
		}
		
		if(element != null){
			extractAllSemanticElements(element, elementList);
		}
	}
	
}
