/**
 * EasierSBS project - Java file
 * Copyright (C) 2011 EBM WebSourcing - Petals Link
 * 
 * EasierSBS is free project: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * EasierSBS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this program.
 * If not, see <http://www.gnu.org/licenses/lgpl-3.0.txt>.	
 * 
 */ 
package com.petalslink.easiersbs.registry.service.impl;

import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.logging.Logger;

import javax.xml.namespace.QName;

import org.petalslink.abslayer.service.api.Description;
import org.petalslink.abslayer.service.api.Operation;
import org.petalslink.abslayer.service.api.Service;

import com.ebmwebsourcing.easybox.api.XmlContext;
import com.ebmwebsourcing.easybox.api.XmlContextFactory;
import com.petalslink.easiersbs.registry.service.api.RegistryException;
import com.petalslink.easiersbs.registry.service.api.SemanticRegistryManager;
import com.petalslink.easiersbs.registry.service.api.model.Partner;
import com.petalslink.easiersbs.registry.service.api.model.SemanticProfile;
import com.petalslink.easiersbs.registry.service.api.registry.SemanticProfileRegistry;
import com.petalslink.easiersbs.registry.service.api.registry.TechnicalServiceRegistry;
import com.petalslink.easiersbs.registry.service.impl.registry.ImportWsdlCallableImpl;
import com.petalslink.easiersbs.registry.service.impl.registry.SemanticProfileRegistryImpl;
import com.petalslink.easiersbs.registry.service.impl.registry.TechnicalServiceRegistryImpl;
import com.petalslink.easiersbs.registry.service.impl.util.ServiceUtil;

/**
 * @author Nicolas Boissel-Dallier - Petals Link
 */
public class SemanticRegistryManagerImpl implements SemanticRegistryManager {

	private Logger logger = Logger.getLogger(this.getClass().getName());
	
	private SemanticProfileRegistry profiles = new SemanticProfileRegistryImpl();
	private TechnicalServiceRegistry services = new TechnicalServiceRegistryImpl();
	
	private static XmlContextFactory xmlContextFactory = new XmlContextFactory();
	private static XmlContext xmlContext = xmlContextFactory.newContext();
	
	public SemanticRegistryManagerImpl(){
		
	}

	public void addSemanticService(QName operationQName, Description wsdlDesc, Partner partner, URL wsdlUrl) {
		SemanticProfile profile = ServiceUtil.extractSemanticProfile(operationQName, wsdlDesc);
		profile.setPartner(partner);
		List<Service> listServices = ServiceUtil.findServicesByOperation(operationQName, wsdlDesc);
		for(Service service : listServices){
			services.addTechnicalService(service.getQName(), wsdlDesc, wsdlUrl);
		}
		profiles.addSemanticProfile(profile);
	}
	
	public void addSemanticService(QName operationQName, Description wsdlDesc, URL wsdlUrl) {
		addSemanticService(operationQName, wsdlDesc, null, wsdlUrl);
	}

	public void addSemanticService(QName operationQName, URL wsdlUrl,
			Partner partner) throws RegistryException {
		Description wsdlDesc = this.getDescription(wsdlUrl);
		this.addSemanticService(operationQName, wsdlDesc, partner, wsdlUrl);
	}
	
	public void addSemanticService(QName operationQName, URL wsdlUrl) throws RegistryException {
		addSemanticService(operationQName, wsdlUrl, null);
	}

	public void addSemanticServices(Description wsdlDesc, Partner partner, URL wsdlUrl) {
		for(Operation op : ServiceUtil.getOperations(wsdlDesc)){
			this.addSemanticService(op.inferQName(), wsdlDesc, partner, wsdlUrl);
		}
	}
	
	public void addSemanticServices(Description wsdlDesc, URL wsdlUrl) {
		addSemanticServices(wsdlDesc, null, wsdlUrl);
	}
	
	public void addSemanticServices(URL wsdlUrl, Partner partner) throws RegistryException {
		Description wsdlDesc = this.getDescription(wsdlUrl);
		this.addSemanticServices(wsdlDesc, partner, wsdlUrl);
	}
	
	public void addSemanticServices(URL wsdlUrl) throws RegistryException {
		addSemanticServices(wsdlUrl, null);
	}
	
	public void addMultipleSemanticServices(Set<URL> wsdlUrls, Partner partner) {
		// MultiThread import
		ExecutorService exec = Executors.newFixedThreadPool(4);
		ExecutorCompletionService<Boolean> compExec = new ExecutorCompletionService<Boolean>(exec);

		Map<URL, Future<Boolean>> results = new HashMap<URL, Future<Boolean>>();
		
		// WSDL Import
		for(URL wsdlUrl : wsdlUrls){
			ImportWsdlCallableImpl call = new ImportWsdlCallableImpl(profiles, services, xmlContext, partner, wsdlUrl);
			Future<Boolean> success = compExec.submit(call);
			results.put(wsdlUrl, success);
		}
		
		// Are imports done?
		boolean done = false;
		while(!done){
			done = true;
			for(Future<Boolean> future : results.values()){
				if(!future.isDone()){
					done = false;
				}
			}
			if(!done){
				try {
					Thread.sleep(500);
				} catch (InterruptedException e) {
					throw new RuntimeException(e);
				}
			}
		}
		
		// Log problems
		for(Entry<URL, Future<Boolean>> result : results.entrySet()){
			try {
				if( ! result.getValue().get()){
					logger.warning("Impossible to import WSDL at " + result.getKey());
				}
			} catch (InterruptedException e) {
				throw new RuntimeException(e);
			} catch (ExecutionException e) {
				throw new RuntimeException(e);
			}
		}
		
		exec.shutdown();
	}
	
	public void clearRegistry() {
		services.removeAllTechnicalServices();
		profiles.removeAllSemanticProfiles();
	}

	public SemanticProfile getSemanticProfile(QName operationQName) {
		return profiles.getSemanticProfile(operationQName);
	}
	
	public Set<SemanticProfile> getAllSemanticProfiles() {
		return profiles.getAllSemanticProfiles();
	}

	public Set<SemanticProfile> findSemanticProfilesByPartner(Partner partner) {
		return profiles.findSemanticProfilesByPartner(partner);
	}

	public Description getTechnicalService(QName serviceQName) throws RegistryException {
//		URL wsdlUrl = services.getWsdlUrl(serviceQName);
//		return getDescription(wsdlUrl);
		return services.getTechnicalService(serviceQName);
	}
	
	public URL getWsdlUrl(QName serviceQName) {
		return services.getWsdlUrl(serviceQName);
	}
	
	public Set<URL> getWsdlUrls() {
		return services.getWsdlUrls();
	}

	public Set<Description> getAllTechnicalServices() {
		return services.getAllTechnicalServices();
	}

	public void removeSemanticServices(Description wsdlDesc) {
		// Remove all relative technical services
		for(Service service: wsdlDesc.getServices()){
			services.removeTechnicalService(service.getQName());
		}
		// Remove all relative semantic profiles
		for(Operation op : ServiceUtil.getOperations(wsdlDesc)){
			profiles.removeSemanticProfile(op.inferQName());
		}
	}

	public void removeSemanticService(QName operationQName) {
		// List relative services
		SemanticProfile profile = profiles.getSemanticProfile(operationQName);
		profiles.removeSemanticProfile(operationQName);
		// Is relative technical services always used?
		for(QName serviceQName : profile.getServiceQNames()){
			boolean used = false;
			Description wsdlDesc = services.getTechnicalService(serviceQName);
			Set<Operation> ops = ServiceUtil.findOperationsByService(serviceQName, wsdlDesc);
			for(Operation op : ops){
				if(profiles.getSemanticProfile(op.inferQName()) != null){
					used = true;
					break;
				}
			}
			// Remove unused technical services
			if(!used){
				services.removeTechnicalService(serviceQName);
			}
		}
	}
	
	public void removeSemanticProfilesByPartner(Partner partner) {
		profiles.removeSemanticProfilesByPartner(partner);
	}

	public Description getDescription(URL wsdlUrl) throws RegistryException {
		return ServiceUtil.getDescription(wsdlUrl, xmlContext);
	}

}
