/**
 * EasierSBS project - Java file
 * Copyright (C) 2011 EBM WebSourcing - Petals Link
 * 
 * EasierSBS is free project: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * EasierSBS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this program.
 * If not, see <http://www.gnu.org/licenses/lgpl-3.0.txt>.	
 * 
 */ 
package com.petalslink.easiersbs.matching.service.matcher;

import java.util.Set;

import com.petalslink.easiersbs.matching.service.api.matcher.MatcherProperties;
import com.petalslink.easiersbs.matching.service.api.matcher.MatchingResult;
import com.petalslink.easiersbs.matching.service.api.matcher.SemanticMatcher;
import com.petalslink.easiersbs.matching.service.api.profile.SearchProfile;
import com.petalslink.easiersbs.matching.service.api.profile.infered.InferedSearchProfile;
import com.petalslink.easiersbs.matching.service.util.SimilarityUtil;
import com.petalslink.easiersbs.reasoner.api.ReasonerException;
import com.petalslink.easiersbs.reasoner.api.ReasonerFactory;
import com.petalslink.easiersbs.reasoner.api.engine.Reasoner;
import com.petalslink.easiersbs.reasoner.api.ontology.OntologyManager;
import com.petalslink.easiersbs.registry.service.api.SemanticRegistryManager;
import com.petalslink.easiersbs.registry.service.api.model.SemanticProfile;

/**
 * @author Nicolas Boissel-Dallier - Petals Link
 */
public class SemanticMatcherImpl extends AbstractMatcherImpl implements
		SemanticMatcher {

	private Reasoner reasoner = null;
	private OntologyManager manager = null;
	
	public SemanticMatcherImpl(ReasonerFactory factory, SemanticRegistryManager registry, MatcherProperties props) throws ReasonerException{
		this.registry = registry;
		this.props = props;
		this.manager = factory.getOntologyManager();
		this.reasoner = factory.newReasoner(manager);
	}
	
	public SemanticMatcherImpl(ReasonerFactory factory, SemanticRegistryManager registry) throws ReasonerException{
		this(factory, registry, new MatcherPropertiesImpl());
	}


	public Reasoner getReasoner() {
		return reasoner;
	}

	public OntologyManager getOntologyManager() {
		return manager;
	}

	public MatchingResult findServices(SearchProfile profile,
			Set<SemanticProfile> serviceList) {
		MatchingResult res = new MatchingResultImpl(profile);
		
		// Infer Search profile
		InferedSearchProfile infProfile = profile.infer(reasoner, props);
		
		// Compare semantic concepts with registry services
		for(SemanticProfile service : serviceList){
			
			double interRate = SimilarityUtil.measureSemanticSimilarity(
					infProfile.getSemanticInterface().getInferedSemanticConcepts(), 
					service.getSemanticInterface().getSemanticConcepts());
			
			double opRate = SimilarityUtil.measureSemanticSimilarity(
					infProfile.getSemanticOperation().getInferedSemanticConcepts(),
					service.getSemanticOperation().getSemanticConcepts());
			
			double inRate = SimilarityUtil.measureElementSimilarity(
					infProfile.getInputSemanticElements(), 
					service.getInputSemanticElements(),
					props);

			double outRate = SimilarityUtil.measureElementSimilarity(
					infProfile.getOutputSemanticElements(),
					service.getOutputSemanticElements(), 
					props);
			
			
			// Qualify concept coverage
			
			double rate = ((interRate + opRate) / 2.0) * props.getOperationWeight()
						+ ((inRate + outRate) / 2.0) * props.getInputOutputWeigth();
			
			if(rate > props.getSemanticThreshold()){
				res.addResultProfile(service, rate);
			}
			
		}
		
		return res;
	}
	
}
