/**
 * EasierSBS project - Java file
 * Copyright (C) 2011 EBM WebSourcing - Petals Link
 * 
 * EasierSBS is free project: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * EasierSBS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this program.
 * If not, see <http://www.gnu.org/licenses/lgpl-3.0.txt>.	
 * 
 */ 
package com.petalslink.easiersbs.matching.service.matcher;

import java.util.List;
import java.util.Set;

import com.petalslink.easiersbs.matching.service.SBSFactoryImpl;
import com.petalslink.easiersbs.matching.service.api.EasierSBSException;
import com.petalslink.easiersbs.matching.service.api.SBSFactory;
import com.petalslink.easiersbs.matching.service.api.matcher.MatcherProperties;
import com.petalslink.easiersbs.matching.service.api.matcher.MatchingResult;
import com.petalslink.easiersbs.matching.service.api.matcher.SyntacticMatcher;
import com.petalslink.easiersbs.matching.service.api.matcher.similarity.SimilarityMeasure;
import com.petalslink.easiersbs.matching.service.api.profile.SearchProfile;
import com.petalslink.easiersbs.matching.service.util.SimilarityUtil;
import com.petalslink.easiersbs.registry.service.api.SemanticRegistryManager;
import com.petalslink.easiersbs.registry.service.api.model.SemanticProfile;

/**
 * @author Nicolas Boissel-Dallier - Petals Link
 */
public class SyntacticMatcherImpl extends AbstractMatcherImpl implements SyntacticMatcher {

	private SBSFactory factory = SBSFactoryImpl.getInstance();
	private SimilarityMeasure similarity;

	public SyntacticMatcherImpl(SemanticRegistryManager registry, MatcherProperties props) throws EasierSBSException{
		this.registry = registry;
		this.props = props;
		this.similarity = factory.newSimilarityMeasure(props.getSimilarityType(), props.getLevenshteinLimit());
	}
	
	public SyntacticMatcherImpl(SemanticRegistryManager registry) throws EasierSBSException{
		this(registry, new MatcherPropertiesImpl());
	}


	public SimilarityMeasure getSimilarityMeasure() {
		return similarity;
	}

	public MatchingResult findServices(SearchProfile profile, 
			Set<SemanticProfile> serviceList){
		MatchingResult res = new MatchingResultImpl(profile);
		
		// Extract functional related terms (operation + interface)
		List<String> functionalWords = SimilarityUtil.extractUriRelatedWords(profile.getSemanticOperation().getSemanticConcepts());
		functionalWords.addAll(SimilarityUtil.extractUriRelatedWords(profile.getSemanticInterface().getSemanticConcepts()));
		
		// Extract input and output terms
		List<String> inputWords = SimilarityUtil.extractElementRelatedWords(profile.getInputSemanticElements());
		List<String> outputWords = SimilarityUtil.extractElementRelatedWords(profile.getOutputSemanticElements());
		
		// Computation of similarity measure for each service
		for(SemanticProfile service : serviceList){
			double finalRate = 0.0;
			Double ioRate = null;
			Double funcRate = null;
			
			if(props.getOperationWeight() != 0.0){
				List<String> serviceFunctionalWords = SimilarityUtil.extractUriRelatedWords(service.getSemanticOperation().getSemanticConcepts());
				serviceFunctionalWords.addAll(SimilarityUtil.extractUriRelatedWords(service.getSemanticInterface().getSemanticConcepts()));
				
				funcRate = similarity.measureSimilarity(functionalWords, serviceFunctionalWords);
			}
			
			if(props.getInputOutputWeigth() != 0.0){
				List<String> serviceInputWords = SimilarityUtil.extractElementRelatedWords(service.getInputSemanticElements());
				double iRate = similarity.measureSimilarity(inputWords, serviceInputWords);
				
				List<String> serviceOutputWords = SimilarityUtil.extractElementRelatedWords(service.getOutputSemanticElements());
				double oRate = similarity.measureSimilarity(outputWords, serviceOutputWords);
				
				ioRate = (iRate + oRate) / 2.0;
			}
			
			// If one weight is set to 0, we only consider the other one
			if(ioRate == null || ioRate.isNaN()){
				finalRate = funcRate;
			} else if(funcRate == null || funcRate.isNaN()){
				finalRate = ioRate;
			} else {
				finalRate = (ioRate * props.getInputOutputWeigth()) + (funcRate * props.getOperationWeight());
			}
			
			// Add service to result map if it feet with acceptance threshold
			if(finalRate >= props.getSyntacticThreshold()){
				res.addResultProfile(service, finalRate);
			}
			
		}
		
		return res;
	}

}
