/**
 * EasierSBS project - Java file
 * Copyright (C) 2011 EBM WebSourcing - Petals Link
 * 
 * EasierSBS is free project: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * EasierSBS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this program.
 * If not, see <http://www.gnu.org/licenses/lgpl-3.0.txt>.	
 * 
 */ 
package com.petalslink.easiersbs.matching.service.matcher;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import com.petalslink.easiersbs.matching.service.SBSFactoryImpl;
import com.petalslink.easiersbs.matching.service.api.EasierSBSException;
import com.petalslink.easiersbs.matching.service.api.SBSFactory;
import com.petalslink.easiersbs.matching.service.api.matcher.MatcherProperties;
import com.petalslink.easiersbs.matching.service.api.matcher.MatchingResult;
import com.petalslink.easiersbs.matching.service.api.matcher.SyntacticMatcher;
import com.petalslink.easiersbs.matching.service.api.matcher.similarity.SimilarityMeasure;
import com.petalslink.easiersbs.matching.service.api.profile.SearchProfile;
import com.petalslink.easiersbs.matching.service.matcher.computation.SyntacticComputationImpl;
import com.petalslink.easiersbs.matching.service.util.SimilarityUtil;
import com.petalslink.easiersbs.registry.service.api.SemanticRegistryManager;
import com.petalslink.easiersbs.registry.service.api.model.SemanticProfile;

/**
 * @author Nicolas Boissel-Dallier - Petals Link
 */
public class SyntacticMatcherImpl extends AbstractMatcherImpl implements SyntacticMatcher {

	private SBSFactory factory = SBSFactoryImpl.getInstance();
	private SimilarityMeasure similarity;
	
	// MultiThread computation
	private ExecutorService exec;
	private ExecutorCompletionService<Double> compExec;

	public SyntacticMatcherImpl(SemanticRegistryManager registry, MatcherProperties props) throws EasierSBSException{
		this.registry = registry;
		this.props = props;
		this.similarity = factory.newSimilarityMeasure(props.getSimilarityType(), props.getLevenshteinLimit());
		exec = Executors.newFixedThreadPool(props.getMaxThreads());
		compExec = new ExecutorCompletionService<Double>(exec);
	}
	
	public SyntacticMatcherImpl(SemanticRegistryManager registry) throws EasierSBSException{
		this(registry, new MatcherPropertiesImpl());
	}


	public SimilarityMeasure getSimilarityMeasure() {
		return similarity;
	}

	public MatchingResult findServices(SearchProfile profile, 
			Set<SemanticProfile> serviceList){
		MatchingResult res = new MatchingResultImpl(profile);
		
		try {
			this.similarity = factory.newSimilarityMeasure(props.getSimilarityType(), props.getLevenshteinLimit());
		} catch (EasierSBSException e) {
			e.printStackTrace();
		}
		
		// Extract functional related terms (operation + interface)
		List<String> functionalWords = SimilarityUtil.extractUriRelatedWords(profile.getSemanticOperation().getSemanticConcepts());
		functionalWords.addAll(SimilarityUtil.extractUriRelatedWords(profile.getSemanticInterface().getSemanticConcepts()));
		
		// Extract input and output terms
		List<String> inputWords = SimilarityUtil.extractElementRelatedWords(profile.getInputSemanticElements());
		List<String> outputWords = SimilarityUtil.extractElementRelatedWords(profile.getOutputSemanticElements());
		
		
		
		Map<SemanticProfile, Future<Double>> results = new HashMap<SemanticProfile, Future<Double>>();
		
		// Computation of similarity measure for each service
		for(SemanticProfile service : serviceList){
			SyntacticComputationImpl compute = new SyntacticComputationImpl(props, similarity, 
														functionalWords, inputWords, outputWords, service);
			Future<Double> rate = compExec.submit(compute);
			results.put(service, rate);
		}

		boolean done = false;
		while(!done){
			done = true;
			for(Future<Double> future : results.values()){
				if(!future.isDone()){
					done = false;
				}
			}
			if(!done){
				try {
					Thread.sleep(100);
				} catch (InterruptedException e) {
					throw new RuntimeException(e);
				}
			}
		}
		
		// Completion of expected result
		for(Entry<SemanticProfile, Future<Double>> result : results.entrySet()){
			// Add service to result map if it feet with acceptance threshold
			try {
				if(result.getValue().get() >= props.getSyntacticThreshold()){
					res.addResultProfile(result.getKey(), result.getValue().get());
				}
			} catch (InterruptedException e) {
				throw new RuntimeException(e);
			} catch (ExecutionException e) {
				throw new RuntimeException(e);
			}
		}
		
		return res;
	}

}
