/**
 * EasierSBS project - Java file
 * Copyright (C) 2011 EBM WebSourcing - Petals Link
 * 
 * EasierSBS is free project: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * EasierSBS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this program.
 * If not, see <http://www.gnu.org/licenses/lgpl-3.0.txt>.	
 * 
 */ 
package com.petalslink.easiersbs.matching.service.matcher;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import com.petalslink.easiersbs.matching.service.SBSFactoryImpl;
import com.petalslink.easiersbs.matching.service.api.EasierSBSException;
import com.petalslink.easiersbs.matching.service.api.SBSFactory;
import com.petalslink.easiersbs.matching.service.api.matcher.MatcherProperties;
import com.petalslink.easiersbs.matching.service.api.matcher.MatchingResult;
import com.petalslink.easiersbs.matching.service.api.matcher.SyntacticMatcher;
import com.petalslink.easiersbs.matching.service.api.matcher.similarity.SimilarityMeasure;
import com.petalslink.easiersbs.matching.service.api.profile.SearchProfile;
import com.petalslink.easiersbs.matching.service.api.profile.rated.RatedSemanticProfile;
import com.petalslink.easiersbs.matching.service.matcher.computation.SyntacticComputationImpl;
import com.petalslink.easiersbs.matching.service.util.SimilarityUtil;
import com.petalslink.easiersbs.registry.service.api.SemanticRegistryManager;
import com.petalslink.easiersbs.registry.service.api.model.SemanticProfile;
import com.petalslink.easiersbs.registry.service.impl.util.ServiceUtil;

/**
 * @author Nicolas Boissel-Dallier - Petals Link
 */
public class SyntacticMatcherImpl extends AbstractMatcherImpl implements SyntacticMatcher {

	private SBSFactory factory = SBSFactoryImpl.getInstance();
	private SimilarityMeasure similarity;
	
	// MultiThread computation
	private ExecutorService exec;
	private ExecutorCompletionService<RatedSemanticProfile> compExec;

	public SyntacticMatcherImpl(SemanticRegistryManager registry, MatcherProperties props) throws EasierSBSException{
		this.registry = registry;
		this.props = props;
		this.similarity = factory.newSimilarityMeasure(props.getSimilarityType(), props.getLevenshteinLimit());
		exec = Executors.newFixedThreadPool(props.getMaxThreads());
		compExec = new ExecutorCompletionService<RatedSemanticProfile>(exec);
	}
	
	public SyntacticMatcherImpl(SemanticRegistryManager registry) throws EasierSBSException{
		this(registry, new MatcherPropertiesImpl());
	}


	public SimilarityMeasure getSimilarityMeasure() {
		return similarity;
	}

	public MatchingResult findServices(SearchProfile profile, 
			Set<SemanticProfile> serviceList){
		MatchingResult res = new MatchingResultImpl(profile);
		
		try {
			this.similarity = factory.newSimilarityMeasure(props.getSimilarityType(), props.getLevenshteinLimit());
		} catch (EasierSBSException e) {
			e.printStackTrace();
		}
		
		// Extract functional related terms (operation + interface)
		List<String> functionalWords = SimilarityUtil.extractUriRelatedWords(profile.getSemanticOperation().getSemanticConcepts());
		functionalWords.addAll(SimilarityUtil.splitInWords(profile.getSemanticOperation().getName()));
		functionalWords.addAll(SimilarityUtil.extractUriRelatedWords(profile.getSemanticInterface().getSemanticConcepts()));
		functionalWords.addAll(SimilarityUtil.splitInWords(profile.getSemanticInterface().getName()));
		
		// Extract input and output terms
		List<String> inputWords = SimilarityUtil.extractElementRelatedWords(ServiceUtil.getFlattenElements(profile.getInputSemanticElement()));
		List<String> outputWords = SimilarityUtil.extractElementRelatedWords(ServiceUtil.getFlattenElements(profile.getOutputSemanticElement()));
		
		Set<Future<RatedSemanticProfile>> results = new HashSet<Future<RatedSemanticProfile>>();
		
		// Computation of similarity measure for each service
		for(SemanticProfile service : serviceList){
			SyntacticComputationImpl compute = new SyntacticComputationImpl(props, similarity, 
														functionalWords, inputWords, outputWords, service);
			results.add(compExec.submit(compute));
		}

		boolean done = false;
		while(!done){
			done = true;
			for(Future<RatedSemanticProfile> future : results){
				if(!future.isDone()){
					done = false;
					break;
				}
			}
			if(!done){
				try {
					Thread.sleep(50);
				} catch (InterruptedException e) {
					throw new RuntimeException(e);
				}
			}
		}
		
		// Completion of expected result
		for(Future<RatedSemanticProfile> result : results){
			// Add service to result map if it feet with acceptance threshold
			try {
				if(result.get().getProfileRate() >= props.getSyntacticThreshold()){
					res.addResultProfile(result.get());
				}
			} catch (InterruptedException e) {
				throw new RuntimeException(e);
			} catch (ExecutionException e) {
				throw new RuntimeException(e);
			}
		}
		
		return res;
	}

}
