/**
 * EasierSBS project - Java file
 * Copyright (C) 2011 EBM WebSourcing - Petals Link
 * 
 * EasierSBS is free project: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * EasierSBS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this program.
 * If not, see <http://www.gnu.org/licenses/lgpl-3.0.txt>.	
 * 
 */ 
package com.petalslink.easiersbs.matching.service.util;

import java.net.URI;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;

import javax.xml.namespace.QName;

import com.petalslink.easiersbs.matching.service.api.matcher.MatcherProperties;
import com.petalslink.easiersbs.matching.service.api.profile.SearchElement;
import com.petalslink.easiersbs.matching.service.api.profile.inferred.InferredConcept;
import com.petalslink.easiersbs.matching.service.api.profile.inferred.InferredElement;
import com.petalslink.easiersbs.matching.service.api.profile.inferred.RatedURI;
import com.petalslink.easiersbs.matching.service.api.profile.rated.RatedSemanticProfile;
import com.petalslink.easiersbs.matching.service.profile.SearchElementImpl;
import com.petalslink.easiersbs.matching.service.profile.inferred.InferredConceptImpl;
import com.petalslink.easiersbs.matching.service.profile.inferred.RatedURIImpl;
import com.petalslink.easiersbs.reasoner.api.engine.Reasoner;
import com.petalslink.easiersbs.registry.service.api.model.SemanticElement;
import com.petalslink.easiersbs.registry.service.api.model.SemanticProfile;

/**
 * @author Nicolas Boissel-Dallier - Petals Link
 */
public class ProfileUtil {
	
	public static Set<RatedURI> getFlattenRatedUris(InferredElement element){
		Set<InferredElement> listElements = new HashSet<InferredElement>();
		return getFlattenRatedUris(element, listElements);
	}
	
	private static Set<RatedURI> getFlattenRatedUris(InferredElement element, Set<InferredElement> listElements){
		Set<RatedURI> res = new HashSet<RatedURI>();
		if( ! listElements.contains(element)){
			listElements.add(element);
			for(InferredConcept concept : element.getInferedSemanticConcepts()){
				res.addAll(concept.getRatedSemanticConcepts());
			}
			for(InferredElement child : element.getChildElements()){
				res.addAll(getFlattenRatedUris(child, listElements));
			}
		}
		return res;
	}
	
	/**
	 * Convert a SemanticElement hierarchy into SearchElement one
	 * 
	 * @param element SemanticElement source
	 * @return target SearchElement
	 */
	public static SearchElement convertSemanticElement(SemanticElement element){
		return convertSemanticElementHierarchy(element, new HashMap<QName, SearchElement>());
	}
	
	private static SearchElement convertSemanticElementHierarchy(SemanticElement elem, Map<QName, SearchElement> elementList){
		SearchElement res = new SearchElementImpl();
		res.setSemanticConcepts(elem.getSemanticConcepts());
		res.setElementQName(elem.getElementQName());
		res.setName(elem.getName());
		res.setRequired(elem.isRequired());
		elementList.put(elem.getElementQName(), res);
		for(SemanticElement child : elem.getChildElements()){
			if(elementList.containsKey(child.getElementQName())){
				res.addChildElement(elementList.get(child.getElementQName()));
			} else {
				res.addChildElement(convertSemanticElementHierarchy(child, elementList));
			}
		}
		return res;
	}
	
	/**
	 * Search for related concepts in available ontologies
	 * @param concept Initial concept URI
	 * @param reasoner EasierSBS Reasoner
	 * @param props Property File
	 * @return InferedSemanticConcept containing all related concepts
	 */
	public static InferredConcept inferConcept(URI concept, Reasoner reasoner, MatcherProperties props){
		InferredConcept inferedConcept = new InferredConceptImpl();
		// Direct concept
		inferedConcept.addRatedSemanticConcept(new RatedURIImpl(concept, 1.0));
		
		if(reasoner.isConcept(concept)){
			// Equivalent concepts
			for(URI eqConcept : reasoner.getEquivalentClasses(concept)){
				if(!reasoner.isThingOrNothing(eqConcept)){
					inferedConcept.addRatedSemanticConcept(new RatedURIImpl(eqConcept, props.getEquivalentMark()));
				}
			}
			
			// Specialization concepts
			Set<URI> specConcepts = reasoner.getSubClasses(concept);
			specConcepts.addAll(reasoner.getIndividuals(concept));
			for(URI specConcept : specConcepts){
				if(!reasoner.isThingOrNothing(specConcept)){
					inferedConcept.addRatedSemanticConcept(new RatedURIImpl(specConcept, props.getSpecializationMark()));
				}
			}
			
			// Generalization concepts
			for(URI genConcept : reasoner.getSuperClasses(concept)){
				if(!reasoner.isThingOrNothing(genConcept)){
					inferedConcept.addRatedSemanticConcept(new RatedURIImpl(genConcept, props.getGeneralizationMark()));
				}
			}
			
		} else if (reasoner.isInstance(concept)) { // URI refers to an instance
			// Equivalent instances
			for(URI eqInstance : reasoner.getSimilarIndividuals(concept)){
				inferedConcept.addRatedSemanticConcept(new RatedURIImpl(eqInstance, props.getEquivalentMark()));
			}
			
			// Parent concepts
			for(URI genConcept : reasoner.getTypes(concept)){
				if(!reasoner.isThingOrNothing(genConcept)){
					inferedConcept.addRatedSemanticConcept(new RatedURIImpl(genConcept, props.getGeneralizationMark()));
				}
			}
		}
		return inferedConcept;
	}
	
	public static Set<SemanticProfile> getSemanticProfiles(SortedSet<RatedSemanticProfile> ratedProfiles) {
		Set<SemanticProfile> res = new HashSet<SemanticProfile>();
		for(RatedSemanticProfile ratedProfile : ratedProfiles) {
			res.add(ratedProfile.getProfile());
		}
		return res;
	}
}
