/**
 * EasierSBS project - Java file
 * Copyright (C) 2011 EBM WebSourcing - Petals Link
 * 
 * EasierSBS is free project: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * EasierSBS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this program.
 * If not, see <http://www.gnu.org/licenses/lgpl-3.0.txt>.	
 * 
 */ 
package com.petalslink.easiersbs.matching.service.matcher;

import java.util.Set;

import com.petalslink.easiersbs.matching.service.SBSFactoryImpl;
import com.petalslink.easiersbs.matching.service.api.EasierSBSException;
import com.petalslink.easiersbs.matching.service.api.SBSFactory;
import com.petalslink.easiersbs.matching.service.api.matcher.HybridMatcher;
import com.petalslink.easiersbs.matching.service.api.matcher.MatcherProperties;
import com.petalslink.easiersbs.matching.service.api.matcher.MatchingResult;
import com.petalslink.easiersbs.matching.service.api.matcher.SemanticMatcher;
import com.petalslink.easiersbs.matching.service.api.matcher.SyntacticMatcher;
import com.petalslink.easiersbs.matching.service.api.profile.SearchProfile;
import com.petalslink.easiersbs.matching.service.api.profile.rated.RatedSemanticProfile;
import com.petalslink.easiersbs.matching.service.profile.rated.RatedSemanticProfileImpl;
import com.petalslink.easiersbs.matching.service.util.ProfileUtil;
import com.petalslink.easiersbs.reasoner.api.ReasonerException;
import com.petalslink.easiersbs.reasoner.api.ReasonerFactory;
import com.petalslink.easiersbs.registry.service.api.SemanticRegistryManager;
import com.petalslink.easiersbs.registry.service.api.model.SemanticProfile;

/**
 * @author Nicolas Boissel-Dallier - Petals Link
 */
public class HybridMatcherImpl extends AbstractMatcherImpl implements HybridMatcher {

	private SBSFactory factory = SBSFactoryImpl.getInstance();
	
	private SemanticMatcher semanticMatcher;
	private SyntacticMatcher syntacticMatcher;
	
	private MatcherProperties props;
	
	
	public HybridMatcherImpl(ReasonerFactory reasoner, SemanticRegistryManager registry, MatcherProperties props) throws ReasonerException, EasierSBSException{
		this.props = props;
		this.registry = registry;
		semanticMatcher = factory.newSemanticMatcher(reasoner, registry, props);
		syntacticMatcher = factory.newSyntacticMatcher(registry, props);
	}
	
	public HybridMatcherImpl(ReasonerFactory reasoner, SemanticRegistryManager registry) throws ReasonerException, EasierSBSException{
		this(reasoner, registry, new MatcherPropertiesImpl());
	}
	
	@Override
	public void setMatcherProperties(MatcherProperties props) {
		this.props = props;
		semanticMatcher.setMatcherProperties(props);
		syntacticMatcher.setMatcherProperties(props);
	}
	
	@Override
	public SyntacticMatcher getSyntacticMatcher() {
		return syntacticMatcher;
	}

	@Override
	public SemanticMatcher getSemanticMatcher() {
		return semanticMatcher;
	}

	@Override
	public MatchingResult findServices(SearchProfile profile,
			Set<SemanticProfile> serviceList) {
		
		MatchingResult hybridRes = null;
		
		if(props.getSemanticWeight() == 0.0) {
			hybridRes = syntacticMatcher.findServices(profile, serviceList);
		} else if(props.getSyntacticWeigth() == 0.0) {
			hybridRes = semanticMatcher.findServices(profile, serviceList);
		} else {
			// Hybrid matching
			hybridRes = new MatchingResultImpl(profile);
			MatchingResult semanticRes = semanticMatcher.findServices(profile, serviceList);
			MatchingResult syntacticRes = syntacticMatcher.findServices(profile, ProfileUtil.getSemanticProfiles(semanticRes.getResultProfiles()));
			
			for(RatedSemanticProfile synProfile : syntacticRes.getResultProfiles()){
				RatedSemanticProfile hybridProfile = new RatedSemanticProfileImpl(synProfile.getProfile());
				
				RatedSemanticProfile semProfile = semanticRes.findProfile(synProfile.getProfile());
				
				hybridProfile.setProfileRate(this.getHybridRate(semProfile.getProfileRate(), synProfile.getProfileRate()));
				hybridProfile.setOperationRate(this.getHybridRate(semProfile.getOperationRate(), synProfile.getOperationRate()));
				hybridProfile.setInputCoverage(semProfile.getInputCoverage());
				hybridProfile.setInputRate(this.getHybridRate(semProfile.getInputRate(), synProfile.getInputRate()));
				hybridProfile.setOutputCoverage(semProfile.getOutputCoverage());
				hybridProfile.setOutputRate(this.getHybridRate(semProfile.getOutputRate(), synProfile.getOutputRate()));
				
				hybridRes.addResultProfile(hybridProfile);
			}
		}
		
		MatchingResult res = new MatchingResultImpl(profile);
		for(RatedSemanticProfile rated : hybridRes.getResultProfiles()){
			if(rated.getProfileRate() >= props.getHybridThreshold()){
				res.addResultProfile(rated);
			}
		}
		return res;
	}
	
	private double getHybridRate(double semanticRate, double syntacticRate) {
		// Weighted Harmonic Mean
		// return (1 / ((props.getSemanticWeight() / semanticRate) + (props.getSyntacticWeigth() / syntacticRate)));
		// Average
		return ((semanticRate * props.getSemanticWeight()) + (syntacticRate * props.getSyntacticWeigth()));
	}

}
