package com.ebmwebsourcing.easybox.api;

import java.net.URI;

import org.w3c.dom.Node;

/**
 * Empty interface flagging an object as being a model object.<br/>
 * A model object can be wrapped into an {@link XmlObject}.<br/>
 * A model object knows which unique {@link XmlObject} is attached to it.
 * Otherly said, as long as the {@link ModelObject} object stays in memory, it
 * remembers the {@link XmlObject} which wrapped it.
 * 
 * @author mjambert
 * 
 */
public interface ModelObject {

    
    /**
     * Duplicate this {@link ModelObject}.
     * 
     * @return Copy of this {@link ModelObject}.
     */
    ModelObject duplicate();
    
    
    /**
     * Duplicate this {@link ModelObject} data, but into a specific
     * target class.
     * 
     * @param <X>
     * @param targetClass
     * @return 
     */
    <X extends ModelObject> X duplicateAs(Class<X> targetClass);
    
    
    /**
     * Get unique {@link XmlObjectNode} wrapping this {@link ModelObject}.
     * 
     * @return {@link XmlObjectNode} wrapping this {@link ModelObject}.
     */
    XmlObjectNode getXmlObject();

    
    /**
     * Get natural parent object.
     * 
     * @return This object natural parent.
     */
    ModelObject getNaturalParent();

    /**
     * Get adoptive parent object.
     * 
     * @return This object adoptive parent.
     */
    ModelObject getAdoptiveParent();
    
    /**
     * Get natural or adoptive parent object.
     * 
     * @return This natural or adoptive object parent.
     */
    ModelObject getParent();
    
    
    /**
     * Get base URI of originating document.
     * 
     * If no base URI was set, this object will look
     * for return its parent base URI (recursively).
     *  
     * @return Originating document base URI.
     */
    URI getBaseURI();
    
    
    /**
     * Set base URI of originating document.
     * 
     * @param baseUri Originating base URI.
     */
    void setBaseURI(URI baseUri);
    

    Node getDOMNode();
}
