/****************************************************************************
 * Copyright (c) 2009-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easybox.api;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.xml.resolver.CatalogManager;
import org.apache.xml.resolver.tools.CatalogResolver;
import org.xml.sax.EntityResolver;

import com.ebmwebsourcing.easycommons.lang.UncheckedException;

public final class Catalog {

    private static Catalog INSTANCE;

    
    private CatalogManager catalogManager;
    private List<File> catalogFiles;

    private final Map<String, URI> resolvedUriCache;
    
    
    public Catalog() {
        this.catalogManager = null;
        this.catalogFiles = new ArrayList<File>();
        this.resolvedUriCache = new HashMap<String, URI>();
    }
    
    
    public static Catalog getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new Catalog();
        }
        return INSTANCE;
    }

    public static void setInstance(Catalog instance) {
        INSTANCE = instance;
    }
    
    
    private void resetCatalogManager() {
        catalogManager = null;
        resolvedUriCache.clear();
    }
    
    
    private CatalogManager getCatalogManager() {
        if (catalogManager == null) {
            catalogManager = new CatalogManager();
            catalogManager.setUseStaticCatalog(false);
            catalogManager.setIgnoreMissingProperties(true);
            StringBuffer sb = new StringBuffer();
            for (File catalogFile : catalogFiles) {
                sb.append(catalogFile.getAbsolutePath());
                sb.append(",");
            }
            if (sb.length() > 0) sb.deleteCharAt(sb.length() - 1);
            catalogManager.setCatalogFiles(sb.toString());
        }
        return catalogManager;
    }
    
    
    public void addCatalogFile(File catalogFile) {
        catalogFiles.add(catalogFile);
        resetCatalogManager();
    }
    
    
    public EntityResolver newEntityResolver() {
        return new CatalogResolver(getCatalogManager());
    }
    
    
    public URI resolveUri(URI uri) {
        assert uri != null;
        return resolveUri(String.valueOf(uri));
    }
    
    
    public URI resolveUri(String uri) {
        assert uri != null;
        if (resolvedUriCache.containsKey(uri)) return resolvedUriCache.get(uri);
        try {
            String resolvedUri = getCatalogManager().getCatalog().resolveURI(uri);
            URI result = null;
            if (resolvedUri != null) {
                result = URI.create(resolvedUri);
            }
            resolvedUriCache.put(uri, result);
            return result;
        } catch (MalformedURLException e) {
            throw new UncheckedException(String.format("Cannot resolve URI '%s' (MalformedURLException).", uri));
        } catch (IOException e) {
            throw new UncheckedException(String.format("Cannot resolve URI '%s' (IOException).", uri));
        }
    }
    
    
}
