/****************************************************************************
 * Copyright (c) 2009-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easybox.api;

import java.util.Map;

import javax.xml.namespace.QName;

import org.w3c.dom.Node;

/**
 * An {@link XmlObject} provides a base interface on a binding on an XML
 * structure.<br/>
 * <br/> {@link XmlObject} are organized as a tree structure. Every object knows its
 * parent. Every object can have children of two sorts :<br/>
 * <ul>
 * <li>natural children : those which directly reflect children in internal
 * model,</li>
 * <li>adopted children : those who reflect a convenience parent-child
 * relationship. They do not have any matching child in the internal model tree.
 * For instance, including an XML document could result in adding the included
 * {@link XmlObject} root as an adopted child of the originating including
 * {@link XmlObject}.
 * </ul>
 * 
 * It is assumed that a third-party library is used to create the binding object
 * itself. {@link XmlObject} can thus be seen as implementation independent
 * wrapping around the generated POJO (the "model").<br/>
 * 
 * @author mjambert
 * 
 */
public interface XmlObject extends XmlObjectNode {

	public static final XmlObject[] EMPTY_ARRAY = new XmlObject[0];

	/**
	 * Get {@link XmlContext} from which this {@link XmlObject} originated.
	 * 
	 * @return {@link XmlContext} which originated this object.
	 */
	XmlContext getXmlContext();

	
	/**
	 * Create a duplicate of this {@link XmlContext}.
	 * All descendants are duplicated as well ; parent hierarchy is NOT duplicated.
	 * 
	 * @return Duplicated instance of this {@link XmlContext};
	 */
	XmlObject duplicateXmlObject();
	
	<X extends XmlObject> X duplicateXmlObjectAs(Class<X> targetInterfaceClass);



	/**
	 * Set qname of this {@link XmlObject}, if allowed.
	 * 
	 * @param qname
	 *            New qname.
	 */
	void setXmlObjectQName(QName qname);

	/**
	 * Get all attributes defined on this {@link XmlObject}.
	 * 
	 * @return All attributes defined on this {@link XmlObject}, collected
	 *         inside a map.
	 */
	Map<QName, Object> getXmlObjectAttributes();

	/**
	 * Get list of all natural child {@link XmlObject}.
	 * 
	 * @return Properly ordered list of all natural child objects.
	 */
	XmlObjectNode[] getXmlObjectNaturalChildren();

	/**
	 * Get list of all adopted child {@link XmlObject}.
	 * 
	 * @return Properly ordered list of all adopted child objects.
	 */
	XmlObjectNode[] getXmlObjectAdoptedChildren();

	/**
	 * Get associated DOM node, if known.
	 * 
	 * @return Associated DOM node, or {@code null} if none.
	 */
	Node getXmlObjectDOMNode();

	/**
	 * Get all namespace bindings in scope of this {@link XmlObject}.
	 * 
	 * @return Map of prefix => namespace URI known on this {@link XmlObject} (ie declared either
	 *     on corresponding XML element, or on ancestors).
	 */
	Map<String, String> getXmlObjectInScopeNamespaces();

	
	
}
